from __future__ import nested_scopes

import thread
import time
import math

from bot_base import *
from utils.ordered_list import *

DEFAULT_ACCURACY = 0.0001 # this is used when we tests if a subtraction of two floating number is equal to zero ... math.fabs(subtraction) < accuracy

# item for scheduler ... you specify UT time and priority, when to run and function (note: to run function with specific params use lamba form)
#                        lower priority means that the function will be run before function with higher priority
class Scheduler_Item_UT_Time:

    def __init__(self, ut_time, function_to_schedule, priority = 0):
        self.ut_time = float(ut_time)
        self.function_to_schedule = function_to_schedule
        self.priority = priority
        self.should_run = 1 # flag, means whetver function should be executed
        self.item_lock = thread.allocate_lock() # we will need to check and sometime change self.should_run 

    def is_trivial(self):
        return self.trivial

    def run(self):
        self.lock()
        if self.should_run:
            self.unlock()    
            self.function_to_schedule()        
            return
        self.unlock()    
            
    def lock(self):
        self.item_lock.acquire()
        
    def unlock(self):
        self.item_lock.release()
        

# this function returns negative number for item1.ut_time < item2.ut_time, 0 for ==, positive for >
def compare_ut_time_items(item1, item2, accuracy = DEFAULT_ACCURACY):
    diff = item1.ut_time - item2.ut_time
    if math.fabs(diff) < accuracy:
        return item1.priority - item2.priority
    else:
        return diff

# item for scheduler ... you specify system time, when to run and function to run (note: to run function with specific params use lamba form)
#                        lower priority means that the function will be run before function with higher priority
class Scheduler_Item_System_Time:

    def __init__(self, system_time, function_to_schedule, priority = 0):
        self.system_time = float(system_time)
        self.function_to_schedule = function_to_schedule
        self.priority = priority
        self.should_run = 1 # flag, means whetver function should be executed
        self.item_lock = thread.allocate_lock() # we will need to check and sometime change self.should_run         

    def is_trivial(self):
        return self.trivial

    def run(self):
        self.lock()
        if self.should_run:
            self.function_to_schedule()
        self.unlock()    
            
    def lock(self):
        self.item_lock.acquire()
        
    def unlock(self):
        self.item_lock.release()
            

# this function returns negative number for item1.system_time < item2.system_time, 0 for ==, positive for >
def compare_system_time_items(item1, item2, accuracy = DEFAULT_ACCURACY):
    diff = item1.system_time - item2.system_time
    if math.fabs(diff) < accuracy:
        return item1.priority - item2.priority
    else:
        return diff

# *************
# BOT_SCHEDULER
# *************

# Bot_Scheduler is used for scheduling functions to run at certain time, mainly used for expire information about some state of the Bot (e.g. see_player)
# runing the schedule - creates two new threds one for scheduler in UT Time, one for scheduler in System time
# you can schedule items with schedule_ut_time / schedule_system_time (time, function, priority) ... lower priority is called first (if there are items scheduled at the same time)

# NOTE: that scheduling non-trivial function can be severe to scheduler, because it doesn't create new threads for this calls

# for each list exixts a lock ... if you acquire it, make sure you also release it! (it's silly for me to say this obvious thing, but ... be sure, ok?)

# to start scheduler call execute_scheduler()

class Bot_Scheduler(Bot_Base):

    def __init__(self, bot, **kw): # we assume that bot has attribute bot.agent

        Bot_Base.__init__(self, bot.agent, **kw)

        self.bot = bot # from self.bot we will get self.bot.actual_ut_time

        # two ordered list, where we will store scheduled items     
        self.ut_time_list = Ordered_List(compare_ut_time_items)
        self.ut_time_list_lock = thread.allocate_lock() # inserting things into list is non-trivial (log n), so we need this to prevent conflicts
        
        self.system_time_list = Ordered_List(compare_system_time_items)
        self.system_time_list_lock = thread.allocate_lock() # inserting things into list is non-trivial (log n), so we need this to prevent conflicts

        self.ut_time_thread_id = 0
        self.ut_time_thread_active = 0
        self.ut_time_exec = 0   # flag, if set true - scheduler (ut time) is working, if set false, scheduler is not working or will stop it's function
        self.ut_time_pause = 0  # flag, if true - scheduler (ut time) will pause it's work until set false

        self.ut_time_sleep_length_when_no_items_left = float(0.4) # if no items is on schedule, the scheduler calls time.sleep( this variable )
                                                                  # I have no idea what the number should be, depends on your needs - experiment :-)
        self.ut_time_max_allowed_difference = float(0.5) # useful for debugging, if difference between the time and item's time to run is greater then this,
                                                         # debug msg is sent                                                           
        self.ut_time_between_iterations = float(0.07) # when top item isn't ready to run, scheduler will sleep for this time                                                         
        
        self.system_time_thread_id = 0
        self.system_time_thread_active = 0
        self.system_time_exec = 0   # flag, if set true - scheduler (system time) is working, if set false, scheduler is not working or will stop it's function
        self.system_time_pause = 0  # flag, if true - scheduler (system time) will pause it's work until set false
        
        self.system_time_sleep_length_when_no_items_left = float(0.4) # if no items is on schedule, the scheduler calls time.sleep( this variable )
                                                                      # I have no idea what the number should be, depends on your needs - experiment :-)
        self.system_time_max_allowed_difference = float(0.5) # useful for debugging, if difference between the time and item's time to run is greater then this,
                                                             # debug msg is sent                                                                                                                          
        self.system_time_between_iterations = float(0.07) # when top item isn't ready to run, scheduler will sleep for this time                                                              

    # ******************
    # SCHEDULING METHODS
    # ******************

    # schedules item at certain UT Time with certain priority
    # NOTE: returns scheduled item, can be stored and if needed set item.should_run = 0 to prevent from running (to save system resources)
    #       don't forget to lock the scheduled item!
    def schedule_ut_time(self, ut_time, function, priority):
        self.ut_time_list_lock.acquire()
        item = Scheduler_Item_UT_Time(ut_time, function, priority)    
        self.ut_time_list.insert( item )
        self.ut_time_list_lock.release()
        return item

    # schedules item at certain System Time with certain priority
    # NOTE: returns scheduled item, can be stored and if needed set item.should_run = 0 to prevent from running (to save system resources)
    #       don't forget to lock the scheduled item!
    def schedule_system_time(self, system_time, function, priority):
        self.system_time_list_lock.acquire()
        item = Scheduler_Item_System_Time(system_time, function, priority)
        self.system_time_list.insert( item )
        self.system_time_list_lock.release()
        return item
        
    # this starts the work of scheduler - creates two independent threads
    def execute_scheduler(self):
        self.ut_time_thread_id = thread.start_new_thread(self.execute_ut_time_scheduler, ())
        self.system_time_thread_id = thread.start_new_thread(self.execute_system_time_scheduler, ())

    # ************************
    # UT TIME SCHEDULER METHOD
    # ************************

    def execute_ut_time_scheduler(self):
        self.ut_time_thread_active = 1
        self.ut_time_exec = 1
        self.ut_time_pause = 0

        self.bot.debug(5, "Bot_Scheduler (UT Time) was started.")

        while self.ut_time_exec:
            self.ut_time_list_lock.acquire()
            ut_time = self.bot.actual_ut_time
            item = self.ut_time_list.top()
            if item != None:
#                print "Scheduler: ", item.ut_time, " ? ", ut_time
                if item.ut_time < ut_time:
#                    print "UT: ", item.ut_time, " < ", ut_time
                    if ut_time - item.ut_time > self.ut_time_max_allowed_difference:
                        self.debug(1, "Bot_Scheduler (UT Time): difference between item's time and ut time is " + str(float(ut_time) - float(item.ut_time)) + " > " + str(self.ut_time_max_allowed_difference))
                        print "Bot_Scheduler (UT Time): difference between item's time and ut time is ", str(float(ut_time) - float(item.ut_time)), " > ", str(self.ut_time_max_allowed_difference)
                    self.ut_time_list.pop()
                    self.ut_time_list_lock.release()
                    item.run()
                else:
                    self.ut_time_list_lock.release()
                    time.sleep(self.ut_time_between_iterations)
            else:
                self.ut_time_list_lock.release()
                time.sleep(self.ut_time_sleep_length_when_no_items_left)

            while self.ut_time_pause:
                if not self.ut_time_exec: break # exits this waiting-while and then exits main loop (because self.ut_time_exec == 1)
                time.sleep(1)        

        self.bot.debug(5, "Bot_Scheduler (UT Time) was stopped.")

        self.ut_time_exec = 0
        self.ut_time_pause = 0
        self.ut_time_thread_active = 0
        self.ut_time_thread_id = 0    

    # ****************************
    # SYSTEM TIME SCHEDULER METHOD
    # ****************************
    
    def execute_system_time_scheduler(self):
        self.system_time_thread_active = 1
        self.system_time_exec = 1
        self.system_time_pause = 0

        self.bot.debug(5, "Bot_Scheduler (System Time) was started.")

        while self.system_time_exec:
            self.system_time_list_lock.acquire()
            system_time = time.time()
            item = self.system_time_list.top()
            if item != None:
#                print "SYSTEM: ", item.system_time, " ? ", system_time
                if item.system_time < system_time:
#                    print "SYSTEM: ", item.system_time, " < ", system_time
                    if float(system_time) - float(item.system_time) > float(self.system_time_max_allowed_difference):
                        self.debug(1, "Bot_Scheduler (System Time): difference between item's time and system time was " + str(float(system_time) - float(item.system_time)) + " > " + str(self.system_time_max_allowed_difference))
                        print "Bot_Scheduler (System Time): difference between item's time and system time was ", str(float(system_time) - float(item.system_time)), " > ", str(self.system_time_max_allowed_difference)
                    self.system_time_list.pop()
                    self.system_time_list_lock.release()
                    item.run()
                else:
                    self.system_time_list_lock.release()
                    time.sleep(self.system_time_between_iterations)
            else:
                self.system_time_list_lock.release()
                time.sleep(self.system_time_sleep_length_when_no_items_left)
#                print "SYSTEM: waked up"

            while self.system_time_pause:
                if not self.system_time_exec: break # exits this waiting-while and then exits main loop (because self.system_time_exec == 1)
                time.sleep(1)        

        self.bot.debug(5, "Bot_Scheduler (System Time) was stopped.")

        self.system_time_exec = 0
        self.system_time_pause = 0
        self.system_time_thread_active = 0
        self.system_time_thread_id = 0

    # -------------------------------------------------
    # methods which can stop / pause / resume scheduler
    # -------------------------------------------------

    def stop_ut_time(self):
        self.bot.debug(5, "Bot_Scheduler (UT Time) - stop command received.")
        self.ut_time_exec = 0

    def pause_ut_time(self):
        self.bot_debug(5, "Bot_Scheduler (UT Time) - pause command received.")
        self.ut_time_pause = 1

    def resume_ut_time(self):
        if self.ut_time_pause:
            self.bot_debug(5, "Bot_Scheduler (UT Time) - resume command received.")
            self.ut_time_pause = 0
        else:
            self.bot_debug(1, "Bot_Scheduler (UT Time) - resume command received while it's not paused.")

    def stop_system_time(self):
        self.bot.debug(5, "Bot_Scheduler (System Time) - stop command received.")
        self.system_time_exec = 0

    def pause_system_time(self):
        self.bot_debug(5, "Bot_Scheduler (System Time) - pause command received.")
        self.system_time_pause = 1

    def resume_system_time(self):
        if self.ut_time_pause:
            self.bot_debug(5, "Bot_Scheduler (System Time) - resume command received.")
            self.system_time_pause = 0
        else:
            self.bot_debug(1, "Bot_Scheduler (System Time) - resume command received while it's not paused.")

    def stop_scheduler(self):
        self.stop_ut_time()
        self.stop_system_time()

    def pause_scheduler(self):
        self.pause_ut_time()
        self.pause_system_time()

    def resume_scheduler(self):
        self.resume_ut_time()
        self.resume_system_time()
                
            
        
        

        

        
        