
# *******************
# FILE WITH BOT CLASS
# *******************

# In this file you can find definition of Bot class which deals with GameBot.

# ********************
# DEALING WITH GAMEBOT
# ********************
#
#   First - you should be familiar with GB message-types and GB commands.
#   Second - here I'll write my expiriences from working with GameBot server a his half-truths /half-lies
#
#   Reachable - so far I didn't have a fatal result that Reachable == "True" wasn't true, that means
#               if you get Reachable = True you can be sure you can run straight to that point
#               However if you receive "False" it doesn't mean you can't reach that point run straight to that point
#               a) if you're checking point which is higher than you (but there are steps, or slope up) you can receive "False"
#               b) sometime a bit of corner also result in "False" though bot can go there straight
#               c) if you test "Reachable" be sure to test it against string "True" / "False" not -> if my_result["Reachable"]:
#               d) sometime you can get NAV with Reachable "True" but if you check it again it's "False" but it's very rare
#   GetPath - isn't really omnious, if you try and run my ExploreBot you can see that on many GETPATH request it receive that path doesn't exist,
#             also GetPath can't handle Teleports, once I've seen my bot (after the path was received) run into teleport then turn back and try
#             to run to another navpoint, unfortunately there was the same teleport between him and that navpoint
#           - I've just found another sad thing, getpath can fail completely, when asking for path to the navpoint which has been far away (or on
#             higher ground), it gives me a result to run to nearest navpoint and try to go straight to that navpoint, so bot bumped to the wall),
#             according to GB mail archive, GB has problem when path has to be 15+ long 
#   Collisions - WAL message type doesn't seem to work properly, sometime you collide with wall but no WAL msg come, so I advise you to also
#                use Bot.bot_is_moving() or other such a thing which take differences between your actual and last location and
#                test both ... Bot.bot_collide() or not Bot.bot_is_moving(), also I've found that after first collision it's good to try to Jump
#                it can solve some problems (Jumping with GB it's not bad as is written in api documentation)



from __future__ import nested_scopes
from socket import *
from posh_bits import *

import re      #re is for Regular Expressions
import thread
import time
import math

from bot_scheduler import *
from bot_executor import *
from bot_expire_items import *
from utils.log_to_file import *
from utils.unreal_utils import *
from utils.ordered_list import *
from utils.python_utils import *

# this class is used for storing function in ordered list according to their priority,
# is used to store functions which are attached to a GB message type
# function suppose to accept one parametr - dictionary with message values
class Function_To_Run:

    def __init__(self, function, priority):
        self.function = function
        self.priority = priority

    def get_fn(self):
        return self.function;

# this function returns negative number for item1.system_time < item2.system_time, 0 for ==, positive for >
def compare_fn_to_run( fn_1, fn_2 ):
    return fn_1.priority - fn_2.priority

# this list is used in Bot class - keeps info about function which should be called upon arrival of message of certain message_type (like "BEG", etc.)
class Reaction_To_GB_Message_List:

    def __init__(self):
        self.dict = {}

    def insert(self, message_type, fn_to_run, priority):
        if self.dict.has_key(message_type):
            self.dict[message_type].insert( Function_To_Run(fn_to_run, priority) )
        else:
            self.dict[message_type] = Ordered_List(compare_fn_to_run)
            self.dict[message_type].insert( Function_To_Run(fn_to_run, priority) )

    # returns list of a specific message_type
    def get_list(self, message_type):
        if (self.dict.has_key(message_type)):
            return self.dict[message_type].get_list()
        else:
            self.dict[message_type] = Ordered_List(compare_fn_to_run)
            return self.dict[message_type].get_list()

# ***
# BOT
# ***

# class which holds the representation of UT world, have also a method to create new thread (communication / conn_thread) which is
# talking with GB and keeps the informations accurate

# method connect() ... calls connect_thread in a new thread, connects to GB a keep reading / interpreting messages from GB
# method process_incoming_message(cmd, dict) ... gets parsed msg a have a duty to serves it
# method send_message(cmd, dict) ... sends new message to a server, dict is a dictionary - keys are attribute names + their values

class Bot(Base): # Bot is descendant of Base (from posh_bits.basic) so it can use debug(level, msg) for debug messages

 # ***************
 # METHOD __INIT__
 # ***************

    def __init__(self, ip, port, botname, agent, gamebot_debug_file = None, **kw): # we gives IP:Port of the GB server + name of the bot
                                                                                   # agent - instance of Agent from posh_agent.py
                                                                                   # gamebot_debug_file - if contains string ... considered as a filename where to log
                                                                                   #                      communication with GB server
        Base.__init__(self, agent, **kw)

        # binds agent and bot - in case when it comes in handy

        agent.bot = self
        #self.agent = agent # well this was already done in Base.__init__

        # information about GameBot Server connection
        
        self.ip = ip     # IP - on which GB is listening
        self.port = port # post - on which GB is listening
        self.botname = botname # actualy didn't work - probably GB bug

        # information about the thread, which is communicating with GB server

        self.conn_thread_active = 0 # wether connection thread is active
        self.conn_ready = 0         # tells to us if is connected to server
        self.conn_thread_id = None  # connection thread ID
        self.kill_connection = 0    # flag, whetver to kill connection (also terminates the thread afterwards)

        # follow the attribute for Bot_Scheduler - used to schedule expirations of various informations

        self.scheduler = Bot_Scheduler(self)

        # next two dictinaries contains [type_message] -> processing function, is used to call specific function for each message event
        # if needed, can be easily rewritten even in run-time

        self.gb_synchronous_msg =  { "BEG": self.process_beg, # begin of a synchronous batch 
                                     "SLF": self.process_slf, # information about your bot's state

                                     "END": self.process_end }# end of a synchronous batch
                                                              # there shouldn't be any other synchronous messages      
        
        self.gb_asynchronous_msg = { "PONG":self.process_pong}# respond to PING msg sent to GB
                                                              # there shouldn't be any other asynchronous messages   
        
        # now, Bot offers you to register a functions which will be called upon arrival of a GB message of specific priority,
        # we will store them in this attribute self.reaction_list ... if nothing else this feature can be used for additional debug info
        #                                                             you won't need to edit process.xxx just add here a new function
        #                                                             with add_reaction_to_msg     

        self.reaction_list = Reaction_To_GB_Message_List()

        # (read previous comment) ... these function can be called from connection thread, but it would be bad if some non-trivial function
        # stops updating the information about UT world, so there is Bot_Executor object with another thread,
        # where these functions will be sequentualy called

        self.executor = Bot_Executor(self)

        # next attribute is considered to be a function with parametrs: type_of_msg == "sent" or "received", msg_type == GB indetification of msg / command,
        #                                                               msg_string | it is used for reporting any communication with GB
        
        self.call_gb_monitor = None

        # a few next rows will create an object which ensure logging of GB msgs to a file (if specified)
        
        if (gamebot_debug_file != None):
            self.log_gb_msg = Log_To_File(gamebot_debug_file, "GameBot messages", 0, self.debug)
            self.log_gb_msg.debug_fn = self.debug
            self.log_gb_msg.start()
        else:
            self.log_gb_msg = None # we logs msgs only if this object is not None

        
        # attribute which holds information from NFO message received from GB as a "welcome" message
        #    "Gametype" - What you are playing (BotDeathMatchPlus, BotTeamGame, BotDomination),
        #    "Level" - name of map in play
        #    "TimeLimit" - maximum time game will last (if tied at end, goes into sudden death overtime),
        #    "FragLimit" - number of kills needed to win game (BotDeathMatchPlus only),
        #    "GoalTeamScore" - number of points a team needs to win game (BotTeamGame, BotDomination),
        #    "MaxTeams" - max number of teams. valid team range will be 0 to (MaxTeams - 1) (BotTeamGame, BotDomination) 
        #    "MaxTeamSize" - Max number of players per side (BotTeamGame, BotDomination)

        self.conninfo = {}      

   # following attributes are under control of process_incoming_message / representing the state and knowlege of the bot from GB
        # for every information here exists a method get_sumthing_sumthing ... 
        # and also some more like see_any_players - check it ... everything here is stored in UT units, nothing is converted ...
        # though you can find converted information under another keys: (this is done in self.process_gb_message() )
        #     if the "item" contains ["Location"] ... you can be sure that it will also have ["LocationTuple"] key
        #                                             with triple (x, y, z) parsed according to ["Location"]
        #     if the "item" contains ["Rotation"] ... you can be sure that it will also have ["RotationTuple"] key
        #                                             with triple (pitch, yaw, roll) -> see gamebot spec.
        #                                             also there will be ["RotationDegreesTuple"] key
        #                                             with triple (pitch_in_degrees, yaw_in_degrees, roll_in_degrees)  
        # there are also comments from GameBot Network API document at each get_method
        # NOTE 1: some converting functions / checking functions, etc. are in utils.unreal_utils
        # NOTE 2: please see Expire_Info class to understand expiration mechanism (in this file before declaration of this class)
        # NOTE 3: all atributes which represents "memory" like self.known_flags, self.all_known_items, etc. is inserted only once,
        #         that means they are never rewritten ... it gives you a chance to write additional useful informations to these items,
        #         to do this you can either override process_xxx method or register next method via self.add_reaction_to_msg
        #         to process the desired message_type

   # FROM SYNCHRONOUS MESSAGES  

        # from BEG msg
        
        self.actual_ut_time = 0 # actual time in UT ... is send with BEG message

        # from SLF msg

        self.bot_status = { "Id": "0",              "Rotation": "-1,-1,-1",      "RotationTuple": (-1,-1,-1), "RotationDegreesTuple": (-1,-1,-1),
                            "Location": "-1,-1,-1", "LocationTuple": (-1,-1,-1), "Velocity": "-1,-1,-1",      "Name": "",
                            "Team": "-1",           "Health": "0",               "Weapon": "",                "PrimaryAmmo": "0",  "SecondaryAmmo": "0",
                            "Armor": "0", "Adrenaline": "0" }

        self.previous_bot_status = { "Id": "0",              "Rotation": "-1,-1,-1",      "RotationTuple": (-1,-1,-1), "RotationDegreesTuple": (-1,-1,-1),
                                     "Location": "-1,-1,-1", "LocationTuple": (-1,-1,-1), "Velocity": "-1,-1,-1",      "Name": "",
                                     "Team": "-1",           "Health": "0",               "Weapon": "",                "CurrentAmmo": "0",
                                     "Armor": "0" } # difference between this and self.bot_status can tell us if something has changed / is changing
        

        self.enemy_teams_numbers = None # will be a list with numbers of opposite teams, can be get with
                                        # self.get_enemy_teams_list() ... which will also compute this if not already present

        self.location_accuracy = 40 # for self.bot_at_location where distance between actual location and provided location is counted,
                                    # note that 40 in UT units isn't so much, RUNTO command isn't so precise and if there is a step before navpoint
                                    # it's pretty common to have difference almost 40!

        self.bot_moving_accuracy = 2 # used for self.bot_is_moving check ... to report true, difference must be greater then this                                    
        
        # from GAM msg
        
        self.game_status = { "PlayerScores": {}, "TeamScores": { "0": 0, "1": 0, "2": 0, "3": 0 }, "DomPoints": [], "HaveFlag": 0, "EnemyHasFlag": 0 } 

        # from PLR msg
        #   following information about plyers will have this keys / info
        #       "Id" - a unique id for this player, assigned by the game, "Rotation" - which direction the player is facing in absolute terms
        #       "Location" - an absolute location for the player ,        "Velocity" - absolute velocity in UT units,
        #       "Team" - what team the player is on,                      "Reachable" - true if the bot can run to this other player directly, false otherwise. Possible reasons for false: pit or obstacle between the two characters
        #       "Weapon" - what class of weapon the character is holding, "TimeStamp" - UT Time when the msg was received

        self.see_players = { "Friend": Expire_Dictionary(self), "Enemy": Expire_Dictionary(self) }
                           # players fall in two categories as you can see ... the keys are UT-Ids of players, each player has also "TimeStamp" with UT-Time
        self.see_players_expire_time_type = "ut_time" # means in which units we will measure the time the information should be valid for items in dictionary                                                      
        self.see_players_expire_time = float(0.5) # for how long the item will survive

        self.last_known_positions = {} # keys ar UT-Ids of players, here we have ... well, what the variable says, also have ["TimeStamp"]

        # from NAV msg
        #   following information about navpoints will have this keys / info
        #   "Id" - a unique id for this pathnode, assigned by the game,           "Location" - an absolute location 
        #   "Reachable" - true if the bot can run here directly, false otherwise, "TimeStamp" - UT Time when the msg was received
        
        self.see_navpoints = Expire_Dictionary(self) # indexed by Nav-Point IDs
        self.see_navpoints_expire_time_type = "ut_time" # means in which units we will measure the time the information should be valid for items in dictionary                                                      
        self.see_navpoints_expire_time = float(0.5) # for how long the item will survive

        self.known_navpoints = {} # indexed by Nav-Point IDs

        # from MOV msg
        #   following information about movers will have this keys / info
        #   "Id" - a unique id for this mover, assigned by the game,           "Location" - an absolute location 
        #   "Reachable" - true if the bot can run here mover, false otherwise, "DamageTrig" - true if the mover needs to be shot to activated. 
        #   "Type" - Class of the mover,                                      "TimeStamp" - UT Time when the msg was received

        self.see_movers = Expire_Dictionary(self) # indexed by Movers IDs
        self.see_movers_expire_time_type = "ut_time" # means in which units we will measure the time the information should be valid for items in dictionary                                                      
        self.see_movers_expire_time = float(0.5) # for how long the item will survive

        self.known_movers = {} # indexed by Movers IDs

        # from DOM msg
        #   following information about domination points will have this keys / info
        #   "Id" - a unique id for this pathnode, assigned by the game,           "Location" - an absolute location 
        #   "Reachable" - true if the bot can run here directly, false otherwise, "Controller" - which team controls this point 
        #   "TimeStamp" - UT Time when the msg was received
        
        self.see_dom_points = Expire_Dictionary(self) # indexed by Dom-Point IDs
        self.see_dom_points_expire_time_type = "ut_time" # means in which units we will measure the time the information should be valid for items in dictionary                                                      
        self.see_dom_points_expire_time = float(0.5) # for how long the item will survive

        self.known_dom_points = {} # indexed by Nav-Point IDs

        # from FLG msg
        #   following information about flags will have this keys / info
        #   "Id" - a unique id for this flag, assigned by the game,  "Location" - an absolute location of the flag,
        #   "Holder" - the identity of player/bot holding the flag (only sent if flag is being carried),
        #   "Team" - the team whose flag this is,                    "Reachable" - true if the bot can run here directly, false otherwise 
        #   "State" - whether the flag is "Held" "Droped" or "Home", "TimeStamp" - UT Time when the msg was received
        
        self.see_flags = { "0": Expire_Dictionary(self), "1": Expire_Dictionary(self), "2": Expire_Dictionary(self), "3": Expire_Dictionary(self) }
                         # indexed by Team Number, then by Flag Ids
        self.see_flags_expire_time_type = "ut_time" # means in which units we will measure the time the information should be valid for items in dictionary                                                      
        self.see_flags_expire_time = float(0.5) # for how long the item will survive
 
        self.known_flags = {} # indexed by Flags IDs - only "Home" (original position) flags are written

        # from INV msg
        #   following information about items will have this keys / info
        #   "Id" - a unique id for this inventory item, assigned by the game,     "Location" - an absolute location
        #   "Reachable" - true if the bot can run here directly, false otherwise, "Type" - a string representing type of object
        #   "TimeStamp" - UT Time when the msg was received
        self.see_items = {} # indexed by category then by Items IDs, under each category will be again Expire_Dictioary instance
                            # if I know all the categories, I could write them all here, but I don't know, so they will be created as the need arises
        self.see_items_expire_time_type = "ut_time" # means in which units we will measure the time the information should be valid for items in dictionary                                                      
        self.see_items_expire_time = float(0.5) # for how long the item will survive                            

        self.all_known_items = {} # indexed by Items IDs

        self.known_items_by_category = {} # indexed by category then by Items IDs, here will be normal dictionaries

   # FROM ASYNCHRONOUS MESSAGES

        # MESSAGES NOTE: probably the best way to deal with messages from other players / bots is to register function which will fire when msg is received
        #                I'm not quite sure about the way, how they can be used, so I've implemented it only just as a lists without locks

        # from VMS msg

        self.global_messages = [] # here will be stored messages from global channel, last item is the newest message
                                  # each item is dictionary with: "String" - human readable message sent by another player in the game
                                  #                               "TimeStamp" - UT Time when the msg was received
        self.global_messages_max_size = 64 # how long the list can be

        # from VMT msg

        self.private_messages = [] # here will be stored messages from private-team channel, last item is the newest message
                                   # each item is dictionary with: "String" - human readable message sent by another player in the game
                                   #                               "TimeStamp" - UT Time when the msg was received
        self.private_messages_max_size = 64 # how long the list can be

        # from VMG msg

        self.tokenized_messages = [] # here will be stored messages from private-team channel, last item is the newest message
                                     # each item is dictionary with: "Sender" - unique id of player who sent message
                                     #                               "Type" - type of message (e.g. Command, Taunt, etc...) 
                                     #                               "Id" - message id. specifies which message is being sent                                     
                                     #                               "TimeStamp" - UT Time when the msg was received
        self.tokenized_messages_max_size = 64 # how long the list can be

        # from ZCF / ZCH / ZCB

        self.bot_zones = { "Head": { "Id": 0, "TimeStamp": 0 },   # last known bot zones
                           "Body": { "Id": 0, "TimeStamp": 0 },   # each key will contain a dictionary with keys ... "Id" - unique zone if
                           "Foots": { "Id": 0, "TimeStamp": 0 } } #                                                  "TimeStamp" - UT Time when the msg was received
                                                                 # unfortunately, I'm not familiar with UT Ids... so can't say what's lava, what's not, etc                      

        # from WAL

        self.last_collision = { "Id": -1, "Normal": 0, "Location": "0,0,0", "LocationTuple": (0,0,0), "TimeStamp": -1 }
                                                 # contains ... "Id" - unique ID of the wall you've collide with
                                                 #              "Normal" - normal of the angle bot colided at.
                                                 #              "Location" - absolute location of bot at time of impact
                                                 #              "TimeStamp" - UT Time when the msg was received
        self.collision_accuracy = float(0.3) # if you call self.bot_collide() ... TimeStamp of last collision minus actual_ut_time
                                      #                                    must be under this value to report true
                                      # remember that initial value of last_collision["TimeStamp"] must be greater then this

        # from FAL

        self.last_falling = { "Fell": 0, "Location": "0,0,0", "LocationTuple": (0,0,0), "TimeStamp": -1 }
                                               # contains ... "Fell" - True if you fell. False if you stopped at edge. 
                                               #              "Location" - absolute location of bot
                                               #              "TimeStamp" - UT Time when the msg was received
        self.falling_accuracy = float(0.3) # if you call self.bot_is_falling() ... actual_ut_time minus TimeStamp of last falling
                                    #                                       must be under this value to report true                                   
                                    # remember that initial value of last_falling["TimeStamp"] must be greater then this

        # from BMP

        self.bump_another_actor = Expire_Dictionary(self) # indexed with players / object id
                                                          # contains ... "Id" - unique id of actor (actors include other players and other
                                                          #                                         physical objects that can block your path.) 
                                                          #              "Location" - location of thing you rammed
                                                          #              "TimeStamp" - UT Time when the msg was received
        self.bump_another_actor_expire_time_type = "ut_time" # means in which units we will measure the time the information should be valid for items in dictionary                                                      
        self.bump_another_actor_expire_time = float(0.5) # for how long the item will survive                                                            

        # from HRP

        self.hear_pickup = Expire_Dictionary(self) # indexed with players id
                                                   # contains ... "Player" - unique ID of player who picked up the object
                                                   #              "TimeStamp" - UT Time when the msg was received
        self.hear_pickup_expire_time_type = "ut_time" # means in which units we will measure the time the information should be valid for items in dictionary                                                      
        self.hear_pickup_expire_time = float(0.5) # for how long the item will survive

        # from HRN

        self.hear_noise = Expire_Dictionary(self) # indexed with source id
                                                  # contains ... "Source" - unique ID of player who picked up the object
                                                  #              "TimeStamp" - UT Time when the msg was received
        self.hear_noise_expire_time_type = "ut_time" # means in which units we will measure the time the information should be valid for items in dictionary                                                      
        self.hear_noise_expire_time = float(0.5) # for how long the item will survive

        # from PRJ

        self.incoming_projectile = Expire_Dictionary(self) # for indexation see ... self.process_prj()
                                                           # contains ... "Time" - estimated time till impact 
                                                           #              "Direction" - rotation value that the projectile is coming from,
                                                           #                            best chance to dodge is to probably head off at a rotation
                                                           #                            normal to this one (add ~ 16000 to the yaw value)
                                                           #              "TimeStamp" - UT Time when the msg was received
        self.incoming_projectile_expire_time_type = "ut_time" # means in which units we will measure the time the information should be valid for items in dictionary                                                      
        self.incoming_projectile_expire_time = float(0.5) # for how long the item will survive

        # from KIL, DIE

        self.was_killed_by = {} # indexed by player ID, Player ID's last killer was 'value'["Killer"]
                                # contains ... "Id" - unique ID of player (who was killed)
                                #              "Killer" - unique ID of player that killed them if any (may have walked off a ledge) 
                                #              "DamageType" - a string describing what kind of damage killed them
                                #              "TimeStamp" - UT Time when the msg was received 
                                
        self.killed_who = {} # indexed by player ID, Player ID's last victim was 'value'["Victim"]
                             # contains ... "Id" - unique ID of player (who is the killer)
                             #              "Victim" - unique ID of player who was killed
                             #              "DamageType" - a string describing what kind of damage killed them
                             #              "TimeStamp" - UT Time when the msg was received

        # from DAM

        self.last_damage_taken = { "Damage": 0, "DamageType": "", "TimeStamp": -1 } # contains ... "Damage" - amount of damage taken
                                                                                    #              "DamageType" - a string describing what kind of damage
                                                                                    #              "TimeStamp" - UT Time when the msg was received
        self.last_damage_taken_accuracy = float(0.3) # if you call self.bot_is_being_damage() ... actual_ut_time minus TimeStamp of last_damage_taken
                                              #                                            must be under this value to report true                                   
                                              # remember that initial value of last_falling["TimeStamp"] must be greater then this
    
        # from HIT

        self.hit_another_player = Expire_Dictionary(self) # indexed by Player Id (who was hit)
                                                          # contains ... "Id" - unique ID of player hit
                                                          #              "Damage" - amount of damage done
                                                          #              "DamageType" - a string describing what kind of damage 
                                                          #              "TimeStamp" - UT Time when the msg was received
        self.hit_another_player_expire_time_type = "ut_time" # means in which units we will measure the time the information should be valid for items in dictionary                                                      
        self.hit_another_player_expire_time = float(0.5) # for how long the item will survive

        # from PTH

        self.requested_path = {} # indexed by specified ID (when sending the request for the path)
                                 # note that Bot won't do anything with those items, just insert them ... if needed you have to delete them by yourself
                                 # under each key will be a dictionary with keys "Id" - provided Id and
                                 #                                          keys 0, 1, 2, ... containing strings: "navpoint_id x,y,z"
                                 # as read: ... Multiple pathnodes: A variable number of attr items will be returned, one for each pathnode that
                                 #                                  needs to be taken. They will be listed in the order in which they should be travled to.
                                 #                                  Each one is of form "{0 id 3,4,5}", with the number of the node (starting with 0)
                                 #                                                        ^ --- is interpreted as a key, rest as a string      
                                 #                                  followed by a space, then a unique id for the node (will never have a space)
                                 #                                  then a location of that node.

        # from RCH

        self.requested_reachcheck = {} # indexed by specified ID (when sending the reachcheck request)
                                       # note that Bot won't do anything with those items, just insert them ... if needed you have to delete them by yourself
                                       # contains: "Id" - an id matching the one sent by client. Allows bot to match answer with right querry.
                                       #           "Reachable" - true if the bot can run here directly, false otherwise 
                                       #           "From" - exact location of bot at time of check ... WARNING -> this won't be translated to a tuple as a default!

        # from FIN

        self.game_finished = 0 # when FIN msg received, set to 1

        # from PONG

        self.last_pong = { "TimeStamp": -1 }
        self.last_pong_accuracy = float(0.4)  # if you call self.pong_received() ... actual_ut_time minus TimeStamp of last_pong
                                       #                                      must be under this value to report true                                   
                                       # remember that initial value of last_pong["TimeStamp"] must be greater then this

   # ADDITIONAL USEFUL INFORMATIONS

        # actualy everything you see here I've found useful while writing different bots
        
        self.last_goto_positions = {} # can serve as a memory ... each runto / strafe command can be indexed, if not index 'last_goto' is provided
                                      # and location / target is written here, each item is tuple (type_of_item, item),
                                      # type_of_item can be either "Location" or "Target" ... "Location" -> item is a string "x,y,z",
                                      #                                                       "Target" -> item is UT unique ID / provided target

   # CORRECTIONS

        self.runto_correction = (0, 0, 10) # it seems that GB doesn't handle steps well, but if we raise Z value of the location to run to, it's OK

 # ***************
 # END OF __INIT__
 # ***************
        
    # this method will starts the bot (connect to GB, start scheduler and executor)
    def start(self):
        self.connect()
        self.scheduler.execute_scheduler()
        self.executor.execute_executor()

    # somthing like a "finalizer" ;-) will close connection, stop scheduler and executor
    def end(self, limit = 10):
        
        self.disconnect()
        self.executor.stop_executor()        
        self.scheduler.stop_scheduler()

        self.debug(5, "Bot - Waiting for connection to GB to be closed.")
        counter = 0
        while self.conn_thread_active and counter < limit:
            counter += 1
            time.sleep(1)

        if counter < limit:
            self.debug(5, "Bot - disconnected from GB.")
        else:
            self.debug(1, "Bot - disconnection timed out, problem?")

        self.debug(5, "Bot - Waiting for Bot_Executor to be stopped.")
        counter = 0
        while self.bot_executor.exec_thread_active and counter < limit:
            counter += 1
            time.sleep(1)

        if counter < limit:
            self.debug(5, "Bot - Bot_Executor stopped.")
        else:
            self.debug(1, "Bot - Bot_Executor still running (timed out), problem?")    

        self.debug(5, "Bot - Waiting for Bot_Scheduler to be stopped.")
        counter = 0
        while (self.bot_scheduler.ut_time_thread_active or self.bot_scheduler.system_time_thred_active) and counter < limit:
            counter += 1
            time.sleep(1)

        if counter < limit:
            self.debug(5, "Bot - Bot_Scheduler stopped.")
        else:
            self.debug(1, "Bot - Bot_Scheduler still running (timed out), problem?")           
            
        if (self.log_gb_msg != None):
            self.log_gb_msg.end()

    # use this method to add a function as a reaction on a certain message type send / received (depends on param message_type) - will be called then
    # this method should accept one parametr which will be a dictionary of params
    def add_reaction_to_msg(self, message_type, function_to_call, priority = 10):
        self.reaction_list.insert(message_type, function_to_call, priority)        

    # adds functions to executer ... assumes that function_list consists of Function_To_Run objects
    def insert_to_executor(self, function_to_run_list, message_values):
        for fn_item in function_to_run_list:
            fn = fn_item.get_fn()
            self.executor.insert(lambda: fn(message_values))

# ***********************************************************
# FOLLOWS METHODS WHICH ENSURE THE COMMUNICATION WITH GAMEBOT
# ***********************************************************
    
    def proc_item(self, string): # parse message from GB, returns tuple (command, dictionary_of_values)
                                 
        (cmd, varstring) = re.compile('\s+').split(string, 1) #\s is a special escape character, which matches any white-space, varstring will hold flags returned from regular expression creation (see sre.py)
        vars = re.compile('\{(.*?)\}').findall(varstring)
        var_dict = {}
        
        for var in vars:
            (attr, value) = re.compile('\s+').split(var, 1)
            var_dict[attr] = value
            
        return (cmd, var_dict)

    def connect(self): # calls connect_thread in a new thread
        
        if not self.conn_thread_id:
            self.debug(5, "Connecting to GameBot Server (" + str(self.ip) + " " + str(self.port) + ")") 
            self.conn_thread_id = thread.start_new_thread(self.connect_thread, ())  # spusteni noveho vlakna, ve kterem se vykonava connect_thread, viz nize
            return 1
        else:
            self.debug(1, "Attempting to Connect() when thread already active")
            return 0

    def disconnect(self): # sets kill_connection = 1, should ends the reading loop if exists and then ends the conn_thred
        if self.conn_thread_active:
            self.debug(5, "Disconnecting. Setting bot.kill_connection = 1.")    
            self.kill_connection = 1

    def report_gb_message(self, type_of_msg, cmd, message): # type of msg is either "sent" or "received"
        if self.call_gb_monitor != None:
            self.call_gb_monitor(type_of_msg, cmd, message) # report the communication with GB

        if self.log_gb_msg != None:
            if type_of_msg == "sent":
                predpona = "-> " # sent
            else:
                predpona = "<- " # receives
            self.log_gb_msg.log_msg(predpona + " " + str(self.actual_ut_time) + " - " + message) # log the msg into a file

    def construct_gb_command(self, cmd, dict = {}):
        string = cmd # do teto promenne string postupne tvorime command, ktery odesleme GB/UT
        
        for (attr, value) in dict.items(): # postupne pripisujeme optioniky do retezce
            string = string + " {" + str(attr) + " " + str(value) + "}"

        string = string + "\r\n" # ukonceni retezce
        
        return string

    def send_message(self, cmd, dict = {}): # posila zpravu GB/UT servriku ... parametrem je retezec prikazu a slovnik [jmeno_atributu] -> hodnota
                                            # pro jednotlive optioniky viz dokumentace k API GameBots
                                       
        string = self.construct_gb_command(cmd, dict)
        print "MSG:", string

        self.report_gb_message("sent", cmd, string) # ohlaseni odeslani zpravy GB

        try:
            self.sockout.write(string) # poslani zpravy do socketu
            self.sockout.flush()       # vyprazdneni pripadneho bufferu
        except:
            self.debug(1, "Bot - Message was unable to send to GB: " + string )
            self.report_gb_message("sent", cmd, "Bot - Message was unable to send to GB: " + string)
            return 0
        else:
            self.insert_to_executor(self.reaction_list.get_list(cmd), dict) # tells executor to execute functions from reaction_list with dict as a parametr
            return 1

    def connect_thread(self): # tahle metoda se nejprve pokusi pripojit na GB/UT a po te opakovane cte ze socketu prichozi informace
                              # pricemz meni stav bota ... proste dochazi ke spracovavani informaci, ktere nam GB/UT posilaji
                              # pro jednotlive zpravy viz manualek k GameBots server

        self.conn_thread_active = 1   # thread byl vytvoren, zaciname praci threadu
        self.kill_connection = 0 # jeste koncit nebudeme
        try:  # pokus o pripojeni na server ... pri failu ... problem se siti / server neexistuje
            self.sockobj = socket(AF_INET, SOCK_STREAM)
            self.sockobj.connect((self.ip, int(self.port)))
            self.sockin = self.sockobj.makefile('r')
            self.sockout = self.sockobj.makefile('w')
        except:
            self.debug(1, "Bot - Connection to server failed.")
            self.kill_connection = 1 # koncime ... preskakuje zaroven nasledujici smycku
        else:
            self.debug(1, "Bot - Connected to server.")
            self.kill_connection = 0
        
        while not self.kill_connection: # cekame na prvni zpravu od serveru,  :-)
            try:
                x = self.sockin.readline()
            except:
                self.debug(1, "Bot - Connection Error on readline(), first NFO message could not been received.")
                self.kill_connection = 1
                break
                
            if not x:
                self.debug(1, "Bot - Connection Closed from Remote End, server didn't send first NFO message.")
                self.kill_connection = 1
                break
            
            (cmd, dict) = self.proc_item(x) # rozparsovani zpravy
            self.report_gb_message("received", cmd, x) # ohlaseni prijeti zpravy od GB
            #if cmd == "HELLO BOT" or cmd == "HELLO CONTROL SERVER": # tohle by to melo byt, ale pro jistotu
                # posleme INIT command, kterym GB inicializuje noveho bota v UT, prideli mu ID a zacne nam posilat zpravy o jeho zivote
                #self.conninfo = dict
                #self.send_message("INIT", {"Name" : self.botname})
            self.debug(1, "Starting the communication")
            self.conn_ready = 1 # Ready to send messages
            break

        # nyni prichazi hlavni smycka, ktera postupne cte informace posilane GB/UT a spracovava je
        while not self.kill_connection:
            try:
                x = self.sockin.readline()
            except:
                self.debug(1, "bot: Connection Error on readline() ... in main loop.")
                break
                
            if not x:
                self.debug(1, "bot: Connection Closed from Remote End ... in main loop.")
                break
            
            (cmd, dict) = self.proc_item(x) # rozparsovani prikazu (prichazi jako big string)
            self.report_gb_message("received", cmd, x) # ohlaseni prijeti zpravy od GB
            self.process_gb_message(cmd, dict); # zpracovani message od servriku, viz hned dalsi metoda

        # koncime a uklizime
        self.debug(5, "bot: Closing Sockets and Cleaning Up...") 
        try:
            self.sockout.flush()
            self.sockout.close()
            self.sockin.close()
            self.sockobj.close()
        except:
            self.debug(1, "bot: Error closing files and sockets.")

        self.conn_ready = 0         # connection byla zavrena == zrusena
        self.conn_thread_active = 0 # thread konci
        self.conn_thread_id = None  # thread konci
        self.debug(5, "bot: Connection Thread Terminating...")

 # ************************************************************
 # NASLEDUJI FUNKCE, KTERE ZAJISTUJI PRIJMANI ZPRAV OD GAMEBOTa
 # ************************************************************

    def process_gb_message(self, cmd, dict): # tato metoda se zabyva zpracovanim zpravy, ktera prisla od GB/UT
                                             # vstupnimi parametry je string s nazvem prikazu a dale slovnik [jmeno_atributu] -> hodnota
                                             # tahle metoda je zodpovedna za udrzovani aktualni podoby agenta

        if dict.has_key("Location"):
            dict["LocationTuple"] = location_string_to_tuple(dict["Location"])
        if dict.has_key("Rotation"):
            dict["RotationTuple"] = rotation_string_to_tuple(dict["Rotation"])
            dict["RotationDegreesTuple"] = ( ut_angle_to_angle(dict["RotationTuple"][0]), ut_angle_to_angle(dict["RotationTuple"][1]),
                                             ut_angle_to_angle(dict["RotationTuple"][2]) )            

        if self.gb_synchronous_msg.has_key(cmd):
            return_value = self.process_gb_synchronous_message(cmd, dict)
        else:
            return_value = self.process_gb_asynchronous_message(cmd, dict)

        self.insert_to_executor(self.reaction_list.get_list(cmd), dict) # tells executor to execute functions from reaction_list with dict as a parametr
        return return_value                                

    # serves synchronous messages
    def process_gb_synchronous_message(self, cmd, values):
        if self.gb_synchronous_msg.has_key(cmd):
            return self.gb_synchronous_msg[cmd](values)
        else:
            return 0

    # serves asynchronous messages
    def process_gb_asynchronous_message(self, cmd, values):
        if self.gb_asynchronous_msg.has_key(cmd):
            return self.gb_asynchronous_msg[cmd](values)
        else:
            return 0

 # *******************************
 # PROCESSING SYNCHRONOUS MESSAGES
 # *******************************

    def process_beg(self, values):
        self.actual_ut_time = float(values["Time"])

    def process_slf(self, values):
        self.previous_bot_status = self.bot_status
        self.bot_status = values

    def process_gam(self, values):
        self.game_status = values

    def process_plr(self, values):
        values["TimeStamp"] = self.actual_ut_time # we add TimeStamp to infos

        if values["Team"] == self.get_bot_team():
            self.see_players["Friend"].safe_insert(values["Id"],
                                                   Expire_Dictionary_Item(self,                              # BOT
                                                                          self.see_players["Friend"],        # owner of the item
                                                                          values["Id"],                      # key of the item
                                                                          values,                            # value to store
                                                                          self.see_players_expire_time_type, # time_type for scheduling
                                                                          self.see_players_expire_time) )    # time before expiration
                                                                                              
        else:
            self.see_players["Enemy"].safe_insert(values["Id"],
                                                  Expire_Dictionary_Item(self,                              # BOT
                                                                         self.see_players["Enemy"],         # owner of the item
                                                                         values["Id"],                      # key of the item
                                                                         values,                            # value to store
                                                                         self.see_players_expire_time_type, # time_type for scheduling
                                                                         self.see_players_expire_time) )    # time before expiration

  
        self.last_known_positions[values["Id"]] = values

    def process_nav(self, values):
        values["TimeStamp"] = self.actual_ut_time # we add TimeStamp to infos

        self.see_navpoints.safe_insert(values["Id"],
                                       Expire_Dictionary_Item(self,                                # BOT
                                                              self.see_navpoints,                  # owner of the item
                                                              values["Id"],                        # key of the item
                                                              values,                              # value to store
                                                              self.see_navpoints_expire_time_type, # time_type for scheduling
                                                              self.see_navpoints_expire_time) )    # time before expiration

        if not self.known_navpoints.has_key(values["Id"]):
            self.known_navpoints[values["Id"]] = values

    def process_mov(self, values):
        values["TimeStamp"] = self.actual_ut_time # we add TimeStamp to infos

        self.see_movers.safe_insert(values["Id"],
                                    Expire_Dictionary_Item(self,                             # BOT
                                                           self.see_movers,                  # owner of the item
                                                           values["Id"],                     # key of the item
                                                           values,                           # value to store
                                                           self.see_movers_expire_time_type, # time_type for scheduling
                                                           self.see_movers_expire_time) )    # time before expiration

        if not self.known_movers.has_key(values["Id"]):
            self.known_movers[values["Id"]] = values

    def process_dom(self, values):
        values["TimeStamp"] = self.actual_ut_time # we add TimeStamp to infos

        self.see_dom_points.safe_insert(values["Id"],
                                        Expire_Dictionary_Item(self,                                 # BOT
                                                               self.see_dom_points,                  # owner of the item
                                                               values["Id"],                         # key of the item
                                                               values,                               # value to store
                                                               self.see_dom_points_expire_time_type, # time_type for scheduling
                                                               self.see_dom_points_expire_time) )    # time before expiration

        if not self.known_dom_points.has_key(values["Id"]):
            self.known_dom_points[values["Id"]] = values

    def process_flg(self, values):
        values["TimeStamp"] = self.actual_ut_time # we add TimeStamp to infos

        self.see_flags[values["Team"]].safe_insert(values["Id"],
                                                   Expire_Dictionary_Item(self,                            # BOT
                                                                          self.see_flags[values["Team"]],  # owner of the item
                                                                          values["Id"],                    # key of the item
                                                                          values,                          # value to store
                                                                          self.see_flags_expire_time_type, # time_type for scheduling
                                                                          self.see_flags_expire_time) )    # time before expiration 

        if values["State"] == "Home":
            if not self.known_flags.has_key[values["Team"]]:
                self.known_flags[values["Team"]] = values

    def process_inv(self, values):
        values["TimeStamp"] = self.actual_ut_time # we add TimeStamp to infos

        if self.see_items.has_key(values["Type"]):
            self.see_items[values["Type"]].safe_insert(values["Id"],
                                                        Expire_Dictionary_Item(self,                            # BOT
                                                                               self.see_items[values["Type"]], # owner of the item
                                                                               values["Id"],                    # key of the item
                                                                               values,                          # value to store
                                                                               self.see_items_expire_time_type, # time_type for scheduling
                                                                               self.see_items_expire_time) )    # time before expiration
        else:
            self.see_items[values["Type"]] = Expire_Dictionary(self)
            self.see_items[values["Type"]].safe_insert(values["Id"],
                                                        Expire_Dictionary_Item(self,                            # BOT
                                                                               self.see_items[values["Type"]], # owner of the item
                                                                               values["Id"],                    # key of the item
                                                                               values,                          # value to store
                                                                               self.see_items_expire_time_type, # time_type for scheduling
                                                                               self.see_items_expire_time) )    # time before expiration

        if not self.all_known_items.has_key(values["Id"]):
            self.all_known_items[values["Id"]] = values
        
        if self.known_items_by_category.has_key(values["Type"]):
            if not self.known_items_by_category.has_key(values["Id"]):
                self.known_items_by_category[values["Type"]][values["Id"]] = values
        else:
            self.known_items_by_category[values["Type"]] = {}
            self.known_items_by_category[values["Type"]][values["Id"]] = values

    def process_end(self, values):
        pass

 # ********************************
 # PROCESSING ASYNCHRONOUS MESSAGES
 # ********************************

    def process_nfo(self, values): # actualy this is sent only once and caputured in self.connect_thread not by self.process_gb_message
                                   # ... if received again, don't know what to think about - debug msg is sent
        self.conn_info = values
        self.debug(1, "Bot - another NFO message received... what's going on?")

    def process_ain(self, values): # so far I haven't had luck to find all the classes of the items bot can pickup ...
                                   # in param values is a dictionary with keys ["Id"] / ["Type"]     

        self.debug(1, "Bot - picked up an item, class: '" + values["Type"] + "'" )

    def process_vms(self, values):
        values["TimeStamp"] = self.actual_ut_time
        self.global_messages.append(values)
        trunc_list(self.global_messages, self.global_messages_max_size)

    def process_vmt(self, values):
        values["TimeStamp"] = self.actual_ut_time
        self.private_messages.append(values)
        trunc_list(self.private_messages, self.private_messages_max_size)

    def process_vmg(self, values):
        values["TimeStamp"] = self.actual_ut_time
        self.tokenized_messages.append(values)
        trunc_list(self.tokenized_messages, self.tokenized_messages_max_size)

    def process_zcf(self, values):
        values["TimeStamp"] = self.actual_ut_time
        self.bot_zones["Foot"] = values


    def process_zch(self, values):
        values["TimeStamp"] = self.actual_ut_time
        self.bot_zones["Head"] = values


    def process_zcb(self, values):
        values["TimeStamp"] = self.actual_ut_time
        self.bot_zones["Head"] = values
        self.bot_zones["Body"] = values
        self.bot_zones["Foots"] = values

    def process_cwp(self, values):
        self.bot_status["Weapon"] = values["Type"]

    def process_wal(self, values):
        values["TimeStamp"] = self.actual_ut_time
        self.last_collision = values

    def process_fal(self, values):
        values["TimeStamp"] = self.actual_ut_time
        self.last_falling = values

    def process_bmp(self, values):
        values["TimeStamp"] = self.actual_ut_time # we add TimeStamp to infos

        self.bump_another_actor.safe_insert(values["Id"],
                                            Expire_Dictionary_Item(self,                                     # BOT
                                                                   self.bump_another_actor,                  # owner of the item
                                                                   values["Id"],                             # key of the item
                                                                   values,                                   # value to store
                                                                   self.bump_another_actor_expire_time_type, # time_type for scheduling
                                                                   self.bump_another_actor_expire_time) )    # time before expiration

    def process_hrp(self, values):
        values["TimeStamp"] = self.actual_ut_time # we add TimeStamp to infos

        self.hear_pickup.safe_insert(values["Player"],
                                     Expire_Dictionary_Item(self,                              # BOT
                                                            self.hear_pickup,                  # owner of the item
                                                            values["Player"],                  # key of the item
                                                            values,                            # value to store
                                                            self.hear_pickup_expire_time_type, # time_type for scheduling
                                                            self.hear_pickup_expire_time) )    # time before expiration

    def process_hrn(self, values):
        values["TimeStamp"] = self.actual_ut_time # we add TimeStamp to infos

        if not values.has_key("Source"):
            self.debug(1, "Bot - HRN msg doesn't have Source attribute!")
            if values.has_key("SourceClass"):
                self.debug(5, "Bot - HRN msg have SourceClass attribute converted to Source.")
                values["Source"] = values["SourceClass"]
            else:
                self.debug(1, "Bot - HRN msg couldn't be processed.")
                return            
        
        self.hear_noise.safe_insert(values["Source"],
                                    Expire_Dictionary_Item(self,                             # BOT
                                                           self.hear_noise,                  # owner of the item
                                                           values["Source"],                 # key of the item
                                                           values,                           # value to store
                                                           self.hear_noise_expire_time_type, # time_type for scheduling
                                                           self.hear_noise_expire_time) )    # time before expiration       

    def process_see(self, values):
        values["TimeStamp"] = self.actual_ut_time # we add TimeStamp to infos

        if values["Team"] == self.get_bot_team():
            self.see_players["Friend"].safe_insert(values["Id"],
                                                   Expire_Dictionary_Item(self,                              # BOT
                                                                          self.see_players["Friend"],        # owner of the item
                                                                          values["Id"],                      # key of the item
                                                                          values,                            # value to store
                                                                          self.see_players_expire_time_type, # time_type for scheduling
                                                                          self.see_players_expire_time) )    # time before expiration
                                                                                              
        else:
            self.see_players["Enemy"].safe_insert(values["Id"],
                                                  Expire_Dictionary_Item(self,                              # BOT
                                                                         self.see_players["Enemy"],         # owner of the item
                                                                         values["Id"],                      # key of the item
                                                                         values,                            # value to store
                                                                         self.see_players_expire_time_type, # time_type for scheduling
                                                                         self.see_players_expire_time) )    # time before expiration

  
        self.last_known_positions[values["Id"]] = values
    
    def process_prj(self, values):
        values["TimeStamp"] = self.actual_ut_time # we add TimeStamp to infos
        time = time.clock() # should be more accurate then time.time() ... more "unique" :-)
        self.incoming_projectile.safe_insert(time,
                                             Expire_Dictionary_Item(self,                                      # BOT
                                                                    self.incoming_projectile,                  # owner of the item
                                                                    time,                                      # key of the item
                                                                    values,                                    # value to store
                                                                    self.incoming_projectile_expire_time_type, # time_type for scheduling
                                                                    self.incoming_projectile_expire_time) )    # time before expiration

    def process_kil(self, values):
        values["TimeStamp"] = self.actual_ut_time # we add TimeStamp to infos
        self.was_killed_by[values["Id"]] = values

        if values.has_key("Killer"):
            new_val = { "Id": values["Killer"], "Victim": values["Id"], "DamageType": values["DamageType"], "TimeStamp": values["TimeStamp"] }
            self.killed_who[new_val["Id"]] = new_val
    
    def process_die(self, values):
        values["TimeStamp"] = self.actual_ut_time # we add TimeStamp to infos
        values["Id"] = self.bot_status["Id"]
        self.was_killed_by[values["Id"]] = values
        if values.has_key("Killer"):
            new_val = { "Id": values["Killer"], "Victim": values["Id"], "DamageType": values["DamageType"], "TimeStamp": values["TimeStamp"] }
            self.killed_who[new_val["Id"]] = new_val

    def process_dam(self, values):
        values["TimeStamp"] = self.actual_ut_time # we add TimeStamp to infos
        self.last_damage_taken = values

    def process_hit(self, values):        
        values["TimeStamp"] = self.actual_ut_time # we add TimeStamp to infos

        self.hit_another_player.safe_insert(values["Id"],
                                            Expire_Dictionary_Item(self,                                     # BOT
                                                                   self.hit_another_player,                  # owner of the item
                                                                   values["Id"],                             # key of the item
                                                                   values,                                   # value to store
                                                                   self.hit_another_player_expire_time_type, # time_type for scheduling
                                                                   self.hit_another_player_expire_time) )    # time before expiration

    def process_pth(self, values):
        if values.has_key("Id"): # it suppose to have the key 'Id' according to GB Api
            self.requested_path[values["Id"]] = values
        elif values.has_key("ID"): # but GB send 'ID'
            self.requested_path[values["ID"]] = values
        else:
            self.debug(1, "Bot - PTH received but neither 'Id' nor 'ID' was found.")
                        
    def process_rch(self, values):
        if values.has_key("Id"): # it suppose to have the key 'Id' according to GB Api
            self.requested_reachcheck[values["Id"]] = values
        elif values.has_key("ID"): # but GB send 'ID'
            self.requested_reachcheck[values["ID"]] = values
        else:
            self.debug(1, "Bot - RCH received but neither 'Id' nor 'ID' was found.")


    def process_fin(self, values):
        self.game_finished = 1

    def process_pong(self, values):
        self.last_pong["TimeStamp"] = self.actual_ut_time

 # *****************************************************************************************
 # FUNCTIONS WHICH CAN BE USED FOR SENSES, QUERYING BOT STATE AND OTHER INFORMATIONS FROM UT
 # *****************************************************************************************

    # NOTE 1: none of this function locks any objects, if you do any time consuming operations with objects, don't forget to lock them
    # NOTE 2: accuracy of the informations depends on expiration values of objects! ...
    #         default 0.5 seconds, if needed change them, or add check for TimeStamp of the informations / expiration
    # NOTE 3: some objects are returned as Expire_Dictionary or Expire_Dictionary_Item (only if member of Expire_Dictionary instance) ...
    #         check __init__ to see what's Expire_Dictionary and what's not

  # BEG

    def get_actual_ut_time(self): # timestamp from the game
        return self.actual_ut_time 

  # SLF  

    def get_bot_id(self): # a unique id, assigned by the game 
        return self.bot_status["Id"]

    def get_bot_rotation(self): # which direction the player is facing in absolute terms ... will be string "pitch,yaw,roll"
        return self.bot_status["Rotation"]

    def get_bot_rotation_tuple(self): # returns location in a tuple (pitch, yaw, roll)
        return rotation_string_to_tuple(self.bot_status["Rotation"])

    def get_bot_location(self): # an absolute location ... will be a string "x,y,z"
        return self.bot_status["Location"]

    def get_bot_location_tuple(self): # returns location as a tuple (x, y, z)
        return self.bot_status["LocationTuple"]

    def get_bot_velocity(self): # absolute velocity in UT units 
        return self.bot_status["Velocity"]

    def get_bot_name(self): # players human readable name 
        return self.bot_status["Name"]

    def get_bot_team(self): # what team the player is on. 255 is no team. 0-3 are red, blue, green, gold in that order
                            # value -1 means we don't have information about what team is bot on
        return self.bot_status["Team"]

    def get_enemy_team_2t(self): # return 0 or 1 ... supposing that there are only 2 teams in game (red == 0, blue == 1)
        if self.get_bot_team() == 0:
            return 1
        else:
            return 0

    def get_enemy_teams_list(self): # returns list with numbers of enemy teams,
                                    # if returns None - means that we don't have information about in which team the bot is
        if self.enemy_teams_numbers != None:
            return self.enemy_teams_numbers
        else:
            if self.conninfo.has_key("MaxTeams"):
                bot_team = int(self.bot_status["Team"])
                if bot_team != -1:
                    teams = []
                    for team_num in range(int(self.conninfo["MaxTeams"])):
                        if team_num != bot_team:
                            teams.append(team_num)
                    self.enemy_teams_numbers = teams
                    return teams
                else:
                    return None
            else:
                return None
    
    def get_bot_health(self): # how much health the bot has left. Starts at 100, ranges from 0 to 200
        return self.bot_status["Health"]

    def get_bot_current_weapon(self): # weapon the player is holding, weapon strings to look for include: "ImpactHammer", "Enforcer", "Translocator",
                                      # "GESBioRifle", "ShockRifle", "PulseGun", "Minigun2", "UT_FlakCannon", "UT_Eightball", "WarheadLauncher"
        return self.bot_status["Weapon"]
    
    def get_bot_current_ammo(self): # how much ammo the bot has left for current weapon
        return self.bot_status["PrimaryAmmo"]
        
    def get_bot_current_alt_ammo(self): # how much ammo the bot has left for current weapon
        return self.bot_status["SecondaryAmmo"]        

    def get_bot_armor(self): # how much armor the bot is wearing, starts at 0, can range up to 200
        return self.bot_status["Armor"]
        
    def get_bot_adrenaline(self):
        return self.bot_status["Adrenaline"]
    

    def _bot_is_rotating_horizontaly(self): # true if bot is rotating horizontaly (left-to-right)
        return self.bot_status["RotationTuple"][1] != self.previous_bot_status["RotationTuple"][1]

    def bot_is_moving(self): # true if bot is changing X / Y location
                             # NOTE: it doesn't suffice to tell that bot bumped to wall! (may still moves)
        return math.fabs(float(self.bot_status["LocationTuple"][0]) - float(self.previous_bot_status["LocationTuple"][0])) > self.bot_moving_accuracy or \
               math.fabs(float(self.bot_status["LocationTuple"][1]) - float(self.previous_bot_status["LocationTuple"][1])) > self.bot_moving_accuracy

    def is_bot_at_location_tuple(self, location_tuple):
        return math.fabs(distance_of_locations(location_tuple, self.bot_status["LocationTuple"])) < self.location_accuracy
               
  # GAM      

    def get_player_scores(self): # player score will have a list of values - one for each player in the game,
                                 # each value will be a list with two values. The first is the id of the player and the second that player's score.
                                 # e.g. "GAM {PlayerScore {player1 2} {player2 5}..."
        return self.game_status["PlayerScores"]

    def get_bot_score(self): # returns a score of the bot, -1 is for an error (the information wasn't received yet)
        return self.get_player_score(self.bot_status["Id"])

    def get_player_score(self, ut_player_id): # returns a score of a specified player ... Id is UT-Id, -1 is for an error
                                              #                                                        (the information wasn't received yet)
        if self.game_status["PlayerScores"].has_key(ut_player_id):
            return self.game_status["PlayerScores"][ut_player_id]
        else:
            return -1

    def get_team_scores(self): # like PlayerScore, but for teams. Team is identified by the team index (same number used to describe team
                               # for PLR and SLF messages. Not sent in normal deathmatch.
        return self.game_status["TeamScores"]

    def get_bot_team_score(self): # returns the score of the team on which the bot is, -1 is for an error (the information wasn't received yet)
        return self.get_team_score(self.bot_status["Team"])

    def get_team_score(self, team_num): # returns the score of a specified team (num from 0 ... 3), -1 is for an error (the information wasn't received yet)
        team_num = str(team_num)
        if self.game_status["TeamScores"].has_key(team_num):
            return self.game_status["TeamScores"][team_num]
        else:
            return -1

    def get_dom_points(self): # like the previous two, this is a multivalued message. This will have one item for each domination point
                              # in a Domination game. First value will be Id of the DOM point, the second will be the index of the team
                              # that owns the domination point.
        return self.game_status["DomPoints"]

    def get_dom_point(seld, dom_point_id): # returns a team who owns the specified dom_point, -1 is for an error (the information wasn't received yet)
        if self.game_status["DomPoints"].has_key(dom_point_id):
            return self.game_status["DomPoints"][dom_point_id]
        else:
            return -1

    def get_bot_have_flag(self): # sent in CTF games if the bot is carrying an enemy's flag. Value is the team number of whose flag you have. 
        return self.game_status["HaveFlag"]

    def get_enemy_has_flag(self): # sent in CTF games if the bot's team's flag has been stolen. Value is meaningless.
        return self.game_status["EnemyHasFlag"]

    def has_enemy_my_flag(self): # synonym for get_enemy_has_flag
        return self.game_status["EnemyHasFlag"]

  # PLR

    def get_see_players(self): # returns the dictionary containing ["Friend"] and ["Enemy"]
        return self.see_players

    def get_see_friends(self): # returns the instance of Expire_Dictionary with friends the bot see -> we can use it to acquire the lock
        return self.see_players["Friend"]

    def get_see_enemies(self): # returns the instance of Expire_Dictionary with enemies the bot see -> we can use it to acquire the lock
        return self.see_players["Enemy"]

    def get_last_known_players_positions(self): # returns dictionary with [Id] -> last player info
        return self.last_known_positions

    def get_last_known_player_position(self, player_id): # returns last known info about player_id, None ... for error (we've never seen this player before)
        if self.last_known_positions.has_key(player_id):
            return self.last_known_positions[player_id]
        else:
            return None

    def see_any_friend(self): # returns true, if we see a friend
        return self.see_players["Friend"].count() > 0

    def see_any_enemy(self): # return true, if we see an enemy
        return self.see_players["Enemy"].count() > 0

    def see_player(self, player_id): # returns true if bot has player_id in the field of view
        return self.see_players["Friend"].has_key(player_id) or self.see_players["Enemy"].has_key(player_id)

  # NAV

    def get_see_navpoints(self): # returns the instance of Expire_Dictionary -> we can use it to acquire the lock
        return self.see_navpoints

    def see_any_navpoint(self): # returns true, if we see at least one nav-point
        return self.see_navpoints.count() > 0

    def see_navpoint(self, navpoint_id): # returns true, if we see specified nav point
        return self.see_navpoints.has_key(navpoint_id)

    def get_known_navpoints(self): # return dictionary with known navpoints
        return self.known_navpoints

    def know_navpoint(self, navpoint_id): # returns true, if we've already seen specified navpoint
        return self.known_navpoints.has_key(navpoint_id)

    def get_navpoint(self, navpoint_id): # returns specified navpoint, if returns None ... we've never seen navpoint with this ID
        if self.known_navpoints.has_key(navpoint_id):
            return self.known_navpoints[navpoint_id]
        else:
            return None

  # MOV

    def get_see_movers(self): # returns the instance of Expire_Dictionary -> we can use it to acquire the lock
        return self.see_movers

    def see_any_mover(self): # returns true, if we see at least one movers
        return self.see_movers.count() > 0

    def see_mover(self, mover_id): # returns true, if we see specified mover
        return self.see_movers.has_key(mover_id)

    def get_known_movers(self): # return dictionary with known movers
        return self.known_movers

    def know_mover(self, mover_id): # returns true, if we've already seen specified mover
        return self.known_movers.has_key(mover_id)

    def get_mover(self, mover_id): # returns specified mover, if returns None ... we've never seen mover with this ID
        if self.known_movers.has_key(mover_id):
            return self.known_movers[mover_id]
        else:
            return None

  # DOM

    def get_see_dom_points(self): # returns the instance of Expire_Dictionary -> we can use it to acquire the lock
        return self.see_dom_points

    def see_any_dom_point(self): # returns true, if we see at least one dom-point
        return self.see_dom_points.count() > 0

    def see_dom_point(self, dom_point_id): # returns true, if we see specified dom point
        return self.see_dom_points.has_key(dom_point_id)

    def get_known_dom_points(self): # return dictionary with known navpoints
        return self.known_dom_points

    def know_dom_point(self, dom_point_id): # returns true, if we've already seen specified navpoint
        return self.known_dom_points.has_key(dom_point_id)

    def get_dom_point(self, dom_point_id): # returns specified navpoint, if returns None ... we've never seen navpoint with this ID
        if self.known_dom_points.has_key(dom_point_id):
            return self.known_dom_points[dom_point_id]
        else:
            return None

  # FLG

    def get_see_flags(self): # returns the instance of Expire_Dictionary -> we can use it to acquire the lock
        return self.see_flags

    def see_flag(self, team_num): # returns true, if we see specified flag (team number)
        team_num = str(team_num)
        if self.see_flags.has_key(team_num):
            return self.see_flags[team_num].count() > 0
        else:
            return 0

    def get_known_flags(self): # return dictionary with known flags
        return self.known_flags

    def know_flag(self, team_num): # returns true, if we've already seen specified flag (team number)
        return self.known_flags.has_key(team_num)

    def get_flag(self, team_num): # returns specified flag, if returns None ... we've never seen a flag of this team
        if self.known_flags.has_key(team_num):
            return self.known_flags[team_num]
        else:
            return None

  # INV

    def get_see_items(self): # returns the instance of Expire_Dictionary -> we can use it to acquire the lock
        return self.see_items

    def see_any_item(self, category): # returns true, if we see at least one item of specified category
        if self.see_items.has_key(category):
            return self.see_flags[category].count() > 0
        else:
            return 0

    def see_item(self, category, item_id): # returns true, if we see specified item
        if self.see_items.has_key(category):
            return self.see_items[category].has_key(item_id)
        else:
            return None

    def get_all_known_items(self): # return dictionary with known items
        return self.all_known_items

    def get_known_items_by_category(self): # return dictionary with known items by category
        return self.known_items_by_category
    
    def know_item(self, item_id): # returns true, if we've already seen specified item
        return self.all_known_items.has_key(item_id)

    def get_item(self, item_id): # returns specified item, if returns None ... we've never seen an item with this ID
        if self.all_known_items.has_key(item_id):
            return self.all_known_items[item_id]
        else:
            return None

  # VMS
    
    def get_global_messages(self): # returns list of messages from global channel
                                   # each item is dictionary with: "String" - human readable message sent by another player in the game
                                   #                               "TimeStamp" - UT Time when the msg was received
        return self.global_messages                                   

  # VMT

    def get_private_messages(self): # returns list of messages from private channel
                                    # each item is dictionary with: "String" - human readable message sent by another player in the game
                                    #                               "TimeStamp" - UT Time when the msg was received
        return self.private_messages

  # VMG

    def get_tokenized_messages(self): # returns list of tokenized messages
                                      # each item is dictionary with: "Sender" - unique id of player who sent message
                                      #                               "Type" - type of message (e.g. Command, Taunt, etc...) 
                                      #                               "Id" - message id. specifies which message is being sent                                     
                                      #                               "TimeStamp" - UT Time when the msg was received
        return self.tokenized_messages                                      
        

  # ZCH / ZCF / ZCB

    def get_head_zone(self): # returns zone of the bot's head ... "Id" - unique zone, "TimeStamp" - UT Time when the msg was received
        return self.bot_zones["Head"]

    def get_body_zone(self): # returns zone of the bot's body ... "Id" - unique zone, "TimeStamp" - UT Time when the msg was received
        return self.bot_zones["Body"]

    def get_foots_zone(self): # returns zone of the bot's foots ... "Id" - unique zone, "TimeStamp" - UT Time when the msg was received
        return self.bot_zones["Foots"]

    
  # WAL

    def bot_collide(self):
        return float(self.actual_ut_time) - float(self.last_collision["TimeStamp"]) < self.collision_accuracy

    def get_last_collision(self): # contains ... "Id" - unique ID of the wall you've collide with
                                  #              "Normal" - normal of the angle bot colided at.
                                  #              "Location" - absolute location of bot at time of impact
                                  #              "TimeStamp" - UT Time when the msg was received
        return self.last_collision

  # FAL

    def bot_is_falling(self):
        return self.actual_ut_time - self.last_falling["TimeStamp"] - self.falling_accuracy

    def get_last_falling(self): # contains ... "Fell" - True if you fell. False if you stopped at edge. 
                                #              "Location" - absolute location of bot
                                #              "TimeStamp" - UT Time when the msg was received
        return self.last_falling                                

  # BMP

    def bot_bumped_to_object(self): # wether bot bumper to something or not
        return self.bump_another_actor.count() > 0

    def get_bump_another_actor(self): # returns Expire_Dictionary -> we can use it to acquire the lock
        return self.bump_another_actor

  # HRP

    def bot_heard_pickup(self): # wether bot heard a pickup sound or not
        return self.hear_pickup.count() > 0

    def get_hear_pickup(self): # returns Expire_Dictionary -> we can use it to acquire the lock
        return self.hear_pickup

  # HRN

    def bot_heard_noise(self): # wether bot heard a noise sound or not
        return self.hear_noise.count() > 0

    def get_hear_noise(self): # returns Expire_Dictionary -> we can use it to acquire the lock
        return self.hear_noise

  # PRJ

    def projectile_is_incoming(self): # wether some projectile is incomind or not
        return self.incoming_projectile.count() > 0

    def get_incoming_projectile(self): # returns Expire_Dictionary -> we can use it to acquire the lock
        return self.incoming_projectile

  # KIL, DIE

    def get_was_killed_by(self): # indexed by player ID, Player ID's last killer was 'value'["Killer"]
        return self.was_killed_by

    def get_killed_who(self): # indexed by player ID, Player ID's last victim was 'value'["Victim"]
        return self.killed_who
        
  # DAM     

    def bot_is_being_damaged(self):
        return self.actual_ut_time - self.last_damage_taken["TimeStamp"] < self.last_damage_taken_accuracy

    def get_last_damage_taken(self): # contains ... "Damage" - amount of damage taken
                                     #              "DamageType" - a string describing what kind of damage
                                     #              "TimeStamp" - UT Time when the msg was received
        return self.last_damage_taken

        self.last_damage_taken = { "Damage": 0, "DamageType": "", "TimeStamp": 0 } 
    
  # HIT

    def bot_hit_player(self, player_id): # wether bot hit specified player
        return self.hit_another_player.has_key(player_id)

    def get_hit_another_player(self): # returns Expire_Dictionary -> we can use it to acquire the lock
        return self.hit_another_player

    def get_hit_player(self, player_id): # returns info about hitting specified player -> None means the bot didn't hit the player / bot
        return self.hit_another_player.get(player_id)
        

  # PTH

    def have_requested_path(self, path_id): # wether requested path was obtained
        return self.requested_path.has_key(path_id)

    def get_specific_path(self, path_id): # returns specified path (None means that path haven't come yet), for info how the data look see __init__
        if self.requested_path.has_key(path_id):
            return self.requested_path[path_id]
        else:
            return None

    def get_requested_path(self): # returns dictionary with paths
        return self.requested_path

  # RCH

    def have_requested_reachcheck(self, reachcheck_id): # wethver requested reachcheck was received
        return self.requested_reachcheck.has_key(reachcheck_id)

    def get_specific_reachcheck(self, reachcheck_id): # returns reachcheck (None means that path haven't come yet)
        if self.requested_reachcheck.has_key(reachcheck_id): # contains: "Id" - an id matching the one sent by client. Allows bot to match answer with right querry.
                                                             #           "Reachable" - true if the bot can run here directly, false otherwise 
                                                             #           "From" - exact location of bot at time of check ... WARNING -> this won't be translated to a tuple as a default!
            return self.requested_reachcheck[reachcheck_id]
        else:
            return None

    def get_requested_reachcheck(self): # returns dictionary with reachchecks
        return self.requested_reachcheck

  # FIN

    def game_is_over(self): # wether game has ended
        return self.game_finished

  # PONG

    def pong_received(self): # wether pong was received lately
        return self.actual_ut_time - self.last_pong["TimeStamp"] < self.last_pong_accuracy

  # follows GET / ... methods which query the "memory" of the bot ... like self.last_goto_positions

    def get_last_goto_positions(self): # returns the dictionary
        return self.last_goto_positions

    def get_last_goto(self): # returns "last_goto" item from self.last_goto_positions
        if self.last_goto_positions.has_key("last_goto"):
            return self.last_goto_positions["last_goto"]
        else:
            return None

    def get_last_goto_item(self, index): # retuns secified item from self.last_goto_positions
        if self.last_goto_positions.has_key(index):
            return self.last_goto_positions[index]
        else:
            return None

 # **********************************
 # METHODS SENDING MESSAGE TO GAMEBOT
 # **********************************

    # NOTE: usually command has two types of method ... send_'name' / send_'name'_dict ... in the first type you should specify params in function call
    #                                                                                  ... in the second one you should provide dictionary
    #                                                                                  ... with [attr_name] -> value

  # INIT ... # message you send to spawn a bot in the game world, you must send this message before you have a character to play in the game.
             # DO NOT SEND UNTIL YOU RECIEVE NFO MESSAGE FROM SERVER
             # NOTE: this msg is send automaticaly in self.connect_thread() ! ... you shouldn't use this   

    def send_init(self,
                  name = "BODbot",  # Desired name. If in use already or argument not provided, one will be provided for you.
                  team = 0): # Prefered team. If illegal or no team provided, one will be provided for you.
                             # Normally a team game has team 0 and team 1. In BotDeathMatchPlus, team is meaningless,
                             # but this will still set you skin color to match what you select
        self.send_message("INIT", { "Name": name, "Team": team } )

    def send_init_dict( self, dict ):
        self.send_message("INIT", dict)

  # SETWALK

    def send_walk(self): # to go into walk mode - you move at about 1/3 normal speed, make less noise, and won't fall off ledges
                         # doesn't mean that bot will start walking / running
        self.send_message("SETWALK", { "Walk": "True" } )

    def send_run(self):
        self.send_message("SETWALK", { "Walk": "Run" } )

  # STOP

    def send_stop(self): # stop the movement / rotation
        self.send_message("STOP", {} )

  # JUMP

    def send_jump(self): # causes bot to jump. Not very useful (of GB) yet, working on this one.
        self.send_message("JUMP", {} )

  # RUNTO ... # turn towards and move directly to your destination, may specify destination via either Target or Location argument,
              # will be parsed in that order. (i.e. if Target provided, Location will be ignored). If you select an impossible place to head to,
              # you will start off directly towards it until you hit a wall, fall off a cliff, or otherwise discover the offending obstacle.

    def send_run_to_target( self,
                            target,               # target - The unique id of a player/object/nav point/whatever. The object must be visible to you
                            index = "last_goto"): #          when the command is recieved or your bot will do nothing. Note that something that was just
                                                  #          visible may not be when the command is recieved, therefore it is recomended you supply
                                                  #          a Location instead of a Target.
                                                  # index -  index under which the command is written to self.last_goto_positions ... see __init__
        self.send_message("RUNTO", { "Target": target } )
        self.last_goto_positions[index] = ("Target", target)

    def send_run_to_xyz( self, x, y, z, index = "last_goto" ): # Location you want to go to.
        string = location_tuple_to_string( (x + self.runto_correction[0], y + self.runto_correction[1], z + self.runto_correction[2]) ) 
        self.send_message("RUNTO", { "Location": string } )
        self.last_goto_positions[index] = ("Location", string)        

    def send_run_to_location_string( self, location_string, index = "last_goto" ): # X / Y / Z ... should be "40 50 45" or "40,50,45"
        location_tuple = location_string_to_tuple(location_string)
        string = location_tuple_to_string( location_tuple[0] + self.runto_correction[0], location_tuple[1] + self.runto_correction[1], location_tuple[2] + self.runto_correction[2] )
        self.send_message("RUNTO", { "Location": string } )
        self.last_goto_positions[index] = ("Location", string)

    def send_run_to_location_tuple( self, location_tuple, index = "last_goto" ):
        string = location_tuple_to_string( (location_tuple[0] + self.runto_correction[0], location_tuple[1] + self.runto_correction[1], location_tuple[2] + self.runto_correction[2]) )
        self.send_message("RUNTO", { "Location": string } )
        self.last_goto_positions[index] = ("Location", string)

    def send_run_to_dict(self, dict, index = "last_goto"):
        if dict.has_key("Location"):
            location_tuple = location_string_to_tuple(dict["Location"])
            dict["Location"] = location_tuple_to_string( (location_tuple[0] + self.runto_correction[0], location_tuple[1] + self.runto_correction[1], location_tuple[2] + self.runto_correction[2]) )
            self.last_goto_positions[index] = ("Location", dict["Location"])
        elif dict.has_key("Target"):
            self.last_goto_positions[index] = ("Target", dict["Target"])
            
        self.send_message("RUNTO", dict)            

  # STRAFE ... # like RUNTO, but you move towards a destination while facing another point/object

    # as there are many combination to write, I'll write only 2

    def send_strafe_target( self, location_tuple_to_strafe_to,
                                  face_target,          # face_target - the unique id of a player/object/nav point/whatever that you want to face while moving,
                                  index = "last_goto"): # must be visible to you currently
                                                        # index -  index under which the command is written to self.last_goto_positions ... see __init__
        string = location_tuple_to_string( (location_tuple_to_strafe_to[0] + self.runto_correction[0], location_tuple_to_strafe_to[1] + self.runto_correction[1], location_tuple_to_strafe_to[2] + self.runto_correction[2]) )
        self.send_message("STRAFE", { "Location": string, "Target": face_target } )
        self.last_goto_positions[index] = ("Location", string)

    def send_strafe(self, location_tuple_to_strafe_to, location_tuple_to_face_to, index = "last_goto"):
        string = location_tuple_to_string( (location_tuple_to_strafe_to[0] + self.runto_correction[0], location_tuple_to_strafe_to[1] + self.runto_correction[1], location_tuple_to_strafe_to[2] + self.runto_correction[2]) )
        self.send_message("STRAFE", { "Location": string,
                                      "Focus": location_tuple_to_string( location_tuple_to_face_to ) } )
        self.last_goto_positions[index] = ("Location", string)

    def send_strafe_dict(self, dict, index = "last_goto"):
        if dict.has_key("Location"):
            location_tuple = location_string_to_tuple(dict["Location"])
            dict["Location"] = location_tuple_to_string( (location_tuple[0] + self.runto_correction[0], location_tuple[1] + self.runto_correction[1], location_tuple[2] + self.runto_correction[2]) )
            self.last_goto_positions[index] = ("Location", dict["Location"])
        
        self.send_message("STRAFE", dict)

  # TURNTO
                                          
    def send_rotate(self, pitch, yaw, roll): # should be in absolute terms and in UT units (2pi = 65535 units)
        self.send_message("TURNTO", { "Rotation": rotation_tuple_to_string( (pitch, yaw, roll) ) } )

    def send_rotate_tuple(self, rotation_tuple):
        self.send_message("TURNTO", { "Rotation": rotation_tuple } )

    def send_turn_to_target(self, target): # the unique id of a player/object/nav point/whatever that you want to face. Must be visible.
        self.send_message("TURNTO", { "Target": target } )

    def send_turn_to_xyz(self, x, y, z): # Location you want to face. Normal rules for location. Only used if no Target or Rotation.                                       
        self.send_message("TURNTO", { "Location": location_tuple_to_string( (x, y, z) ) } )
        
    def send_turn_to_location_string(self, location_string): # X / Y / Z ... should be "40 50 45" or "40,50,45"
        self.send_message("TURNTO", { "Location": location_string } )

    def send_turn_to_location_tuple(self, location_tuple):
        self.send_message("TURNTO", { "Location": location_tuple_to_string( location_tuple ) } )

    def send_turn_to_dict(self, dict):
        self.send_message("TURNTO", dict)    

  # ROTATE

    def send_rotate_horizontal(self, amount): # amount in UT units to rotate, may be negative to rotate counter clockwise.
        self.send_message("ROTATE", { "Amount": amount, "Axis": "Horizontal" } )

    def send_rotate_vertical(self, amount): # amount in UT units to rotate, may be negative to rotate counter clockwise.
        self.send_message("ROTATE", { "Amount": amount, "Axis": "Vertical" } )

    def send_rotate_dict(self, dict):
        self.send_message("ROTATE", dict)        

  # SHOOT

    def send_shoot(self, location_tuple, # Location you want to shoot at.
                         target): # the unique id of your target, if you specify a target, and it is visible to you, the server will provide aim
                                  # correction and target leading for you - if not you just shoot at the location specified,
                                  # Note you still must provide location.
        self.send_message("SHOOT", { "Location": location_tuple_to_string( location_tuple ), "Target": target } )

    def send_shoot_alt(self, location_tuple, target): # ou will alt fire instead of normal fire
        self.send_message("SHOOT", { "Location": location_tuple_to_string( location_tuple ), "Target": target, "Alt": "True" } )

    def send_shoot_dict(self, dict):
        self.send_message("SHOOT", dict)    

  # CHANGEWEAPON
                         
    def send_change_weapon(self, weapon_id): # Unique Id of weapon to switch to. Obtain Unique Id's from AIN events.
        self.send_message("CHANGEWEAPON", { "Id": weapon_id } )

    def send_change_weapon_best(self): # change to the best weapon you have
        self.send_message("CHANGEWEAPON", { "Id": "Best" } )

    def send_change_weapon_dict(self, dict):
        self.send_message("CHANGEWEAPON", dict)    

  # STOPSHOOT

    def send_stop_shoot(self): # stop firing your weapon
        self.send_message("STOPSHOOT", {} )

  # CHECKREACH
  
    def send_reachcheck_target(self, msg_id,      # message id made up by you and echoed in respose so you can match up response with querry 
                                     target,      # the unique id of a player/object/nav point/whatever, must be visible 
                                     from_tuple): # exact location of bot at time of check
        self.send_message("CHECKREACH", { "Id": msg_id, "Target": target, "From": location_tuple_to_string( from_tuple ) } )

    def send_reachcheck_location(self, msg_id, to_tuple, from_tuple):
        self.send_message("CHECKREACH", { "Id": msg_id, "Location": location_tuple_to_string( to_tuple ),
                                                        "From": location_tuple_to_string( from_tuple ) } )

    def send_reachcheck_dict(self, dict):
        self.send_message("CHECKREACH", dict)        

  # GETPATH

    def send_getpath_xyz(self, msg_id, # message id made up by you and echoed in respose so you can match up response with querry 
                               x, y, z):
        self.send_message("GETPATH", { "Id": msg_id, "Location": location_tuple_to_string( ( x, y, z ) ) } )

    def send_get_path_location_string(self, msg_id, location_string): # X / Y / Z ... should be "40 50 45" or "40,50,45"
        self.send_message("GETPATH", { "Id": msg_id, "Location": location_string } )

    def send_get_path_location_tuple(self, msg_id, location_tuple):
        self.send_message("GETPATH", { "Id": msg_id, "Location": location_tuple_to_string(location_tuple) } )

    def send_get_path_dict(self, dict):
        self.send_message("GETPATH", dict)   

  # MESSAGE

    def send_global_message(self, message_string): # send a message to the world
        self.send_message("MESSAGE", { "Text": message_string, "Global": "True" } )

    def send_private_message(self, message_string): # send a message to your team
        self.send_message("MESSAGE", { "Text": message_string } )

    def send_message_dict(self, dict):
        self.send_message("MESSAGE", dict)       

  # PING

    def send_ping(self): # if for some reason 10 updates a second or whatever your default is isn't frequent enough connection detection for your
                         # tastes, use PING, Server will return "PONG"
        self.send_message("PING", {})
    
                            

        
            
        

        
                                 


        
    

    

    

    

    

    