package bot;

import cz.cuni.amis.pogamut.base.agent.navigation.PathNotConstructable;
import cz.cuni.amis.pogamut.base3d.worldview.objects.ILocated;

import com.google.inject.Inject;

import cz.cuni.amis.pogamut.base.agent.navigation.PathPlannerListener;
import cz.cuni.amis.pogamut.base.agent.worldview.WorldEventListener;
import cz.cuni.amis.pogamut.base.communication.commands.ICommandSerializer;
import cz.cuni.amis.pogamut.base.exceptions.PogamutException;
import cz.cuni.amis.pogamut.base.factory.guice.AgentScoped;
import cz.cuni.amis.pogamut.base.utils.logging.AgentLogger;
import cz.cuni.amis.pogamut.base3d.worldview.objects.Location;

import cz.cuni.amis.pogamut.base3d.worldview.objects.Rotation;
import cz.cuni.amis.pogamut.base3d.worldview.objects.Velocity;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.AgentInfo;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.Game;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.Players;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.UTAstar;
import cz.cuni.amis.pogamut.ut2004.agent.worldview.UT2004SyncLockableWorldView;
import cz.cuni.amis.pogamut.ut2004.bot.SyncUT2004Bot;
import cz.cuni.amis.pogamut.ut2004.communication.messages.UnrealId;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbcommands.*;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.*;

import cz.cuni.amis.pogamut.ut2004.communication.translator.events.MapPointListObtained;
import info.ScenarioType;
import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Random;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.vecmath.Vector3d;
import utils.Algeb;



/**
 * Emotional scenario polymorph.
 *
 * @author Michal "Knight" Bida 
 */
@AgentScoped
public class PolymorphBot extends SyncUT2004Bot<UT2004SyncLockableWorldView>{

    /** agent pathing helper variable */
    public ArrayList<ILocated> myPath = new ArrayList<ILocated>();
    /** agent pathing helper variable */
    public boolean pathRequested = false;
    /** agent pathing helper variable */
    public boolean pathReceived = false;

    /** All NavPoints on the map are stored here */
    public HashMap<UnrealId, NavPoint> knownNavPoints = new HashMap<UnrealId,NavPoint>();

    /** NavPoint i am heading to */
    public NavPoint myNavTarget;

    /** Current UnrealTournament time */
    public double currentTime = 0;

    /** Last time we've actually moved */
    public double lastMovementTime = 0;

    /** Last time we have bitten someone */
    public double lastBiteTime = 0;

    /** Last time we have sent text message */
    public double lastMessageTime = 0;

    /** If we are good or bad polymorph */
    public boolean bHostilePolymorph = false;

    /** Ff our polymorph is just staying at its place */
    public boolean bStationaryPolymorph = false;

    /** Our start location */
    public Location startLocation = null;
    /** Our start rotation */
    public Rotation startRotation = null;

    //some CONSTS here:
    /** How long we will pause logic when we are biting. */
    public double biteDurationConst = 3;
    /** Minimal delay between two bites */
    public double biteDelayConst = 10;
    /** Minimal delay between two messages sent */
    public double messageDelayConst = 10;
    /** Message text bubble fade out time */
    public double messageFadeOutConst = 5;

    //OBSTACLE AVOIDANCE:
    private static final double AVOID_OBSTACLE_FUNCTION = 150;
	private static final double AVOID_OBSTACLE_FUNCTION_MULTI = 0.4;
	private static final double AVOID_OBSTACLE_MIN_DISTANCE = 400;
	private static final double AVOID_OBSTACLE_RUN_VECTOR_LENGTH = 100;
	private static final double AVOID_OBSTACLE_RUN_LENGTH = 100;
	public static double AVOID_OBSTACLE_SECOND_RUN_MULTI = 0.5;

    /** bot memory module - general info about the game - map, timelimit, etc */
    public Game game;
    /** bot memory module - general info about this agent - location, rotation, etc. */
    public AgentInfo agentInfo;
    /** bot memory module - general info about other agents we can see - location, rotation, etc. */
    public Players players;
    /** Path finding module */
    public UTAstar myPathPlanner;

    /** Here we will listen to map event and store all nav points in our internals */
    public WorldEventListener myMapListObtainedListener = new WorldEventListener() {

        @Override
        public void notify(Object event) {

            MapPointListObtained map;

            map = (MapPointListObtained) event;
            knownNavPoints.putAll(map.getNavPoints());

        }
    };

    /** Listens to BeginMessage and stores current UnrealTournament time */
    public WorldEventListener myBegListener = new WorldEventListener() {

        @Override
        public void notify(Object event) {
            if (event instanceof BeginMessage){
                BeginMessage bm = (BeginMessage) event;
                currentTime = bm.getTime();
            }
        }
    };

    /** Path listener - initialized in goToLocation method */
    public PathPlannerListener myPathListener = new PathPlannerListener(){

        @Override
        public void pathEvent(List path) {
            myPath = (ArrayList<ILocated>) path;
            pathReceived = true;
        }
    };


    /**
     * Constructor.
     * 
     * @param logger
     * @param worldView
     * @param commandSerializer
     */
    @Inject
	public PolymorphBot(AgentLogger logger, UT2004SyncLockableWorldView worldView, ICommandSerializer commandSerializer) {
		super(logger, worldView, commandSerializer);
	}

    /**
     * Initialize all necessary variables here, before the bot actually receives anything
     * from the environment.
     */
	@Override
	protected void prePrepareBot() {

        //sets the logger level, if we would let the logger level to be Level.All the bot would
        //be slowed significantly due to high number of messages
        getLogger().setLevel(Level.WARNING);

        //Register two listeners - one for navigation point, second for begine message (so we can update currentTime)
        this.getWorldView().addListener(MapPointListObtained.class, myMapListObtainedListener);
        this.getWorldView().addListener(BeginMessage.class, myBegListener);

        //initialize memory modules
        this.game = new Game(this.getWorldView(),this.getLogger().user());
        this.agentInfo = new AgentInfo(this.getWorldView(), game, this.getLogger().user());
        this.players = new Players(this.getWorldView(), agentInfo, this.getLogger().user());

        //initialize path planner
        this.myPathPlanner = new UTAstar(this.getAct(),this.getWorldView());
	}

    /**
     * Here we have already received information about game in GameInfo
     *
     * @param info
     */
	@Override
	protected void postPrepareBot(GameInfo info) {

	}

    /**
     * Here we can modify initializing command for our bot.
     *
     * @return
     */
	@Override
	protected Initialize createInitializeCommand() {
        Initialize myInit = new Initialize();

        //just set the name of the bot, nothing else
        myInit.setName("Emohawk");
        myInit.setClassName("GameBots2004.GBNaliCow");
        if (startLocation != null)
            myInit.setLocation(startLocation);
        if (startRotation != null)
            myInit.setRotation(startRotation);
        //TODO: SET class to nalicow
        //myInit.setSkin(Skin);

		return myInit;
	}

    /**
     * The bot is initilized in the environment - a physical representation of the
     * bot is present in the game.
     *
     * @param config information about configuration
     * @param init information about configuration
     */
	@Override
	protected void botInitialized(ConfigChange config, InitedMessage init) {
        //getAct().act(new SendMessage().setText("I am alive!"));
	}

    /**
     * Main method that controls the bot - makes decisions what to do next.
     * It is called iteratively by Pogamut engine every time a synchronous batch
     * from the environement is received. This is usually 4 times per second - it
     * is affected by visionTime variable, that can be adjusted in GameBots ini file in
     * UT2004/System folder.
     *
     * @throws cz.cuni.amis.pogamut.base.exceptions.PogamutException
     */
	@Override
	protected void doLogic() throws PogamutException {

        //if we are biting pause a logic for the time
        if (currentTime - lastBiteTime < biteDurationConst){
            return;
        }

        //update our lastMovementTime counter
        if (agentInfo.getVelocity().size() > 10){
            lastMovementTime = currentTime;
        }

        if ((currentTime - lastMessageTime) > messageDelayConst){
            String text = "";
            if (bHostilePolymorph)
                text = "RRRRRRRRRR !!";
            else
                text = "mrr mrr mrr";
            getAct().act(new SendMessage().setText(text).setFadeOut(messageFadeOutConst));
            lastMessageTime = currentTime;
        }

        Player nearestPlayer = players.getNearestPlayer(players.getVisiblePlayers().values());
        if ((nearestPlayer != null) && agentInfo.atLocation(nearestPlayer.getLocation(), 300)){
            if (!bHostilePolymorph){
                getAct().act(new TurnTo().setTarget(nearestPlayer.getId()));
                if (agentInfo.isMoving())
                    getAct().act(new Stop());
                return;
            } else {
                getAct().act(new Move().setFirstLocation(nearestPlayer.getLocation()));
                if (agentInfo.atLocation(nearestPlayer.getLocation(), 100)) {
                    if (agentInfo.isMoving())
                        getAct().act(new Stop());

                    if (currentTime - lastBiteTime > biteDelayConst){
                        getAct().act(new SendMessage().setText("To:" + nearestPlayer.getName() + "Action BITE").setFadeOut(messageFadeOutConst));
                        lastBiteTime = currentTime;
                    }
                }
                return;
            }
        }

        if (!bStationaryPolymorph) {//we can move
            //set new random navigation point to go to
            if (myNavTarget == null){
                myPath.clear();
                pathRequested = false;
                pathReceived = false;
                myNavTarget = pickNewRandomNavTarget(); //random movement
            }

            if (myNavTarget != null){
                //follow the path to NavPoint and set to null if we reach it
                if ((myNavTarget.getLocation().getDistance(agentInfo.getLocation()) < 80)){
                    myNavTarget = null;
                } else {
                    goToLocation(myNavTarget.getLocation());
                }
            }

            //anti stuck policy here - if we are not moving for 3 seconds..
            if ((currentTime - lastMovementTime) > 3){
                //jump bot and pick different Nav Point.
                getAct().act(new Jump());
                myNavTarget = null;
            }
        }
        //this.getLogger().user().warning("VelocitySize: "+agentInfo.getVelocity().size() + " Cur.time: " + currentTime + " LastmoveTime: " + lastMovementTime);
    }

    /**
     * Called each time our bot die. Good for reseting all bot state dependent variables.
     *
     * @param event
     */
	@Override
	protected void botKilled(BotKilled event) {
        myPath.clear();
        pathReceived = false;
        pathRequested = false;
        myNavTarget = null;
	}

    /**
     * Will set up our polymorph internal variables so he can be later set up correctly
     * in the environment by INIT command.
     *
     * @param type
     */
    public void initPolymoprh(ScenarioType type) {
        if (type == ScenarioType.SCENARIO_ONE){
            startLocation = new Location(3839,614,-4556);
            startRotation = new Rotation(0,16000,0);
            bHostilePolymorph = false;
            bStationaryPolymorph = true;

        }
    }

    /**
     * Does nothign for the polymorph yet.
     */
    public void shutdown() {
        //this.stop();
    }

    /**
     * Used by obstacle avoidance code.
     *
     * @param x
     * @param distance
     * @return
     */
    private double forceFunction(double x, double distance) {
		double multi = distance / AVOID_OBSTACLE_MIN_DISTANCE;
		if (x < -AVOID_OBSTACLE_FUNCTION || x > AVOID_OBSTACLE_FUNCTION) return 0;
		if (x < 0) return (-AVOID_OBSTACLE_FUNCTION - x) * multi;
		return (AVOID_OBSTACLE_FUNCTION - x) * multi;
	}

    /**
	 * This will compute new Location to avoid hitting other players.
	 * @param moveLocation
	 * @return
	 */
	private Location[] adjustLocations(Location[] moveLocation) {
		Player player = players.getNearestPlayer(players.getVisiblePlayers().values());
		if (player == null) return moveLocation;
		if (player.getLocation().getPoint3d().distance(agentInfo.getLocation().getPoint3d()) > AVOID_OBSTACLE_MIN_DISTANCE) {
			return moveLocation;
		}
		if (moveLocation[0] == null) return moveLocation;
		Location[] result = new Location[2];

		Location location = agentInfo.getLocation();
		Velocity velocity = new Velocity(moveLocation[0].x - location.x, moveLocation[0].y - location.y, moveLocation[0].z - location.z);
		Vector3d runningVector = velocity.getVector3d();
		runningVector.normalize();
		Point2D force = Algeb.rotate(Algeb.projection(runningVector), Algeb.rad(90));

		Point2D[] forces = new Point2D[2];
		forces[0] = Algeb.multi(Algeb.projection(runningVector), AVOID_OBSTACLE_RUN_VECTOR_LENGTH);
		double distance = Algeb.distanceFromRunningVector(location, velocity, player.getLocation());
		double multi = forceFunction(distance, location.getPoint3d().distance(player.getLocation().getPoint3d()));
		forces[1] = Algeb.multi(force, multi * AVOID_OBSTACLE_FUNCTION_MULTI);

		Point2D sum = Algeb.vectorSum(forces);

		result[0] = new Location(location.getX() + sum.getX(), location.getY() + sum.getY(), location.getZ());
		forces[0] = Algeb.multi(Algeb.projection(runningVector), AVOID_OBSTACLE_RUN_VECTOR_LENGTH*2);
		sum = Algeb.vectorSum(forces);
		result[1] = new Location(location.getX() + sum.getX(), location.getY() + sum.getY(), location.getZ());

		return result;
	}

    /**
     * Goes to target location. Keeps returning true while we are on our way there.
     * If we are at location or some problem encountered, then returns false.
     *
     * @param targetLocation
     * @return
     */
    public boolean goToLocation(Location targetLocation) {

        if (agentInfo.atLocation(targetLocation, 100)){
            //restarting
            myPath.clear();
            pathReceived = false;
            pathRequested = false;
            return false;
        }

        if (myPath.isEmpty() && !pathRequested){
            try {
                myPathPlanner.addPathListener(myPathListener);
                myPathPlanner.computePath(agentInfo.getLocation(), targetLocation);
                pathRequested = true;
                return true;
            } catch (PathNotConstructable ex) {
                Logger.getLogger(EmotionalBot.class.getName()).log(Level.SEVERE, null, ex);
                return false;
            }
        }

        //waiting for the path
        if (pathRequested && !pathReceived)
            return true;

        //if we are so close we get 0 path, we will add our target location to the path
        if (myPath.isEmpty() && pathReceived){
            myPath.add(targetLocation);
        }

        //throw out points we've been to
        if (myPath.get(0).getLocation().getDistance(agentInfo.getLocation()) < 40 ){
            myPath.remove(0);
        }

        Iterator<ILocated> it = myPath.iterator();
        if (it.hasNext()){
            Location l1 = it.next().getLocation();
            Location[] moveLocation;
            if(it.hasNext()) {
                // there are at least two points in the path
                Location l2 = it.next().getLocation();
                moveLocation = adjustLocations(new Location[]{l1 , l2});
                getAct().act(new Move().setFirstLocation(moveLocation[0]).setSecondLocation(moveLocation[1]));
                return true;
            } else {
                // there is only one point to go to
                moveLocation = adjustLocations(new Location[]{l1});
                getAct().act(new Move().setFirstLocation(moveLocation[0]));
                return true;
            }
        }

        return false;
    }

    /**
     * Returns nearest navigation point to input location in the map.
     *
     * @param targetLocation
     * @return
     */
    protected Location getNearestNavLocation(Location targetLocation) {

        NavPoint result = null;

        for (NavPoint nav : knownNavPoints.values()){
            if (result == null){
                result = nav;
            } else {
                if (nav.getLocation().getDistance(targetLocation) < result.getLocation().getDistance(targetLocation)){
                    result = nav;
                }
            }
        }
        return result.getLocation();
    }

    /**
     * Rendomly picks some navigation point to head to.
     *
     * @return
     */
    private NavPoint pickNewRandomNavTarget() {
        Random rand = new Random();
        int i,counter;

        counter = rand.nextInt(knownNavPoints.values().size());

        i = 0;
        for (NavPoint nav : knownNavPoints.values()){
            if (i == counter){
                return nav;
            }
            i++;
        }

        return null;
    }
}