package almabasedmodel;

/*
 * AffectEngine.java
 *
 * Copyright (c) 2005, 2006, 2007, 2008, Patrick Gebhard, DFKI GmbH
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *   - Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *
 *   - Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution.
 *
 *   - Neither the name of the DFKI GmbH nor the names of its contributors
 *     may be used to endorse or promote products derived from this software
 *     without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

import bot.EmotionalBot;
import cz.cuni.amis.pogamut.base3d.worldview.objects.ILocated;
import cz.cuni.amis.pogamut.base3d.worldview.objects.Location;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Player;
import de.affect.emotion.Emotion;
import de.affect.emotion.EmotionType;
import de.affect.manage.AffectManager;
import de.affect.manage.event.AffectInputEvent;
import de.affect.manage.event.AffectInputListener;
import de.affect.manage.event.AffectUpdateEvent;
import de.affect.manage.event.AffectUpdateListener;
import de.affect.manage.event.EmotionChangeEvent;
import de.affect.manage.event.EmotionChangeListener;
import de.affect.manage.event.EmotionMaintenanceEvent;
import de.affect.manage.event.EmotionMaintenanceListener;
import de.affect.mood.Mood;
import de.affect.util.AppraisalTag;
import de.affect.xml.AffectOutputDocument;
import de.affect.xml.AffectOutputDocument.AffectOutput.CharacterAffect;

import de.affect.xml.CharacterType;
import de.affect.xml.ActionTypes;
import de.affect.xml.EventTypes;
import de.affect.xml.ObjectTypes;
import de.affect.xml.Intensity;
import de.affect.xml.AffectInputDocument;
import de.affect.xml.AffectInputDocument.AffectInput;
import de.affect.xml.AffectInputDocument.AffectInput.Character;
import de.affect.xml.AffectInputDocument.AffectInput.Act;
import de.affect.xml.AffectInputDocument.AffectInput.Action;
import de.affect.xml.AffectInputDocument.AffectInput.Event;
import de.affect.xml.AffectInputDocument.AffectInput.Object;
import de.affect.xml.AffectInputDocument.AffectInput.BasicEEC;

import info.EventId;
import java.io.IOException;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.xmlbeans.XmlException;

/**
 * Provides the first interface between Pogamut and ALMA (additional methods are defined in
 * AEmotionState.java class).
 *
 * This code is based on the AffectEngine.java class provided with ALMA as an example
 * implementation.
 *
 * @author Patrick Gebhard, Knight
 */
public class PogamutALMA implements EmotionMaintenanceListener, AffectInputListener, AffectUpdateListener,EmotionChangeListener {


  /** The agent this PogamutALMA instance is for. */
  public EmotionalBot myAgent;
  
  /** The ALMA Java implementation */
  public static AffectManager fAM = null;

  /** ALMA affect computation definition file */
  private static String sALMACOMP ="./conf/AffectComputation.aml";
  //private static String sALMACOMP = "../conf/AffectComputation.aml";
  //private static String sALMADEF = "../conf/CharacterDefinition.aml";
  /** ALMA character definition file */
  private static String sALMADEF = "./conf/CharacterDefinition.aml";

  /** Current level time in seconds (contains also milliseconds after dot) */
  public double currentTime;

  /* ALMA mode:
       false - output on console
       true - graphical user interface CharacterBuilder
              NOTE: No runtime windows (defined in AffectComputation or
                    AffectDefinition will be displayed!) */
  private static final boolean sGUIMode = false;

  /** Console logging */
  public static Logger log = Logger.getLogger(Logger.GLOBAL_LOGGER_NAME);


  /**
   * The constructor that sets up ALMA engine.
   *
   * @param agent
   */
  public PogamutALMA(EmotionalBot agent) {
    // Starting the ALMA affect engine
    myAgent = agent;

    try {

      fAM = new AffectManager(sALMACOMP, sALMADEF, sGUIMode);
      //disable alma logging
      fAM.log.setLevel(Level.OFF);
      fAM.addAffectUpdateListener(this);
      fAM.addAffectInputListener(this);

      //fAM.getCharacterByName("Bruno").
      //fAM.
    } catch(IOException io) {
      log.info("Error during ALMA initialisation");
      io.printStackTrace();
      System.exit(-1);
    } catch(XmlException xmle) {
      log.info("Error in ALMA configuration");
      xmle.printStackTrace();
      System.exit(-1);
    }
  }

  /**
   * Listens to affect updates computed by ALMA. This
   * implements the AffectUpdateListener
   * @param event
   */
  public synchronized void update(AffectUpdateEvent event) {
    AffectOutputDocument aod = event.getUpdate();

   // myAgent.getLogger().user().warning("AFFECT UPDATE: " +event.getUpdate().toString());
    try {
      for(Iterator<CharacterAffect> it = aod.getAffectOutput().getCharacterAffectList().iterator(); it.hasNext();) {
	CharacterAffect character = it.next();

	// access cached data or create new cache
	String name = character.getName();


	String emotion = character.getDominantEmotion().getName().toString();
	double eIntensity = Double.parseDouble(character.getDominantEmotion().getValue());
	String mood = character.getMood().getMoodword().toString();
	String mIntensity = character.getMood().getIntensity().toString();
	String mTendency = character.getMoodTendency().getMoodword().toString();

	//TODO use affect for something!
    //event.getUpdate().getAffectOutput().getCharacterAffectArray(1).

	//myAgent.getLogger().user().info(name + " has dominant emotion " + emotion + "(" + eIntensity + ")");
    //getCurrentEmotions();
      }
    } catch (Exception e)  {
      e.printStackTrace();
    }
  }

  /**
   * Returns current mode for target agent.
   *
   * @param agentName
   * @return
   */
  public Mood getCurrentMood(String agentName)
  {
    return fAM.getCharacterByName(agentName).getCurrentMood();
  }

  /**
   * Gets all emotions for target agent, that are felt toward the elicitor provided.
   *
   * @param agentName
   * @param elicitor
   * @return
   */
  public List<Emotion> getAllEmotionsForElicitor(String agentName, String elicitor) {

    ArrayList<Emotion> emotionList = new ArrayList<Emotion>();
    Emotion tempEm = null;

    for (int j = 0; j < EmotionType.values().length; j++) {

        //we get emotions one by one from ALMA history
        tempEm = fAM.getCharacterByName(agentName).getEmotionHistory().getEmotionByElicitor(EmotionType.values()[j], elicitor);
        if (tempEm != null)
            emotionList.add(tempEm);
    }

    return emotionList;
  }

  /**
   * Gets emotion for target agent name of input type and for input elicitor.
   *
   * @param agentName
   * @param elicitor
   * @param type
   * @return
   */
  public Emotion getEmotionForElicitor(String agentName, String elicitor, EmotionType type) {
    return fAM.getCharacterByName(agentName).getEmotionHistory().getEmotionByElicitor(type, elicitor);
  }

  /**
   * Gets the ALMA current emotions (in characters focus) for target agent.
   *
   * @param agentName
   * @return
   */
  public List<Emotion> getCurrentEmotions(String agentName)
  {
      if (fAM.getCharacterByName(agentName) != null)
        return fAM.getCharacterByName(agentName).getCurrentEmotions().getEmotions();
      else
        return new ArrayList<Emotion>();

    /*  myAgent.getLogger().user().info("----------------------------------------");
      myAgent.getLogger().user().info(fAM.getCharacterByName("Bruno").getCurrentEmotions().toString());

      myAgent.getLogger().user().info(fAM.getCharacterByName("Bruno").getCurrentMood().toString());
    */
  }

  /**
   * Creates an AffectInput document containing an BasicEEC Element and returns a AffectInput object
   *
   * @param actor 
   * @param desirability 
   * @param agency
   * @param praiseworthiness
   * @param appealingness
   * @param liking
   * @param likelihood
   * @param elicitor
   * @param realization
   * @return
   */
  public AffectInput createAffectInputBasicEEC(String actor,
      double desirability, double praiseworthiness, double appealingness, double likelihood,
      double liking, double realization, String elicitor, String agency) {

    AffectInput aiInput = AffectInput.Factory.newInstance();
    // Building the Character element
    Character perfCharacter = Character.Factory.newInstance();
    perfCharacter.setName(actor);

    BasicEEC eec = BasicEEC.Factory.newInstance();

    eec.setDesirability(desirability);
    eec.setPraiseworthiness(praiseworthiness);
    eec.setAppealingness(appealingness);
    eec.setLikelihood(likelihood);
    eec.setLiking(liking);
    eec.setRealization(realization);
    eec.setAgency((agency.toLowerCase() == "self") ? BasicEEC.Agency.SELF : BasicEEC.Agency.OTHER);
    eec.setElicitor(elicitor);



    aiInput.setCharacter(perfCharacter);
    aiInput.setBasicEEC(eec);

    return aiInput;
  }

   /**
    * Creates an AffectInput document containing an BasicEEC Element and returns a AffectInput object
    * This type eec element is constructed elsewhere
    * @param actor 
    * @param eec 
    * @return
    */
  public AffectInput createAffectInput(String actor, BasicEEC eec) {

    AffectInput aiInput = AffectInput.Factory.newInstance();
    // Building the Character element
    Character perfCharacter = Character.Factory.newInstance();
    perfCharacter.setName(actor);

    aiInput.setCharacter(perfCharacter);
    aiInput.setBasicEEC(eec);

    return aiInput;
  }

  /**
   * The <code>processAffectInput</code>  passes instance of AffectInput to AffectManager
   * event string is here just for logging
   * @param ai
   * @param event
   */
  public void processAffectInput(AffectInput ai, EventId event) {
    //debug
    //log.info(ai.toString());

    if (myAgent.recentReceivedProposal != null)
        myAgent.getLogger().user().warning("Name: " + myAgent.getMyName() + " RProp.: " + myAgent.recentReceivedProposal.getType() + " RProp.Targ.: " + myAgent.recentReceivedProposal.getTarget());// + " AW: "+ myAgent.agentWith);
    String msg = myAgent.getMyName() + " State: " + myAgent.getState() + ", ";
    if (myAgent.agentWith != null)
        msg += "AGENTW: " + myAgent.agentWith.getName() + " ";
    if (myAgent.myGoalTarget != null)
        msg += "GOAL: " + myAgent.myGoalTarget + " ";

    msg += "PATH:" + ",req:" +  myAgent.pathRequested + ",rec:" + myAgent.pathReceived;
    Iterator<ILocated> iterator = myAgent.myPath.iterator();
    while (iterator.hasNext()) {
        msg += iterator.next().getLocation().toString() + ",";
    }

    if (!myAgent.agentInterrupters.isEmpty()){
        msg += " INTERRUPTERS: ";
        for (Player plr : myAgent.agentInterrupters)
            msg += plr.getName() + ", ";        
    }
    myAgent.getLogger().user().warning(msg);

/*
    myAgent.getLogger().user().warning(" Event: "+ event  + " RProposal: " + myAgent.recentReceivedProposal + " \n" + myAgent.currentTime + " Name: " + ai.getCharacter().getName() +
            " Vars: D: " + ai.getBasicEEC().getDesirability() + "; P: " + ai.getBasicEEC().getPraiseworthiness() +
            "; A: " + ai.getBasicEEC().getAppealingness() + "; Lh: " + ai.getBasicEEC().getLikelihood() +
            "; Li: " + ai.getBasicEEC().getLiking() + "; R: " + ai.getBasicEEC().getRealization() +
            "; Ag: " + ai.getBasicEEC().getAgency() + "; El.: " + ai.getBasicEEC().getElicitor()
            );
*/
    if (myAgent.generatedEvents.containsKey(event) && myAgent.generatedEvents.get(event)) {

        myAgent.getLogger().user().warning("Name: " + ai.getCharacter().getName() + " Event: "+ event + " \n" + myAgent.currentTime +
            " ElicName: " + myAgent.getName(ai.getBasicEEC().getElicitor()) + " Vars: D: " + ai.getBasicEEC().getDesirability() + "; P: " + ai.getBasicEEC().getPraiseworthiness() +
            "; A: " + ai.getBasicEEC().getAppealingness() + "; Lh: " + ai.getBasicEEC().getLikelihood() +
            "; Li: " + ai.getBasicEEC().getLiking() + "; R: " + ai.getBasicEEC().getRealization() +
            "; Ag: " + ai.getBasicEEC().getAgency() + "; El.: " + ai.getBasicEEC().getElicitor()
            ); 
        
        myAgent.logEmotionEvent(ai, event);
        fAM.processSignal(ai);
    } else
        myAgent.getLogger().user().warning("NOT_GENERATED! Name: " + ai.getCharacter().getName() + " Event: "+ event + " \n" + myAgent.currentTime +
            " ElicName: " + myAgent.getName(ai.getBasicEEC().getElicitor()) + " Vars: D: " + ai.getBasicEEC().getDesirability() + "; P: " + ai.getBasicEEC().getPraiseworthiness() +
            "; A: " + ai.getBasicEEC().getAppealingness() + "; Lh: " + ai.getBasicEEC().getLikelihood() +
            "; Li: " + ai.getBasicEEC().getLiking() + "; R: " + ai.getBasicEEC().getRealization() +
            "; Ag: " + ai.getBasicEEC().getAgency() + "; El.: " + ai.getBasicEEC().getElicitor()
            );

  }

  /**
   *  Not used right now.
   *
   * @param event
   */
  @Override
    public void maintainEmotion(EmotionMaintenanceEvent event) {
        //myAgent.myEmotionState.notifyEmotionChange(arg0.);

        //myAgent.getLogger().user().warning("EM MAINTENANCE: " +event.toString());
    }

    /**
     * Not used right now.
     *
     * @param event
     */
    @Override
    public void emotionChanged(EmotionChangeEvent event) {
        //myAgent.getLogger().user().warning("EM CHANGED: " +event.emotions().toString());
    }

    /**
     * Not used right now.
     * 
     * @param event
     */
    @Override
    public void updateInput(AffectInputEvent event) {
        // myAgent.getLogger().user().warning("EM INPUT: " + event.getAffectInput().toString());
    }

}



