/*
 * This license does NOT supersede the original license of GPC.  Please see:
 * http://www.cs.man.ac.uk/~toby/alan/software/#Licensing
 *
 * The SEI Software Open Source License, Version 1.0
 *
 * Copyright (c) 2004, Solution Engineering, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Solution Engineering, Inc. (http://www.seisw.com/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 3. The name "Solution Engineering" must not be used to endorse or
 *    promote products derived from this software without prior
 *    written permission. For written permission, please contact
 *    admin@seisw.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL SOLUTION ENGINEERING, INC. OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 */

package com.seisw.util.geom;

import java.util.ArrayList;
import java.util.List;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;

/**
 * <code>PolySimple</code> is a simple polygon - contains only one inner
 * polygon.
 * <p>
 * <strong>WARNING:</strong> This type of <code>Poly</code> cannot be used
 * for an inner polygon that is a hole.
 * 
 * @author Dan Bridenbecker, Solution Engineering, Inc.
 */
public class PolySimple implements Poly, Cloneable {
	// -----------------
	// --- Constants ---
	// -----------------

	private static final boolean collide(Point2D e1p1, Point2D e1p2,
			Point2D e2p1, Point2D e2p2) {
		double a;
		a = (e1p2.getX() - e1p1.getX()) / (e1p2.getY() - e1p1.getY())
				* (e2p1.getY() - e1p1.getY()) + e1p1.getX();
		if (Math.abs(a - e2p1.getX()) < 0.001) {
			return true;
		}
		a = (e2p2.getX() - e2p1.getX()) / (e2p2.getY() - e2p1.getY())
				* (e1p1.getY() - e2p1.getY()) + e2p1.getX();
		if (Math.abs(a - e1p1.getX()) < 0.001) {
			return true;
		}
		return false;
	}

	/** Flag used by the Clip algorithm */
	private boolean m_Contributes = true;

	// ------------------------
	// --- Member Variables ---
	// ------------------------
	/**
	 * The list of Point2D objects in the polygon.
	 */
	protected List<Point2D> m_List = new ArrayList<Point2D>();

	// ----------------------
	// --- Object Methods ---
	// ----------------------

	// --------------------
	// --- Constructors ---
	// --------------------
	/** Creates a new instance of PolySimple */
	public PolySimple() {
	}

	/**
	 * Add a point to the first inner polygon.
	 */
	public void add(double x, double y) {
		add(new Point2D.Double(x, y));
	}

	/**
	 * Add a point to the first inner polygon.
	 */
	public void add(Point2D p) {
		m_List.add(p);
	}

	/**
	 * Throws IllegalStateexception if called
	 */
	public void add(Poly p) {
		throw new IllegalStateException("Cannot add poly to a simple poly.");
	}

	// --------------------
	// --- Poly Methods ---
	// --------------------
	/**
	 * Remove all of the points. Creates an empty polygon.
	 */
	public void clear() {
		m_List.clear();
	}

	/**
	 * Returns a shallow copy of this object
	 */
	@SuppressWarnings("unchecked")
	@Override
	public Object clone() throws CloneNotSupportedException {
		PolySimple pd = (PolySimple) super.clone();
		if (this.m_List instanceof ArrayList) {
			pd.m_List = (List<Point2D>) ((ArrayList<Point2D>) this.m_List)
					.clone();
		} else {
			throw new CloneNotSupportedException();
		}
		return pd;
	}

	@Override
	public boolean contains(double x, double y) {
		return contains(new Point2D.Double(x, y));
	}

	@Override
	public boolean contains(Point2D point) {
		/*
		 * Simple ray-casting algorithm, casts one ray parallel to x-axis going
		 * through the specified point, stores crossing points and then
		 * computes, whether it has odd or even crosses left from the point
		 */
		ArrayList<Double> crosses = new ArrayList<Double>();
		int from = 0;
		int n = this.m_List.size();
		for (int i = 0; i < n; ++i) {
			Point2D last = this.m_List.get(i);
			Point2D current = this.m_List.get((i + 1) % n);
			if (last.getY() == point.getY()) {
				if (current.getY() < point.getY() && from == +1) {
					crosses.add(new Double(last.getX()));
				} else if (current.getY() > point.getY() && from == -1) {
					crosses.add(new Double(last.getX()));
				}
			} else if (last.getY() < point.getY()
					&& current.getY() > point.getY()
					|| last.getY() > point.getY()
					&& current.getY() < point.getY()) {
				// if crosses the line
				crosses.add(new Double(last.getX()
						+ (point.getY() - last.getY())
						* (current.getX() - last.getX())
						/ (current.getY() - last.getY())));
			}
			if (last.getY() < point.getY()) {
				from = -1;
			} else if (last.getY() > point.getY()) {
				from = +1;
			}
		}

		int left = 0;
		for (Double d : crosses) {
			if (d.doubleValue() == point.getX()) {
				// the point on borders is considered as inner
				return true;
			}
			if (d.doubleValue() < point.getX()) {
				left++;
			}
		}
		return left % 2 == 1;
	}

	@Override
	public Poly difference(Poly p) {
		//return Clip.difference(this, p, this.getClass());
		return Clip.difference(this, p);
	}

	/**
	 * Return true if the given object is equal to this one.
	 * <p>
	 * <strong>WARNING:</strong> This method failse if the first point appears
	 * more than once in the list.
	 */
	@Override
	public boolean equals(Object obj) {
		if (!(obj instanceof PolySimple)) {
			return false;
		}
		PolySimple that = (PolySimple) obj;

		int this_num = this.m_List.size();
		int that_num = that.m_List.size();
		if (this_num != that_num)
			return false;

		// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
		// !!! WARNING: This is not the greatest algorithm. It fails if !!!
		// !!! the first point in "this" poly appears more than once. !!!
		// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
		if (this_num > 0) {
			double this_x = this.getX(0);
			double this_y = this.getY(0);
			int that_first_index = -1;
			for (int that_index = 0; (that_first_index == -1)
					&& (that_index < that_num); that_index++) {
				double that_x = that.getX(that_index);
				double that_y = that.getY(that_index);
				if ((this_x == that_x) && (this_y == that_y)) {
					that_first_index = that_index;
				}
			}
			if (that_first_index == -1)
				return false;
			int that_index = that_first_index;
			for (int this_index = 0; this_index < this_num; this_index++) {
				this_x = this.getX(this_index);
				this_y = this.getY(this_index);
				double that_x = that.getX(that_index);
				double that_y = that.getY(that_index);

				if ((this_x != that_x) || (this_y != that_y))
					return false;

				that_index++;
				if (that_index >= that_num) {
					that_index = 0;
				}
			}
		}
		return true;
	}

	/**
	 * Returns the area of the polygon.
	 * <p>
	 * The algorithm for the area of a complex polygon was take from code by
	 * Joseph O'Rourke author of " Computational Geometry in C".
	 */
	public double getArea() {
		if (getNumPoints() < 3) {
			return 0.0;
		}
		double ax = getX(0);
		double ay = getY(0);
		double area = 0.0;
		for (int i = 1; i < (getNumPoints() - 1); i++) {
			double bx = getX(i);
			double by = getY(i);
			double cx = getX(i + 1);
			double cy = getY(i + 1);
			double tarea = ((cx - bx) * (ay - by)) - ((ax - bx) * (cy - by));
			area += tarea;
		}
		area = 0.5 * Math.abs(area);
		return area;
	}

	/**
	 * Returns the bounding rectangle of this polygon.
	 */
	public Rectangle2D getBounds() {
		double xmin = Double.MAX_VALUE;
		double ymin = Double.MAX_VALUE;
		double xmax = -Double.MAX_VALUE;
		double ymax = -Double.MAX_VALUE;

		for (int i = 0; i < m_List.size(); i++) {
			double x = getX(i);
			double y = getY(i);
			if (x < xmin)
				xmin = x;
			if (x > xmax)
				xmax = x;
			if (y < ymin)
				ymin = y;
			if (y > ymax)
				ymax = y;
		}

		return new Rectangle2D.Double(xmin, ymin, (xmax - xmin), (ymax - ymin));
	}

	/**
	 * Returns <code>this</code> if <code>polyIndex = 0</code>, else it
	 * throws IllegalStateException.
	 */
	public Poly getInnerPoly(int polyIndex) {
		if (polyIndex != 0) {
			throw new IllegalStateException("PolySimple only has one poly");
		}
		return this;
	}

	/**
	 * Always returns 1.
	 */
	public int getNumInnerPoly() {
		return 1;
	}

	/**
	 * Return the number points of the first inner polygon
	 */
	public int getNumPoints() {
		return m_List.size();
	}

	/**
	 * Return the X value of the point at the index in the first inner polygon
	 */
	public double getX(int index) {
		return m_List.get(index).getX();
	}

	/**
	 * Return the Y value of the point at the index in the first inner polygon
	 */
	public double getY(int index) {
		return m_List.get(index).getY();
	}

	@Override
	public boolean hasContact(Poly p) {
		if (p.getNumInnerPoly() > 1) {
			for (int i = 0; i < p.getNumInnerPoly(); ++i) {
				if (p.getInnerPoly(i).hasContact(this)) {
					return true;
				}
			}
		} else {
			if (!this.intersection(p).isEmpty()) {
				return true;
			}
			if (this.getNumPoints() < 3) {
				return false;
			}
			while (!(p instanceof PolySimple)) {
				p = p.getInnerPoly(0);
			}
			PolySimple simple = (PolySimple) p;
			for (int i = 0; i < this.m_List.size(); ++i) {
				for (int j = 1; j < simple.m_List.size(); ++j) {
					if (collide(this.m_List.get(i), this.m_List.get((i + 1)
							% this.m_List.size()), simple.m_List.get(j),
							simple.m_List.get((j + 1) % simple.m_List.size()))) {
						return true;
					}
				}
			}
		}
		return false;
	}

	/**
	 * Return the hashCode of the object.
	 * <p>
	 * <strong>WARNING:</strong>Hash and Equals break contract.
	 * 
	 * @return an integer value that is the same for two objects whenever their
	 *         internal representation is the same (equals() is true)
	 */
	@Override
	public int hashCode() {
		// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
		// !!! WARNING: This hash and equals break the contract. !!!
		// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
		int result = 17;
		result = 37 * result + m_List.hashCode();
		return result;
	}

	/**
	 * Return a Poly that is the intersection of this polygon with the given
	 * polygon. The returned polygon is simple.
	 * 
	 * @return The returned Poly <!--is of type PolySimple-->
	 */
	public Poly intersection(Poly p) {
		//return Clip.intersection(this, p, this.getClass());
		return Clip.intersection(this, p);
	}

	/**
	 * Return true if the given inner polygon is contributing to the set
	 * operation. This method should NOT be used outside the Clip algorithm.
	 * 
	 * @throws IllegalStateException
	 *             if <code>polyIndex != 0</code>
	 */
	public boolean isContributing(int polyIndex) {
		if (polyIndex != 0) {
			throw new IllegalStateException("PolySimple only has one poly");
		}
		return m_Contributes;
	}

	/**
	 * Return true if the polygon is empty
	 */
	public boolean isEmpty() {
		return m_List.isEmpty();
	}

	/**
	 * Always returns false since PolySimples cannot be holes.
	 */
	public boolean isHole() {
		return false;
	}

	/**
	 * Set whether or not this inner polygon is constributing to the set
	 * operation. This method should NOT be used outside the Clip algorithm.
	 * 
	 * @throws IllegalStateException
	 *             if <code>polyIndex != 0</code>
	 */
	public void setContributing(int polyIndex, boolean contributes) {
		if (polyIndex != 0) {
			throw new IllegalStateException("PolySimple only has one poly");
		}
		m_Contributes = contributes;
	}

	/**
	 * Throws IllegalStateException if called.
	 */
	public void setIsHole(boolean isHole) {
		throw new IllegalStateException("PolySimple cannot be a hole");
	}

	/**
	 * Return a string briefly describing the polygon.
	 */
	@Override
	public String toString() {
		return "PolySimple: num_points=" + getNumPoints();
	}

	/**
	 * Return a Poly that is the union of this polygon with the given polygon.
	 * The returned polygon is simple.
	 * 
	 * @return The returned Poly<!-- is of type PolySimple-->
	 */
	public Poly union(Poly p) {
		//return Clip.union(this, p, this.getClass());
		return Clip.union(this, p);
	}

	/**
	 * Return a Poly that is the exclusive-or of this polygon with the given
	 * polygon. The returned polygon is simple.
	 * 
	 * @return The returned Poly<!-- is of type PolySimple -->
	 */
	public Poly xor(Poly p) {
		//return Clip.xor(p, this, this.getClass());
		return Clip.xor(p, this);
	}

	@Override
	public void set(int index, double x, double y) {
		while (index >= m_List.size()) {
			m_List.add(new Point2D.Double());
		}
		m_List.get(index).setLocation(x, y);
	}

	@Override
	public void set(int index, Point2D point) {
		while (index >= m_List.size()) {
			m_List.add(new Point2D.Double());
		}
		m_List.get(index).setLocation(point);		
	}

	@Override
	public void remove(int index) {
		m_List.remove(index);		
	}

	// -----------------------
	// --- Package Methods ---
	// -----------------------
}
