package cz.cuni.amis.pogamut.ut2004.server.commands;

import java.awt.Color;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import cz.cuni.amis.pogamut.base.agent.worldview.IWorldView;
import cz.cuni.amis.pogamut.base.agent.worldview.WorldEventListener;
import cz.cuni.amis.pogamut.base.server.commands.IKeyEventListener;
import cz.cuni.amis.pogamut.base.server.commands.IPlayerCommunication;
import cz.cuni.amis.pogamut.base.server.commands.dialog.IDialog;
import cz.cuni.amis.pogamut.base.server.commands.dialog.IDialogComponent;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbcommands.DialogBegin;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbcommands.DialogCommand;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbcommands.DialogEnd;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbcommands.DialogItem;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbcommands.PlaySound;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbcommands.SetSendKeys;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbcommands.ShowText;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.DialogReply;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.KeyEvent;
import cz.cuni.amis.pogamut.ut2004.server.AbstractUT2004Server;
import cz.cuni.amis.pogamut.ut2004.server.commands.dialog.Dialog;
import cz.cuni.amis.pogamut.ut2004.server.commands.dialog.DialogComponent;

/**
 * Class handling communication with player - e.g. displaying of text messages,
 * playing sounds, creating dialogues etc.
 * 
 * @author Radim Vansa <radim.vansa@matfyz.cz>
 */
public class PlayerCommunication extends ServerCommands implements
		IPlayerCommunication {

	/**
	 * Dialog reply listener
	 */
	protected class DialogReplyListener implements
			WorldEventListener<DialogReply> {

		/**
		 * Constructor. Registers itself on the given WorldView object.
		 * 
		 * @param worldView
		 *            WorldView object to listen to.
		 */
		public DialogReplyListener(IWorldView worldView) {
			worldView.addListener(DialogReply.class, this);
		}

		@Override
		public void notify(DialogReply event) {
			IDialog dialog = dialogs.get(event.getId());
			if (dialog == null) {
				throw new IllegalArgumentException(
						"No apropriate dialog found!");
			}
			dialog.setData(event.getData());
			if (dialog.processCommand(event.getSourceId(), event.getCommand())) {
				dialogs.remove(event.getId());
			}
		}

	}

	/**
	 * Key event listener
	 */
	protected class KeyEventListener implements WorldEventListener<KeyEvent> {

		/**
		 * Constructor. Registers itself on the given WorldView object.
		 * 
		 * @param worldView
		 *            WorldView object to listen to.
		 */
		public KeyEventListener(IWorldView worldView) {
			worldView.addListener(KeyEvent.class, this);
		}

		@Override
		public void notify(KeyEvent event) {
			List<IKeyEventListener> list = keyListenersMap
															.get(event
																		.getPlayer());
			if (list == null) {
				return;
			}
			for (IKeyEventListener listener : list) {
				if (event.getAction().equalsIgnoreCase("PRESS")) {
					listener.keyPressed(event.getKey());
				} else if (event.getAction().equalsIgnoreCase("RELEASE")) {
					listener.keyReleased(event.getKey());
				} else if (event.getAction().equalsIgnoreCase("HOLD")) {
					listener.keyHeld(event.getKey());
				}
			}
		}

	}

	protected DialogReplyListener dialogReplyListener;

	protected KeyEventListener keyEventListener;

	protected Map<String, IDialog> dialogs = new HashMap<String, IDialog>();

	protected Map<String, List<IKeyEventListener>> keyListenersMap = new HashMap<String, List<IKeyEventListener>>();

	public PlayerCommunication(AbstractUT2004Server server, Logger log) {
		super(server, log);
		dialogReplyListener = new DialogReplyListener(server.getWorldView());
		keyEventListener = new KeyEventListener(server.getWorldView());
	}

	/**
	 * Displays a dialog on player's screen
	 * 
	 * @param dialog
	 */
	@Override
	public void showDialog(IDialog dialog) {
		if (!(dialog instanceof Dialog)) {
			throw new IllegalArgumentException(
					"This object should cooperate with UT2004 Dialog!");
		}
		if (dialog.isFinished()) {
			throw new IllegalArgumentException("This dialog was already used.");
		}
		if (dialog.getTargetPlayer() == null) {
			throw new IllegalArgumentException("No player specified");
		}
		dialog.setDisplayed(true);
		if (dialog.isDeployed()) {
			DialogCommand dialogShow = new DialogCommand();
			dialogShow.setId(dialog.getId());
			dialogShow.setCommand("SHOW");
			dialogShow.setPlayer(dialog.getTargetPlayer());
			server.getAct().act(dialogShow);
		} else {
			dialog.setPlayerCommunication(this);
			Dialog d = (Dialog) dialog;
			DialogBegin begin = d.createBeginCommand();
			server.getAct().act(begin);
			List<DialogItem> items = d.createItemCommands();
			for (DialogItem itemCommand : items) {
				itemCommand.setPlayer(d.getTargetPlayer());
				server.getAct().act(itemCommand);
			}
			DialogEnd end = d.createEndCommand();
			server.getAct().act(end);
			dialog.setDeployed();
			dialogs.put(dialog.getId(), dialog);
		}
	}

	/**
	 * Destroys the dialog
	 * 
	 * @param dialog
	 */
	@Override
	public void destroyDialog(IDialog dialog) {
		if (dialog.isFinished()) {
			throw new IllegalArgumentException(
					"The dialog was already destroyed!");
		}
		if (dialog.isDeployed()) {
			DialogCommand dialogCancel = new DialogCommand();
			dialogCancel.setId(dialog.getId());
			dialogCancel.setCommand("CANCEL");
			dialogCancel.setPlayer(dialog.getTargetPlayer());
			server.getAct().act(dialogCancel);
			dialogs.remove(dialog.getId());
		} else {
			throw new IllegalArgumentException(
					"The dialog wasn't displayed yet!");
		}
	}

	@Override
	public void hideDialog(IDialog dialog) {
		if (dialog.isFinished()) {
			throw new IllegalArgumentException("The dialog was destroyed!");
		}
		if (dialog.isDisplayed()) {
			dialog.setDisplayed(false);
			if (dialog.isDeployed()) {
				DialogCommand dialogHide = new DialogCommand();
				dialogHide.setId(dialog.getId());
				dialogHide.setCommand("HIDE");
				dialogHide.setPlayer(dialog.getTargetPlayer());
				server.getAct().act(dialogHide);
			}
		}
	}

	/**
	 * Updates currently displayed dialog
	 * 
	 * @param dialog
	 */
	@Override
	public void updateDialog(IDialog dialog) {
		if (!(dialog instanceof Dialog)) {
			throw new IllegalArgumentException(
					"This object should cooperate with UT2004 Dialog!");
		}
		updateDialogComponent(dialog, ((Dialog) dialog).getMainPanel());
	}

	/**
	 * Updates specific component of some displayed dialog
	 * 
	 * @param dialog
	 * @param component
	 */
	@Override
	public void updateDialogComponent(IDialog dialog, IDialogComponent component) {
		if (!(component instanceof DialogComponent)) {
			throw new IllegalArgumentException(
					"This object should cooperate with UT2004 dialog components!");
		}
		List<DialogItem> items = ((DialogComponent) component)
																.createItemCommands();
		for (DialogItem itemCommand : items) {
			itemCommand.setPlayer(dialog.getTargetPlayer());
			itemCommand.setDialogId(dialog.getId());
			if (itemCommand.getEffect() == null) {
				itemCommand.setEffect("EDIT");
			}
			server.getAct().act(itemCommand);
		}
	}

	/**
	 * Plays a sound to the player
	 * 
	 * @param playerName
	 *            Name of the affected player
	 * @param sound
	 *            Unreal reference to the sound, e.g.
	 *            "MyPackage.MySoundGroup.MySound". The sound has to be already
	 *            in some .uax file.
	 */
	@Override
	public void playSound(String playerName, String sound) {
		PlaySound playSound = new PlaySound();
		playSound.setPlayer(playerName);
		playSound.setSound(sound);
		server.getAct().act(playSound);
	}

	/**
	 * Plays a sound to the player in an infinite loop
	 * 
	 * @param playerName
	 *            Name of the affected player
	 * @param sound
	 *            Unreal reference to the sound, e.g.
	 *            "MyPackage.MySoundGroup.MySound". The sound has to be already
	 *            in some .uax file.
	 */
	@Override
	public void playSoundInLoop(String playerName, String sound) {
		PlaySound playSound = new PlaySound();
		playSound.setPlayer(playerName);
		playSound.setSound(sound);
		playSound.setLoop(true);
		server.getAct().act(playSound);
	}
	
	@Override
	public void stopSoundLoop(String playerName) {
		PlaySound playSound = new PlaySound();
		playSound.setPlayer(playerName);
		playSound.setSound(null);
		playSound.setLoop(true);
		server.getAct().act(playSound);
	}
	
	/**
	 * Removes a text from player's screen
	 * 
	 * @param playerName
	 *            Name of the affected player
	 * @param message
	 *            Text of message that should disappear
	 */
	@Override
	public void removeText(String playerName, String message) {
		ShowText showText = new ShowText();
		showText.setPlayer(playerName);
		showText.setText(message);
		showText.setShow(false);
		server.getAct().act(showText);
	}

	/**
	 * Adds a listener listening to player's key events
	 * 
	 * @param playerName
	 * @param listener
	 */
	@Override
	public void addKeyEventListener(String playerName,
			IKeyEventListener listener) {
		List<IKeyEventListener> list = keyListenersMap.get(playerName);
		if (list == null) {
			list = new ArrayList<IKeyEventListener>();
			keyListenersMap.put(playerName, list);
		}
		if (list.size() == 0) {
			SetSendKeys setSendKeys = new SetSendKeys();
			setSendKeys.setPlayer(playerName);
			setSendKeys.setSend(true);
			server.getAct().act(setSendKeys);
		}
		list.add(listener);
	}

	/**
	 * Removes a listener listening to player's key events
	 * 
	 * @param playerName
	 * @param listener
	 */
	@Override
	public void removeKeyEventListener(String playerName,
			IKeyEventListener listener) {
		List<IKeyEventListener> list = keyListenersMap.get(playerName);
		if (list == null) {
			return;
		}
		list.remove(listener);
		if (list.size() == 0) {
			SetSendKeys setSendKeys = new SetSendKeys();
			setSendKeys.setPlayer(playerName);
			setSendKeys.setSend(false);
			server.getAct().act(setSendKeys);
		}
	}

	/**
	 * Displays a text on top of screen in white color. The text will persist
	 * there until removed.
	 * 
	 * @param playerName
	 *            Name of the player who should see the message
	 * @param message
	 *            The message displayed
	 */
	@Override
	public void showText(String playerName, String message) {
		showText(playerName, message, 0, new Color(255, 255, 255));
	}

	/**
	 * Displays a text on top of screen in white color.
	 * 
	 * @param playerName
	 *            Name of the player who should see the message
	 * @param message
	 *            The message displayed
	 * @param timeout
	 *            Amount of time (in seconds) after which the text disappears.
	 *            Zero means that the text will not disappear until removed.
	 */
	@Override
	public void showText(String playerName, String message, double timeout) {
		showText(playerName, message, timeout, new Color(255, 255, 255));
	}

	/**
	 * Displays a text on top of screen
	 * 
	 * @param playerName
	 *            Name of the player who should see the message
	 * @param message
	 *            The message displayed
	 * @param timeout
	 *            Amount of time (in seconds) after which the text disappears.
	 *            Zero means that the text will not disappear until removed.
	 * @param color
	 *            Color in which will be the text printed
	 */
	@Override
	public void showText(String playerName, String message, double timeout,
			Color color) {
		ShowText showText = new ShowText();
		showText.setPlayer(playerName);
		showText.setShow(true);
		showText.setText(message);
		showText.setTextColor(color);
		showText.setTime(timeout);
		server.getAct().act(showText);
	}

	@Override
	public IDialog createDialog(String name) {
		return new Dialog(name);
	}
}
