package cz.cuni.amis.pogamut.edu.utils;

/**
 * Implements the ability to carry some name. The name can be
 * changed during object's lifetime. The last different name is
 * also stored.
 * 
 * @author Radim Vansa <radim.vansa@matfyz.cz>
 * 
 */
public abstract class NamedObject implements INamed, Cloneable {

	private static long objectsCounter = 100000; // we start far away to
	/**
	 * Creates a new unique name
	 * 
	 * @return
	 */
	public static String createName() {
		return "o" + Long.toString(objectsCounter++);
	}
	private boolean generated = false;
	// prevent form o1, o2...
	private String name = null;

	private String previousName = null;

	/**
	 * Creates a new NamedObject with generated random name
	 */
	public NamedObject() {
		this.name = createName();
		generated = true;
	}

	/**
	 * Creates a new NamedObject with specified name
	 * 
	 * @param name
	 */
	public NamedObject(String name) {
		this.name = name;
	}

	/**
	 * Copies the object but generates new unique name to the new instance
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		NamedObject no = (NamedObject) super.clone();
		// we don't want two object with the same name
		no.name = createName();
		return no;
	}

	/**
	 * Two objects are considered equal if they share the same name.
	 * 
	 * @param o
	 */
	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}
		if (o instanceof NamedObject) {
			return name.equals(((NamedObject) o).getName());
		}
		return false;
	}
	
	public void generateNewName() {
		setName(createName());
	}

	/**
	 * @return object's name
	 */
	@Override
	public String getName() {
		return name;
	}

	/**
	 * @return previous name of this object
	 */
	@Override
	public String getPreviousName() {
		return previousName;
	}

	/**
	 * True if the name was generated, false if it was manually assigned
	 * 
	 * @return
	 */
	public boolean hasGeneratedName() {
		return generated;
	}

	/**
	 * Two objects are considered equal if they share the same name and
	 * therefore this returns the hashCode of it's name.
	 */
	@Override
	public int hashCode() {
		return name.hashCode();
	}

	/**
	 * Sets object's name
	 * 
	 * @param name
	 */
	@Override
	public void setName(String name) {
		this.previousName = name;
		this.name = name;
		generated = false;
	}

	/**
	 * @return String in form "name: super.toString()"
	 */
	@Override
	public String toString() {
		return this.name + ": " + super.toString();
	}
}
