package cz.cuni.amis.pogamut.edu.ut2004.dialogs;

import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import cz.cuni.amis.pogamut.base.server.commands.dialog.DialogListenerAdapter;
import cz.cuni.amis.pogamut.base.server.commands.dialog.IDialog;
import cz.cuni.amis.pogamut.base.server.commands.dialog.IDialogListener;
import cz.cuni.amis.pogamut.edu.agent.Agent;
import cz.cuni.amis.pogamut.edu.map.marks.IMark;
import cz.cuni.amis.pogamut.edu.map.marks.POI;
import cz.cuni.amis.pogamut.edu.map.marks.Path;
import cz.cuni.amis.pogamut.edu.map.marks.PathPoint;
import cz.cuni.amis.pogamut.edu.ut2004.dialogs.MapDialog.IAgentDecorator;
import cz.cuni.amis.pogamut.edu.utils.ICoordsTransformer;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbcommands.DialogItem;
import cz.cuni.amis.pogamut.ut2004.server.commands.dialog.DialogButton;
import cz.cuni.amis.pogamut.ut2004.server.commands.dialog.DialogImageLayer;
import cz.cuni.amis.pogamut.ut2004.server.commands.dialog.DialogLayerContainer;
import cz.cuni.amis.pogamut.ut2004.server.commands.dialog.DialogPanel;
import cz.cuni.amis.pogamut.ut2004.server.commands.dialog.DialogPathLayer;
import cz.cuni.amis.pogamut.ut2004.server.commands.dialog.DialogTextLayer;
import cz.cuni.amis.utils.Dimension2D_Double;

/**
 * The MapDialogComponent is a DialogPanel with four scroll buttons on all sides
 * of the DialogLayerContainer with map and marked objects and with a confirm
 * button in the selection mode.
 * 
 * @author Radim Vansa <radim.vansa@matfyz.cz>
 * 
 */
public class MapDialogComponent extends DialogPanel {

	protected static final String SYMBOL_PACKAGE = "EducationalScenarios.Symbols";
	protected static final String OTHER_PACKAGE = "EducationalScenarios.Other";
	protected static final int SYMBOL_SIZE = 16;
	protected DialogLayerContainer mapContainer = new DialogLayerContainer();
	protected DialogButton left = new DialogButton();
	protected DialogButton right = new DialogButton();
	protected DialogButton up = new DialogButton();
	protected DialogButton down = new DialogButton();
	protected DialogButton confirm = new DialogButton();
	protected Scroller scroller = new Scroller();
	protected DialogImageLayer mapImage = new DialogImageLayer();
	protected DialogImageLayer selection = new DialogImageLayer();

	protected Map<POI, DialogImageLayer> poiSymbols = new HashMap<POI, DialogImageLayer>();
	protected Map<POI, DialogTextLayer> poiTexts = new HashMap<POI, DialogTextLayer>();
	protected Map<Path, DialogPathLayer> pathLayers = new HashMap<Path, DialogPathLayer>();
	protected List<IMark> marks = new ArrayList<IMark>();

	protected Map<Agent, DialogImageLayer> agentSymbols = new HashMap<Agent, DialogImageLayer>();
	protected Map<Agent, DialogTextLayer> agentNames = new HashMap<Agent, DialogTextLayer>();

	protected boolean selectionEnabled = false;
	protected Point2D lastClick = null;

	/**
	 * The listener reacting on selection of the spot (in selection mode).
	 * 
	 * @author Radim Vansa <radim.vansa@matfyz.cz>
	 *
	 */
	protected class ClickListener extends DialogListenerAdapter {
		@Override
		public void dialogCommand(IDialog dialog, String command,
				Map<String, String> data) {
			if (selectionEnabled) {
				double clickX = Double
										.parseDouble(data
															.get(mapContainer
																				.getId()
																	+ ".ClickX"));
				double clickY = Double
										.parseDouble(data
															.get(mapContainer
																				.getId()
																	+ ".ClickY"));
				lastClick = new Point2D.Double(clickX, clickY);
				selection.setShift(lastClick);
				selection.setVisible(true);
				left.setSize(new Dimension2D_Double(0.075, 0.6));
				right.setSize(new Dimension2D_Double(0.075, 0.6));
				down.setPosition(new Point2D.Double(0.075, 0.7));
				mapContainer.setSize(new Dimension2D_Double(0.85, 0.6));
				confirm.setVisible(true);
				dialog.getPlayerCommunication().updateDialogComponent(dialog,
					MapDialogComponent.this);
			}
		}
	}

	/**
	 * The listener scrolling the DialogLayerContainer with the map.
	 * @author Radim Vansa <radim.vansa@matfyz.cz>
	 *
	 */
	protected class Scroller extends DialogListenerAdapter {

		@Override
		public void dialogCommand(IDialog dialog, String command,
				Map<String, String> data) {
			if (command.equalsIgnoreCase("SI_LEFT")) {
				double x = mapContainer.getViewport().getX() - 50;
				if (x < 0) {
					x = 0;
				}
				mapContainer.setViewport(new Point2D.Double(x,
						mapContainer.getViewport().getY()));
			} else if (command.equalsIgnoreCase("SI_RIGHT")) {
				double x = mapContainer.getViewport().getX() + 50;
				double maxX = Double.valueOf(data.get(mapContainer.getId()
						+ ".Width"))
						- Double.valueOf(data.get(mapContainer.getId()
								+ ".DisplayedWidth"));
				if (x > maxX) {
					x = maxX;
				}
				mapContainer.setViewport(new Point2D.Double(x,
						mapContainer.getViewport().getY()));
			} else if (command.equalsIgnoreCase("SI_UP")) {
				double y = mapContainer.getViewport().getY() - 50;
				if (y < 0) {
					y = 0;
				}
				mapContainer.setViewport(new Point2D.Double(
						mapContainer.getViewport().getX(), y));
			} else if (command.equalsIgnoreCase("SI_DOWN")) {
				double y = mapContainer.getViewport().getY() + 50;
				double maxY = Double.valueOf(data.get(mapContainer.getId()
						+ ".Height"))
						- Double.valueOf(data.get(mapContainer.getId()
								+ ".DisplayedHeight"));
				if (y > maxY) {
					y = maxY;
				}
				mapContainer.setViewport(new Point2D.Double(
						mapContainer.getViewport().getX(), y));
			}
			dialog.getPlayerCommunication().updateDialogComponent(dialog,
				mapContainer);
		}
	}

	public MapDialogComponent() {
		super();
		init();
	}

	private void init() {
		mapContainer.setPosition(new Point2D.Double(0.075, 0.1));
		mapContainer.setSize(new Dimension2D_Double(0.85, 0.8));
		mapContainer.setViewport(new Point2D.Double(0, 0));
		mapContainer.addListener(new ClickListener());
		mapContainer.setAction("CLICK");
		this.addComponent(mapContainer);
		left.setPosition(new Point2D.Double(0, 0.1));
		left.setSize(new Dimension2D_Double(0.075, 0.8));
		left.setImage(OTHER_PACKAGE + ".arrowleft");
		left.setAction("SI_LEFT");
		left.addListener(scroller);
		this.addComponent(left);
		right.setPosition(new Point2D.Double(0.925, 0.1));
		right.setSize(new Dimension2D_Double(0.075, 0.8));
		right.setAction("SI_RIGHT");
		right.setImage(OTHER_PACKAGE + ".arrowright");
		right.addListener(scroller);
		this.addComponent(right);
		up.setPosition(new Point2D.Double(0.075, 0));
		up.setSize(new Dimension2D_Double(0.85, 0.1));
		up.setAction("SI_UP");
		up.setImage(OTHER_PACKAGE + ".arrowup");
		up.addListener(scroller);
		this.addComponent(up);
		down.setPosition(new Point2D.Double(0.075, 0.9));
		down.setSize(new Dimension2D_Double(0.85, 0.1));
		down.setAction("SI_DOWN");
		down.setImage(OTHER_PACKAGE + ".arrowdown");
		down.addListener(scroller);
		this.addComponent(down);
		confirm.setVisible(false);
		confirm.setAction("CONFIRM");
		confirm.setPosition(new Point2D.Double(0.3, 0.85));
		confirm.setSize(new Dimension2D_Double(0.4, 0.1));
		confirm.setText("OK");
		this.addComponent(confirm);
		mapContainer.addLayer(mapImage);
		mapContainer.addLayer(selection);
		selection.setSourcePart(new Rectangle2D.Double(0, 0, 0, 0));
		selection.setVisible(false);
		selection.setImage(SYMBOL_PACKAGE + ".pointingarrow");
	}

	public MapDialogComponent(String id) {
		super(id);
		init();
	}

	public void setMap(String image) {
		mapImage.setImage(image);
	}

	public void addPOI(POI poi, ICoordsTransformer transformer) {
		Point2D point = transformer.location2point(poi.getLocation());
		if (poi.getSymbol() != POI.Symbol.NONE) {
			DialogImageLayer symbol = new DialogImageLayer();
			symbol.setImage(SYMBOL_PACKAGE + "."
					+ poi.getSymbol().name().toLowerCase());
			symbol.setSourcePart(new Rectangle2D.Double(0, 0, SYMBOL_SIZE,
					SYMBOL_SIZE));
			symbol.setColor(poi.getColor());
			symbol.setVisible(poi.isVisible());
			symbol.setShift(new Point2D.Double(point.getX() - SYMBOL_SIZE / 2,
					point.getY() - SYMBOL_SIZE));
			poiSymbols.put(poi, symbol);
			mapContainer.addLayer(symbol);
		}
		if (poi.getDescription().length() != 0) {
			DialogTextLayer text = new DialogTextLayer();
			text.setText(poi.getDescription());
			text.setTextColor(poi.getColor());
			text.setTextAlign("CENTER");
			text.setVisible(poi.isVisible());
			text.setShift(point);
			poiTexts.put(poi, text);
			mapContainer.addLayer(text);
		}
		marks.add(poi);
	}

	public void addPath(Path path, ICoordsTransformer transformer) {
		DialogPathLayer pathLayer = new DialogPathLayer();
		pathLayers.put(path, pathLayer);
		for (PathPoint point : path.getPoints()) {
			pathLayer.addPoint(transformer.location2point(point.getLocation()));
		}
		pathLayer.setVisible(path.isVisible());
		pathLayer.setWidth(path.getWidth());
		pathLayer.setColor(path.getColor());
		mapContainer.addLayer(pathLayer);
		marks.add(path);
	}

	public void removePath(Path path) {
		mapContainer.removeLayer(pathLayers.remove(path));
		marks.remove(path);
	}

	public void removePOI(POI poi) {
		mapContainer.removeLayer(poiSymbols.remove(poi));
		mapContainer.removeLayer(poiTexts.remove(poi));
		marks.remove(poi);
	}

	public boolean updateMarksVisiblity() {
		boolean needsUpdate = false;
		for (IMark mark : marks) {
			if (mark instanceof POI) {
				DialogImageLayer symbol = poiSymbols.get((POI) mark);
				if (symbol != null && mark.isVisible() != symbol.isVisible()) {
					symbol.setVisible(mark.isVisible());
					needsUpdate = true;
				}
				DialogTextLayer text = poiTexts.get((POI) mark);
				if (text != null) {
					text.setVisible(mark.isVisible());
				}
			} else if (mark instanceof Path) {
				DialogPathLayer layer = pathLayers.get((Path) mark);
				if (layer != null && mark.isVisible() != layer.isVisible()) {
					layer.setVisible(mark.isVisible());
					needsUpdate = true;
				}
			}
		}
		return needsUpdate;
	}

	@Override
	public List<DialogItem> createItemCommands() {
		left.setBgColor(getBgColor());
		right.setBgColor(getBgColor());
		up.setBgColor(getBgColor());
		down.setBgColor(getBgColor());
		confirm.setBgColor(getBgColor());
		return super.createItemCommands();
	}

	public boolean updateAgents(Collection<Agent> agents,
			IAgentDecorator decorator, ICoordsTransformer transformer) {
		boolean needsUpdate = false;
		for (Agent agent : agents) {
			DialogImageLayer dil = agentSymbols.get(agent);
			DialogTextLayer dtl = agentNames.get(agent);
			if (dil == null) {
				dil = new DialogImageLayer();
				dil.setImage(decorator.getSymbol(agent));
				dil.setColor(decorator.getColor(agent));
				dil.setSourcePart(new Rectangle2D.Double(0, 0, SYMBOL_SIZE,
						SYMBOL_SIZE));
				dtl = new DialogTextLayer();
				dtl.setText(agent.getInGameName());
				dtl.setTextColor(decorator.getColor(agent));
				dtl.setTextAlign("CENTER");
				agentSymbols.put(agent, dil);
				agentNames.put(agent, dtl);
				mapContainer.addLayer(dil);
				mapContainer.addLayer(dtl);
				needsUpdate = true;
			}
			Point2D point = transformer.location2point(agent.getLocation());
			if (!point.equals(dtl.getShift())) {
				dil.setShift(new Point2D.Double(point.getX() - SYMBOL_SIZE / 2,
						point.getY() - SYMBOL_SIZE));
				dil.setVisible(true);
				dtl.setShift(point);
				dtl.setVisible(true);
				needsUpdate = true;
			}
		}
		HashSet<Agent> agentSet = new HashSet<Agent>(agents);
		for (Agent agent : agentSymbols.keySet()) {
			if (!agentSet.contains(agent)) {
				agentSymbols.get(agent).setVisible(false);
				agentNames.get(agent).setVisible(false);
				needsUpdate = true;
			}
		}
		return needsUpdate;
	}

	public void setSelectionEnabled(boolean enabled, IDialog dialog) {
		selectionEnabled = enabled;
		lastClick = null;
		if (!enabled) {
			selection.setVisible(false);
			left.setSize(new Dimension2D_Double(0.075, 0.8));
			right.setSize(new Dimension2D_Double(0.075, 0.8));
			down.setPosition(new Point2D.Double(0.075, 0.9));
			mapContainer.setSize(new Dimension2D_Double(0.85, 0.8));
			confirm.setVisible(false);
			dialog.getPlayerCommunication().updateDialogComponent(dialog, this);
		}
	}

	public Point2D getLastClick() {
		return lastClick;
	}

	public void addSelectionListener(IDialogListener listener) {
		confirm.addListener(listener);
	}

	public boolean hasSelectionEnabled() {
		return selectionEnabled;
	}
}
