package cz.cuni.amis.pogamut.edu.ut2004.controlserver;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import cz.cuni.amis.pogamut.base.agent.worldview.WorldEventListener;
import cz.cuni.amis.pogamut.base.communication.connection.socket.ISocketConnectionAddress;
import cz.cuni.amis.pogamut.base.communication.connection.socket.SocketConnectionAddress;
import cz.cuni.amis.pogamut.base.exceptions.PogamutException;
import cz.cuni.amis.pogamut.base.factory.IAgentFactory;
import cz.cuni.amis.pogamut.base.factory.guice.GuiceAgentFactory;
import cz.cuni.amis.pogamut.base.server.commands.IBotControl;
import cz.cuni.amis.pogamut.base.server.commands.IPlayerCommunication;
import cz.cuni.amis.pogamut.base.server.commands.IPlayerControl;
import cz.cuni.amis.pogamut.edu.Scenario;
import cz.cuni.amis.pogamut.edu.agent.Bot;
import cz.cuni.amis.pogamut.edu.agent.BotExecutor;
import cz.cuni.amis.pogamut.edu.controlserver.IControlServer;
import cz.cuni.amis.pogamut.edu.ut2004.UT2004Scenario;
import cz.cuni.amis.pogamut.edu.ut2004.agent.UT2004Player;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbcommands.GetPlayers;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Player;
import cz.cuni.amis.pogamut.ut2004.communication.translator.events.PlayerListObtained;
import cz.cuni.amis.pogamut.ut2004.factory.guice.UT2004ServerModule;
import cz.cuni.amis.pogamut.ut2004.server.UT2004Server;
import cz.cuni.amis.pogamut.ut2004.server.commands.BotControl;
import cz.cuni.amis.pogamut.ut2004.server.commands.PlayerCommunication;
import cz.cuni.amis.pogamut.ut2004.server.commands.PlayerControl;

/**
 * The UT2004 implementation of IControlServer. It starts the server from
 * PogamutUT2004 and connects it to the Unreal Tournament. It also repeatedly
 * asks for a list of agents present in the world. After the message with actual
 * agents and their coordinates comes it creates new players or bots and
 * registers them to the Scenario or removes those who are not mentioned in the
 * message. If there are no agent additions or removals it just updates their
 * position and notifies the Map about the change.
 * 
 * It also holds references to UT2004ScenarioAddons modules.
 * 
 * @author Radim Vansa <radim.vansa@matfyz.cz>
 * 
 */
public class UT2004ControlServer implements IControlServer {

	protected UT2004Server server = null;
	protected BotControl botControl;
	protected PlayerControl playerControl;
	protected PlayerCommunication playerCommunication;
	protected UT2004Scenario scenario;
	protected PlayersUpdater playersUpdater;

	/**
	 * The thread that updates the rest of ES about changes in players list.
	 * 
	 * @author Radim Vansa <radim.vansa@matfyz.cz>
	 * 
	 */
	protected class PlayersUpdater extends Thread implements
			WorldEventListener<PlayerListObtained> {
		private Set<Player> playerSet = new HashSet<Player>();
		private List<Player> playerList = new ArrayList<Player>();
		private long mark = 0;

		public void notify(PlayerListObtained event) {
			List<Player> list = event.getPlayers();
			synchronized (scenario.getWorkingMemory()) {
				for (Player p : list) {
					if (!playerSet.contains(p)) {
						playerSet.add(p);
						playerList.add(p);
						addAgent(p);
					}
					cz.cuni.amis.pogamut.edu.agent.Agent agent;
					if (p.getId().getStringId().indexOf("Player") != -1) {
						agent = scenario.getPlayerWithInGameName(p.getName());
					} else {
						agent = scenario.getAgent(p.getName());
					}
					agent.setLocation(p.getLocation());
					agent.setRotation(p.getRotation());
					scenario.getMap().updateSimpleAreas(agent, mark);
					++mark;
				}
				Set<Player> set = new HashSet<Player>(list);
				for (Player p : playerList) {
					if (!set.contains(p)) {
						removeAgent(p);
					}
				}
			}
		}

		private void removeAgent(Player p) {
			scenario.removeAgent(scenario.getAgent(p.getName()));
		}

		private void addAgent(Player p) {
			if (p.getId().getStringId().indexOf("Player") != -1) {
				// ScenarioxPlayer, GBxPlayer...
				UT2004Player player = new UT2004Player("Player"
						+ (scenario.getPlayerCount() + 1), p.getName(),
						scenario);
				player.setLocation(p.getLocation());
				player.setRotation(p.getRotation());
				scenario.addPlayer(player);
			} else if (p.getId().getStringId().indexOf("Bot") != -1) {
				Bot bot = new Bot(BotExecutor.yieldBot(p.getName()),
						p.getName());
				bot.setLocation(p.getLocation());
				bot.setRotation(p.getRotation());
				scenario.addBot(bot);
			}
		}

		@Override
		public void run() {
			boolean notInterrupted = true;
			while (notInterrupted) {
				server.getAct().act(new GetPlayers());
				try {
					sleep(1000);
				} catch (InterruptedException e) {
					notInterrupted = false;
				}
			}
		}
	}

	public UT2004ControlServer(UT2004Scenario scenario, Logger logger) {
		this.scenario = scenario;
		scenario.setControlServer(this);
		UT2004ServerModule module = new UT2004ServerModule();
		IAgentFactory<ISocketConnectionAddress> factory = new GuiceAgentFactory<ISocketConnectionAddress>(
				module);

		try {
			server = (UT2004Server) factory
											.newAgent(new SocketConnectionAddress(
													"localhost", 3001));
		} catch (PogamutException e) {
			logger.severe("Can't create server: " + e.getMessage());
		}

		server.getLogger().addConsoleHandlersToAllCategories();
		server.getLogger().setLevel(Level.SEVERE);

		try {
			server.start();
		} catch (PogamutException e) {
			logger.severe("Can't start the server: " + e.getMessage());
		}

		playerControl = new PlayerControl(server, logger);
		playerCommunication = new PlayerCommunication(server, logger);
		botControl = new BotControl(server, logger);

		playersUpdater = new PlayersUpdater();
		server.getWorldView().addListener(PlayerListObtained.class,
			playersUpdater);
		playersUpdater.start();
	}

	@Override
	public IPlayerControl getPlayerControl() {
		return playerControl;
	}

	@Override
	public IPlayerCommunication getPlayerCommunication() {
		return playerCommunication;
	}

	public void waitUntilPlayersConnect(int numberOfPlayers) {
		while (scenario.getPlayerCount() < numberOfPlayers) {
			try {
				Thread.sleep(100);
			} catch (InterruptedException e) {
				e.printStackTrace();
			}
		}
	}

	@Override
	public void setScenario(Scenario scenario) {
		if (scenario instanceof UT2004Scenario) {
			this.scenario = (UT2004Scenario) scenario;
		} else {
			throw new IllegalArgumentException(
					"The argument should be UT2004 scenario!");
		}
	}

	@Override
	public IBotControl getBotControl() {
		return botControl;
	}

}
