package cz.cuni.amis.pogamut.edu.map.areas;

import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.LinkedList;
import java.util.List;

import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;

import com.seisw.util.geom.Poly;
import com.seisw.util.geom.PolyDefault;

import cz.cuni.amis.pogamut.base3d.worldview.objects.Location;
import cz.cuni.amis.pogamut.edu.l10n.CannotTranslateException;
import cz.cuni.amis.pogamut.edu.l10n.ITranslator;
import cz.cuni.amis.pogamut.edu.map.MapData;
import cz.cuni.amis.pogamut.edu.map.utils.LocationComparator;
import cz.cuni.amis.pogamut.edu.utils.IRescaler;

/**
 * The only simple shaped area. It is limited to non-intersecting polygons for
 * writing into the map file but theoretically it could have multiple inner
 * polygons (that can happen after intersection).
 * The area is limited by it's top and bottom. The z-coordinates are by default
 * from minus infinity to infinity.
 * For editor and storage into map file the polygon uses shape hints but they
 * have no other effects.
 * 
 * @author Radim Vansa <radim.vansa@matfyz.cz>
 * 
 */
public class PolygonArea extends AbstractArea implements Cloneable {

	public enum Shape {
		CIRCLE, POLYGON, RECT, SPACE
	}

	protected static final int CIRCLE_POINTS = 16;

	public static boolean haveContact(PolygonArea area1, PolygonArea area2) {
		return area1.poly.hasContact(area2.poly);
	};

	protected double bottom = Double.NEGATIVE_INFINITY;
	protected Location center;
	protected Poly poly = new PolyDefault();
	protected Shape shapeHint = Shape.POLYGON;
	protected double top = Double.POSITIVE_INFINITY;

	public PolygonArea() {
	}

	public void addPoint(Location p) {
		poly.add(new Point2D.Double(p.getX(), p.getY()));
		center = null;
	}

	@Override
	public Object clone() throws CloneNotSupportedException {
		PolygonArea pa = (PolygonArea) super.clone();
		pa.poly = (Poly) this.poly.clone();
		return pa;
	}

	public boolean contains(Location p, boolean ignoreZCoord) {
		if (poly.getNumPoints() < 3) {
			return false;
		} else {
			return (ignoreZCoord || LocationComparator.hasZBetween(p,
				this.bottom, this.top))
					&& poly.contains(new Point2D.Double(p.getX(), p.getY()));
		}
	}

	public boolean containsExactly(IArea area) {
		return this.equals(area);
	}

	public void differentiate(PolygonArea polygon) {
		this.poly = this.poly.difference(polygon.poly);
	}

	@Override
	public boolean equals(Object other) {
		if (other == this)
			return true;
		if (other == null)
			return false;
		if (this.getClass() != other.getClass())
			return false;
		PolygonArea area = (PolygonArea) other;
		if (this.bottom != area.bottom || this.top != area.top)
			return false;
		if (!this.poly.equals(area.poly))
			return false;
		return true;
	}

	public void generateCircle(double cx, double cy, double radius) {
		setShapeHint(Shape.CIRCLE);
		poly.clear();
		center = null;
		for (int i = 0; i < CIRCLE_POINTS; ++i) {
			double angle = 2 * Math.PI * (double) i / CIRCLE_POINTS;
			poly.add(new Point2D.Double(cx + radius * Math.sin(angle), cy
					- radius * Math.cos(angle)));
		}
	}

	/**
	 * This returns all points defining this polygon, even if it has more
	 * sub-polygons. You mostly don't need this method, because polygons with
	 * sub-polygons are created only after intersection and they can't be
	 * declared in the map directly.
	 * 
	 * @return
	 */
	public List<List<Location>> getAllPoints() {
		List<List<Location>> list = new LinkedList<List<Location>>();
		for (int i = 0; i < poly.getNumInnerPoly(); ++i) {
			List<Location> list2 = new LinkedList<Location>();
			list.add(list2);
			Poly p = poly.getInnerPoly(i);
			for (int j = 0; j < p.getNumPoints(); ++j) {
				list2.add(new Location(p.getX(j), p.getY(j)));
			}
		}
		return list;
	}

	public double getBottom() {
		return this.bottom;
	}

	public Rectangle2D getBounds() {
		return poly.getBounds();
	}

	public Location getCenter() {
		if (center == null) {
			double x = 0;
			double y = 0;
			int sum = 0;
			for (int i = 0; i < poly.getNumInnerPoly(); ++i) {
				Poly p = poly.getInnerPoly(i);
				if (!p.isHole()) {
					for (int j = 0; j < poly.getNumPoints(); ++j) {
						x += p.getX(j);
						y += p.getY(j);
						++sum;
					}
				}
			}
			center = new Location(x / sum, y / sum);
		}
		return new Location(center.x, center.y);
	}

	public int getNumPoints() {
		return poly.getNumPoints();
	}

	public Location getPoint(int index) {
		Location l = new Location();
		l.x = poly.getX(index);
		l.y = poly.getY(index);
		return l;
	}

	@Override
	public PolygonArea getPolygonArea() {
		return this;
	}

	public Shape getShapeHint() {
		return shapeHint;
	}

	@Override
	public double getSurface() {
		return this.poly.getArea();
	}

	public double getTop() {
		return this.top;
	}

	public void intersect(PolygonArea polygon) {
		this.poly = this.poly.intersection(polygon.poly);
	}

	public void removeAllPoints() {
		poly.clear();
		center = null;
	}

	public void removePoint(int index) {
		poly.remove(index);
		center = null;
	}

	public void setPoint(int index, double x, double y) {
		poly.set(index, x, y);
		center = null;
	}

	public void setPoint(int index, Location l) {
		poly.set(index, l.x, l.y);
		center = null;
	}

	public void setShapeHint(Shape shapeHint) {
		this.shapeHint = shapeHint;
	}

	public void setToUniverse() {
		this.poly.add(new Point2D.Double(-10e100, -10e100));
		this.poly.add(new Point2D.Double(-10e100, 10e100));
		this.poly.add(new Point2D.Double(10e100, 10e100));
		this.poly.add(new Point2D.Double(10e100, -10e100));
	}

	public void setZLimits(double bottom, double top) {
		if (bottom < top) {
			this.bottom = bottom;
			this.top = top;
		} else {
			this.bottom = top;
			this.top = bottom;
		}
	}

	@Override
	public void writeXML(XMLStreamWriter writer, int indent,
			ITranslator translator) throws XMLStreamException {
		MapData.writeXMLIndent(writer, indent);
		switch (shapeHint) {
		case CIRCLE: {
			try {
				writer.writeStartElement(translator.translateTo("circle"));
			} catch (CannotTranslateException e) {
				writer.writeStartElement("circle");
			}
			Location center = getCenter();
			double radius = Math.sqrt((center.x - poly.getX(0))
					* (center.x - poly.getX(0)) + (center.y - poly.getY(0))
					* (center.y - poly.getY(0)));
			if (!hasGeneratedName()) {
				writer.writeAttribute("id", getName());
			}
			writer.writeAttribute("cx", String.valueOf(center.x));
			writer.writeAttribute("cy", String.valueOf(center.y));
			writer.writeAttribute("radius", String.valueOf(radius));
			break;
		}
		case RECT:
		case SPACE: {
			if (shapeHint == Shape.SPACE) {
				try {
					writer.writeStartElement(translator.translateTo("space"));
				} catch (CannotTranslateException e) {
					writer.writeStartElement("space");
				}
			} else {
				try {
					writer.writeStartElement(translator.translateTo("rect"));
				} catch (CannotTranslateException e) {
					writer.writeStartElement("rect");
				}
			}

			Rectangle2D rect = getBounds();
			if (!hasGeneratedName()) {
				writer.writeAttribute("id", getName());
			}
			writer.writeAttribute("x1", String.valueOf(rect.getMinX()));
			writer.writeAttribute("y1", String.valueOf(rect.getMinY()));
			if (shapeHint == Shape.SPACE) {
				writer.writeAttribute("z1", String.valueOf(bottom));
			}
			writer.writeAttribute("x2", String.valueOf(rect.getMaxX()));
			writer.writeAttribute("y2", String.valueOf(rect.getMaxY()));
			if (shapeHint == Shape.SPACE) {
				writer.writeAttribute("z2", String.valueOf(top));
			}
			break;
		}
		case POLYGON: {
			try {
				writer.writeStartElement(translator.translateTo("polygon"));
			} catch (CannotTranslateException e) {
				writer.writeStartElement("CannotTranslate");
			}
			if (!hasGeneratedName()) {
				writer.writeAttribute("id", getName());
			}
			for (int i = 0; i < poly.getNumPoints(); ++i) {
				writer.writeAttribute("x" + (i + 1),
					String.valueOf(poly.getX(i)));
				writer.writeAttribute("y" + (i + 1),
					String.valueOf(poly.getY(i)));
			}
			break;
		}
		}
		MapData.writeXMLIndent(writer, indent);
		writer.writeEndElement();
	}

	@Deprecated
	public String toXMLString(ITranslator translator) {
		try {
			switch (shapeHint) {
			case CIRCLE: {
				double cx = 0;
				double cy = 0;
				double radius = 0;
				for (int i = 0; i < poly.getNumPoints(); ++i) {
					cx += poly.getX(i);
					cy += poly.getY(i);
				}
				cx /= poly.getNumPoints();
				cy /= poly.getNumPoints();
				radius = Math.sqrt((cx - poly.getX(0)) * (cx - poly.getX(0))
						+ (cy - poly.getY(0)) * (cy - poly.getY(0)));
				StringBuilder xml = new StringBuilder();
				xml.append('<');
				xml.append(translator.translateTo("circle"));
				if (!hasGeneratedName()) {
					xml.append(" id=\"");
					xml.append(getName());
					xml.append('"');
				}
				xml.append(" cx=\"");
				xml.append(cx);
				xml.append("\" cy=\"");
				xml.append(cy);
				xml.append("\" radius=\"");
				xml.append(radius);
				xml.append("\"/>");
				return xml.toString();
			}
			case RECT: {
				Rectangle2D rect = getBounds();
				StringBuilder xml = new StringBuilder();
				xml.append('<');
				xml.append(translator.translateTo("rect"));
				if (!hasGeneratedName()) {
					xml.append(" id=\"");
					xml.append(getName());
					xml.append('"');
				}
				xml.append(" x1=\"");
				xml.append(rect.getMinX());
				xml.append("\" y1=\"");
				xml.append(rect.getMinY());
				xml.append("\" x2=\"");
				xml.append(rect.getMaxX());
				xml.append("\" y2=\"");
				xml.append(rect.getMaxY());
				xml.append("\"/>");
				return xml.toString();
			}
			case SPACE: {
				Rectangle2D rect = getBounds();
				StringBuilder xml = new StringBuilder();
				xml.append('<');
				xml.append(translator.translateTo("rect"));
				if (!hasGeneratedName()) {
					xml.append(" id=\"");
					xml.append(getName());
					xml.append('"');
				}
				xml.append(" x1=\"");
				xml.append(rect.getMinX());
				xml.append("\" y1=\"");
				xml.append(rect.getMinY());
				xml.append("\" z1=\"");
				xml.append(bottom);
				xml.append("\" x2=\"");
				xml.append(rect.getMaxX());
				xml.append("\" y2=\"");
				xml.append(rect.getMaxY());
				xml.append("\" z2=\"");
				xml.append(top);
				xml.append("\"/>");
				return xml.toString();
			}
			case POLYGON:
			default: {
				StringBuilder xml = new StringBuilder();
				xml.append('<');
				xml.append(translator.translateTo("polygon"));
				xml.append(" id=\"");
				xml.append(getName());
				xml.append('"');
				for (int i = 0; i < poly.getNumPoints(); ++i) {
					xml.append(" x");
					xml.append(i + 1);
					xml.append("=\"");
					xml.append(poly.getX(i));
					xml.append("\" y");
					xml.append(i + 1);
					xml.append("=\"");
					xml.append(poly.getY(i));
					xml.append('"');
				}
				xml.append(" top=\"");
				xml.append(top);
				xml.append("\" bottom=\"");
				xml.append(bottom);
				xml.append("\"/>");
				return xml.toString();
			}
			}
		} catch (Exception e) {
			return "<CannotTranslate/>";
		}
	}

	public void unite(PolygonArea polygon) {
		this.poly = this.poly.union(polygon.poly);
	}

	@Override
	public void rescale(IRescaler rescaler) {
		for (int i = 0; i < poly.getNumInnerPoly(); ++i) {
			Poly inner = poly.getInnerPoly(i);
			for (int j = 0; j < poly.getInnerPoly(i).getNumPoints(); ++j) {
				inner.set(j, rescaler.rescaleX(inner.getX(j)),
					rescaler.rescaleY(inner.getY(j)));
			}
		}
		bottom = rescaler.rescaleZ(bottom);
		top = rescaler.rescaleZ(top);
		center = null;
	}
}
