package cz.cuni.amis.pogamut.edu.map.areas;

import java.awt.geom.Rectangle2D;
import java.util.Set;

import cz.cuni.amis.pogamut.base3d.worldview.objects.ILocated;
import cz.cuni.amis.pogamut.base3d.worldview.objects.Location;
import cz.cuni.amis.pogamut.edu.agent.Agent;
import cz.cuni.amis.pogamut.edu.drools.IFact;
import cz.cuni.amis.pogamut.edu.map.IMapDataUnit;
import cz.cuni.amis.pogamut.edu.utils.INamed;
import cz.cuni.amis.pogamut.edu.utils.IRescaler;

/**
 * Basic interface for all areas
 * 
 * @author Radim Vansa <radim.vansa@matfyz.cz>
 * 
 */
public interface IArea extends INamed, IFact, IMapDataUnit, ILocated {

	/**
	 * Registers, that the agent is currently in this area.
	 * 
	 * @param agent
	 * @param mark -
	 *            used for prevention of addAgent - removeAgent sequences, where
	 *            the removeAgent should be discarded.
	 */
	public void addAgent(Agent agent, long mark);

	/**
	 * Tests whether this area contains specified location
	 * 
	 * @param location
	 * @param ignoreZCoord -
	 *            should be vertical part of location's coordinate be taken
	 *            taken into account?
	 * @return <b>true</b> if it contains, <b>false</b> otherwise
	 */
	public boolean contains(Location location, boolean ignoreZCoord);

	/**
	 * Unregister the agent at all circumstances
	 * 
	 * @param agent -
	 *            removed agent
	 */
	public void forceRemoveAgent(Agent agent);

	/**
	 * @return List of all agents that are present in this area
	 */
	public Set<Agent> getAgents();

	/**
	 * @return smallest rectangle that contains whole area
	 */
	public Rectangle2D getBounds();

	/**
	 * Returns some point that will represent the position of this area. It is
	 * not necessary to be the gravity centre of the area, although it
	 * represents the area pretty well most times.
	 * 
	 * @return location of centre
	 */
	public Location getCenter();

	/**
	 * Sometimes you need polygon representing the whole area. It is not nice,
	 * but necessary (when computing paths from one complex area to another).
	 * This polygon SHOULDN'T be inserted into working memory - it's already
	 * there for polygons and for complex areas we have their rules. Remember
	 * that this method is very slow for complex areas and therefore the result
	 * should be cached.
	 * 
	 * @return the polygon
	 */
	public PolygonArea getPolygonArea();

	/**
	 * The implementation of this method could require the instantiation of the
	 * polygon and therefore be very slow.
	 * 
	 * @return surface in square units
	 */
	public double getSurface();

	/**
	 * Returns the area that is superior to this, e.g. as district is superior
	 * to street.
	 * 
	 * @return
	 */
	public IComplexArea getSuperior();

	/**
	 * 
	 * @param area
	 * @return <b>true</b> if there is an nonempty intersection of these two
	 *         areas or they border, <b>false</b> otherwise
	 */
	public boolean hasContact(IArea area);

	/**
	 * Unregisters the agent from this area
	 * 
	 * @param agent
	 * @param mark -
	 *            if smaller than current mark of this area the removeAgent
	 *            command shouldn't be performed
	 */
	public void removeAgent(Agent agent, long mark);

	/**
	 * The area has to call rescaler.rescale(location) to all it's Location
	 * variables, clear caches containing Location variables and so on. If this
	 * is a complex area, it should NOT call rescale for it's children areas -
	 * this would result in multiple rescaling
	 */
	public void rescale(IRescaler rescaler);

	/**
	 * Sets the area superior to this.
	 * 
	 * @param area
	 */
	public void setSuperior(IComplexArea area);
}
