package cz.cuni.amis.pogamut.edu.map.areas;

import java.awt.Rectangle;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;

import cz.cuni.amis.pogamut.base3d.worldview.objects.Location;
import cz.cuni.amis.pogamut.edu.drools.Rule;
import cz.cuni.amis.pogamut.edu.drools.RulesPackage;
import cz.cuni.amis.pogamut.edu.l10n.CannotTranslateException;
import cz.cuni.amis.pogamut.edu.l10n.ITranslator;
import cz.cuni.amis.pogamut.edu.map.MapData;
import cz.cuni.amis.pogamut.edu.map.exceptions.AreaException;
import cz.cuni.amis.pogamut.edu.utils.IRescaler;

/**
 * The area which connects other areas by union operation. There can be also one
 * area (this can be also complex, of course) that excludes itself from the
 * compound area - so the agent is in the compound area if he is in at least one
 * of the subareas and when he is not in the excluded area.
 * 
 * @author Radim Vansa <radim.vansa@matfyz.cz>
 * 
 */
public class CompoundArea extends AbstractComplexArea implements ICompoundArea {

	protected ICompoundArea excluded = null;
	protected PolygonArea polyCache = null;
	protected ArrayList<IArea> subareas = new ArrayList<IArea>();
	protected String type;

	public String getType() {
		return type;
	}

	public void setType(String type) {
		this.type = type;
	}

	public CompoundArea() {

	}

	public CompoundArea(String name) {
		super(name);
	}

	public CompoundArea(List<IArea> path) {
		for (IArea a : path) {
			add(a);
		}
	}

	public void add(IArea area) {
		if (!subareas.contains(area)) {
			subareas.add(area);
			this.polyCache = null;
		}
	}

	public boolean contains(Location l, boolean ignoreZCoord) {
		boolean contains = false;
		for (IArea area : subareas) {
			if (area.contains(l, ignoreZCoord)) {
				contains = true;
				break;
			}
		}
		if (excluded != null && excluded.contains(l, ignoreZCoord)) {
			contains = false;
		}
		return contains;
	}

	public boolean containsExactly(IArea a) {
		if (excluded != null && excluded.containsExactly(a)) {
			return false;
		}
		for (IArea area : subareas) {
			if (area == a) {
				return true;
			}
			if (area instanceof IComplexArea) {
				if (((IComplexArea) area).containsExactly(a)) {
					return true;
				}
			}
		}
		return false;
	}

	public void exclude(IArea area) {
		if (!subareas.contains(area)
				&& !(excluded != null && excluded.containsExactly(area))) {
			if (excluded == null) {
				excluded = new CompoundArea();
				excluded.setSuperior(this);
				excluded.setType("exclude");
			}
			excluded.add(area);
			this.polyCache = null;
		}
	}

	@Override
	public void generateRules(RulesPackage pkg) {
		if (!this.generatedRules.isEmpty()) {
			return;
		}
		this.pkg = pkg;
		for (IArea a : subareas) {
			Rule r1 = new Rule();
			r1.addAttribute("salience 1003");
			r1.addCondition("$a: cz.cuni.amis.pogamut.edu.agent.Agent()");
			r1
				.addCondition("cz.cuni.amis.pogamut.edu.map.areas.IArea(name == \""
						+ a.getName() + "\", agents contains $a)");
			r1
				.addCondition("$m: cz.cuni.amis.pogamut.edu.map.areas.CompoundArea(name == \""
						+ this.getName() + "\", agents not contains $a)");
			// r1.addConsequence("System.out.print(\"add at " + getName() + " \"
			// + $m.getAgents().size());");
			r1.addConsequence("$m.addAgent($a, scenario.getMark());");
			// r1.addConsequence("System.out.println(\"changed to \" +
			// $m.getAgents().size());");
			pkg.addRule(r1);
			generatedRules.add(r1);
			Rule r2 = new Rule();
			r2.addAttribute("salience 1002");
			r2.addCondition("$a: cz.cuni.amis.pogamut.edu.agent.Agent()");
			r2
				.addCondition("$m: cz.cuni.amis.pogamut.edu.map.areas.CompoundArea(name == \""
						+ this.getName() + "\", agents contains $a)");
			r2
				.addCondition("cz.cuni.amis.pogamut.edu.map.areas.IArea(name == \""
						+ a.getName() + "\", agents not contains $a)");
			// r2.addConsequence("System.out.print(\"remove at " + getName() +
			// "\" + $m.getAgents().size());");
			r2.addConsequence("$m.removeAgent($a, scenario.getMark());");
			// r2.addConsequence("System.out.println(\"changed to \" +
			// $m.getAgents().size());");
			pkg.addRule(r2);
			generatedRules.add(r2);
			if (a instanceof IComplexArea) {
				((IComplexArea) a).generateRules(pkg);
			}
		}
		if (excluded != null) {
			Rule r3 = new Rule();
			r3.addAttribute("salience 1001");
			r3.addCondition("$a: cz.cuni.amis.pogamut.edu.agent.Agent()");
			r3
				.addCondition("cz.cuni.amis.pogamut.edu.map.areas.IArea(name == \""
						+ this.excluded.getName() + "\", agents contains $a)");
			r3
				.addCondition("$m: cz.cuni.amis.pogamut.edu.map.areas.CompoundArea(name == \""
						+ this.getName() + "\", agents contains $a)");
			// r3.addConsequence("System.out.println(\"forceRemove at " +
			// r3.getName() + "\");");
			r3.addConsequence("$m.forceRemoveAgent($a);");
			pkg.addRule(r3);
			generatedRules.add(r3);
		}
		// DEBUG
		/*
		 * Rule debug = new Rule();
		 * debug.addCondition("cz.cuni.amis.pogamut.edu.map.areas.CompoundArea(name ==
		 * \"o100217\")"); debug.addConsequence("System.out.println(\"I am
		 * in\");"); pkg.addRule(debug); generatedRules.add(debug);
		 */
	}

	public Rectangle2D getBounds() {
		Rectangle2D rect = new Rectangle();
		for (IArea a : subareas) {
			rect.add(a.getBounds());
		}
		return rect;
	}

	/**
	 * Computes the central point as average of subareas points with excluded
	 * area taken as negated.
	 */
	@Override
	public Location getCenter() {
		Location l = new Location(0, 0, 0);
		for (IArea a : subareas) {
			l.add(a.getCenter());
		}
		if (excluded != null) {
			l.add(excluded.getCenter().scale(-1));
			return l.scale(1 / ((double) subareas.size() + 1));
		} else {
			return l.scale(1 / ((double) subareas.size()));
		}
	}

	public ICompoundArea getExcludedArea() {
		if (excluded == null) {
			excluded = new CompoundArea();
			excluded.setSuperior(this);
			excluded.setType("exclude");
		}
		return excluded;
	}

	@Override
	public PolygonArea getPolygonArea() {
		if (this.polyCache == null) {
			polyCache = new PolygonArea();
			for (IArea a : subareas) {
				polyCache.unite(a.getPolygonArea());
			}
			if (excluded != null) {
				polyCache.differentiate(excluded.getPolygonArea());
			}
		}
		return polyCache;
	}

	public List<IArea> getSubareas() {
		return Collections.unmodifiableList(subareas);
	}

	public void reinclude(IArea area) {
		excluded.remove(area);
		this.polyCache = null;
	}

	public void remove(IArea area) {
		if (subareas.remove(area)) {
			this.polyCache = null;
		}
	}

	public void setExcludedArea(ICompoundArea area) throws AreaException {
		if (excluded == null) {
			excluded = area;
			excluded.setType("exclude");
			this.polyCache = null;
		} else {
			throw new AreaException("Already has excluded area.");
		}
	}

	@Override
	public void writeXML(XMLStreamWriter writer, int indent,
			ITranslator translator) throws XMLStreamException {
		if (subareas.size() == 0 && excluded == null) {
			return;
		}
		MapData.writeXMLIndent(writer, indent);
		try {
			writer.writeStartElement(translator.translateTo(getType()));
		} catch (CannotTranslateException e) {
			writer.writeStartElement(getType());
		}
		writer.writeAttribute("id", getName());
		for (IArea area : subareas) {
			area.writeXML(writer, indent + 1, translator);
		}
		if (excluded != null) {
			excluded.writeXML(writer, indent + 1, translator);
		}
		MapData.writeXMLIndent(writer, indent);
		writer.writeEndElement();
	}

	@Override
	public void rescale(IRescaler rescaler) {
		polyCache = null;
	}

	@Override
	public boolean equals(Object o) {
		if (o instanceof CompoundArea) {
			CompoundArea ca = (CompoundArea) o;
			if (excluded == null) {
				if (ca.excluded != null) {
					return false;
				}
			} else {
				if (!excluded.equals(ca.excluded)) {
					return false;
				}
			}
			return subareas.containsAll(ca.subareas)
					&& ca.subareas.containsAll(subareas);
		} else {
			return false;
		}
	}
}
