package cz.cuni.amis.pogamut.edu.map;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;

import cz.cuni.amis.pogamut.edu.map.areas.IArea;
import cz.cuni.amis.pogamut.edu.map.areas.PolygonArea;
import cz.cuni.amis.pogamut.edu.utils.Pair;

/**
 * An evaluated graph with areas as vertices and their collisions as edges. The
 * value of an edge is the distance of areas' centres. 
 * 
 * @author Radim Vansa <radim.vansa@matfyz.cz>
 * 
 */
public class MapGraph {

	/**
	 * The vertex with a list of edges.
	 * 
	 * @author Radim Vansa <radim.vansa@matfyz.cz>
	 *
	 */
	protected class Node {
		protected List<Pair<Node, Double>> edges = new ArrayList<Pair<Node, Double>>();
		protected PolygonArea polygon;
		protected boolean temporary = false;
		protected double distFromStart = Double.POSITIVE_INFINITY;

		public Node(IArea area) {
			this.polygon = area.getPolygonArea();
		}

		private void addEdge(Node node, double distance) {
			edges.add(new Pair<Node, Double>(node, new Double(distance)));
		}

		public void connect() {
			List<IArea> areas = map.getQuadTree().getAreasOn(
				this.polygon.getBounds());
			for (IArea a : areas) {
				if (a != this.polygon) {
					if (a.hasContact(this.polygon)) {
						Node node = areaToNode.get(a);
						double distance = a.getCenter().getDistance(
							this.polygon.getCenter());
						node.addEdge(this, distance);
						this.addEdge(node, distance);
					}
				}
			}
		}

		public void disconnect() {
			for (Pair<Node, Double> p : this.edges) {
				p.getFirst().removeEdge(this);
			}
			this.edges.clear();
		}

		public boolean isTemporary() {
			return temporary;
		}

		private void removeEdge(Node node) {
			this.edges.remove(node);
		}

		public void setTemporary(boolean temporary) {
			this.temporary = temporary;
		}

		public double getDistFromStart() {
			return distFromStart;
		}

		public void setDistFromStart(double distFromStart) {
			this.distFromStart = distFromStart;
		}

	}

	protected HashMap<IArea, Node> areaToNode = new HashMap<IArea, Node>();
	protected Map map;

	public MapGraph(Map map) {
		this.map = map;
	}

	/**
	 * Constructs the graph from a list of areas.
	 * @param areas
	 */
	public void construct(List<IArea> areas) {
		for (IArea area : areas) {
			Node node = new Node(area);
			areaToNode.put(area, node);
		}
		for (Node node : areaToNode.values()) {
			node.connect();
		}
	}

	/**
	 * Searches for the shortest path between two areas using Dijkstra's algorithm.
	 * The path is returned as a list of areas including the area1 on beginning and
	 * area2 on the end. The areas can be run-time generated, they do not have to be
	 * included in the graph.
	 * 
	 * @param area1
	 * @param area2
	 * @return
	 */
	public List<IArea> findPath(IArea area1, IArea area2) {
		// try if the areas are already in
		Node node1 = areaToNode.get(area1);
		Node node2 = areaToNode.get(area2);
		if (node1 == null) {
			node1 = new Node(area1);
			node1.setTemporary(true);
			node1.connect();
		}
		if (node2 == null) {
			node2 = new Node(area2);
			node2.setTemporary(true);
			node2.connect();
		}
		LinkedList<IArea> path = findPath(node1, node2);
		if (node1.isTemporary()) {
			node1.disconnect();
			path.removeFirst();
		}
		if (node2.isTemporary()) {
			node2.disconnect();
			path.removeLast();
		}
		return path;
	}

	/**
	 * See findPath(IArea, IArea), this is internal implementation.
	 * 
	 * @param start
	 * @param target
	 * @return
	 */
	protected LinkedList<IArea> findPath(Node start, Node target) {
		List<Node> affectedDistances = new LinkedList<Node>();
		List<Node> nodesToProcess = new LinkedList<Node>();
		Comparator<Node> comp = new Comparator<Node>() {
			@Override
			public int compare(Node n1, Node n2) {
				if (n1.getDistFromStart() > n2.getDistFromStart()) {
					return 1;
				} else if (n1.getDistFromStart() < n2.getDistFromStart()) {
					return -1;
				} else {
					return 0;
				}
			}
		};
		// Dijkstra's algorithm
		nodesToProcess.add(start);
		start.setDistFromStart(0);
		affectedDistances.add(start);
		while (!nodesToProcess.isEmpty()) {
			// TODO: optimise using heap
			Node node = Collections.min(nodesToProcess, comp);
			nodesToProcess.remove(node);
			if (node == target) {
				break;
			}
			for (Pair<Node, Double> edge : node.edges) {
				double dist = node.getDistFromStart()
						+ edge.getSecond().doubleValue();
				if (edge.getFirst().getDistFromStart() == Double.POSITIVE_INFINITY) {
					// add not processed ones to the queue
					affectedDistances.add(edge.getFirst());
					nodesToProcess.add(edge.getFirst());
				}
				if (dist < edge.getFirst().getDistFromStart()) {
					edge.getFirst().setDistFromStart(dist);
				}
			}
		}
		// tracking the path back
		/*
		 * Comparator<Pair<Node, Double>> comp2 = new Comparator<Pair<Node,
		 * Double>>() { @Override public int compare(Pair<Node, Double> p1,
		 * Pair<Node, Double> p2) { if (p1.getFirst().getDistFromStart() >
		 * p2.getFirst() .getDistFromStart()) { return 1; } else if
		 * (p1.getFirst().getDistFromStart() < p2.getFirst()
		 * .getDistFromStart()) { return -1; } else { return 0; } } };
		 */
		LinkedList<IArea> path = new LinkedList<IArea>();
		Node node = target;
		while (node != start) {
			path.addFirst(node.polygon);
			for (Pair<Node, Double> edge : node.edges) {
				// the *1.001 is for rounding errors
				if (edge.getFirst().getDistFromStart() + edge.getSecond() <= node
																					.getDistFromStart() * 1.001) {
					node = edge.getFirst();
					break;
				}
			}
			// node = Collections.min(node.edges, comp2).getFirst();
		}
		path.addFirst(start.polygon);
		// cleanup
		for (Node aff : affectedDistances) {
			aff.setDistFromStart(Double.POSITIVE_INFINITY);
		}
		return path;
	}
}
