package cz.cuni.amis.pogamut.edu.map;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Stack;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.stream.FactoryConfigurationError;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

import cz.cuni.amis.pogamut.base3d.worldview.objects.Location;
import cz.cuni.amis.pogamut.edu.l10n.CannotTranslateException;
import cz.cuni.amis.pogamut.edu.l10n.ITranslator;
import cz.cuni.amis.pogamut.edu.map.areas.CompoundArea;
import cz.cuni.amis.pogamut.edu.map.areas.IArea;
import cz.cuni.amis.pogamut.edu.map.areas.IComplexArea;
import cz.cuni.amis.pogamut.edu.map.areas.ICompoundArea;
import cz.cuni.amis.pogamut.edu.map.areas.IIntersectArea;
import cz.cuni.amis.pogamut.edu.map.areas.IntersectArea;
import cz.cuni.amis.pogamut.edu.map.areas.PolygonArea;
import cz.cuni.amis.pogamut.edu.map.exceptions.AreaException;
import cz.cuni.amis.pogamut.edu.map.exceptions.MapException;
import cz.cuni.amis.pogamut.edu.map.marks.IMark;
import cz.cuni.amis.pogamut.edu.map.marks.POI;
import cz.cuni.amis.pogamut.edu.map.marks.Path;
import cz.cuni.amis.pogamut.edu.map.marks.PathPoint;
import cz.cuni.amis.pogamut.edu.utils.ColorTable;
import cz.cuni.amis.pogamut.edu.utils.IRescaler;

/**
 * Loads and saves the map file, holds information about the plan and
 * all areas and marks. 
 * 
 * @author Radim Vansa <radim.vansa@matfyz.cz>
 *
 */
public class MapData {

	/**
	 * Parses the map file and stores information from it to the MapData object.
	 * 
	 * @author Radim Vansa <radim.vansa@matfyz.cz>
	 *
	 */
	private class MapDataParser extends DefaultHandler {

		private Stack<IMapDataUnit> stack = new Stack<IMapDataUnit>();
		private ITranslator translator;

		public MapDataParser(ITranslator translator) {
			this.translator = translator;
			stack.push(rootArea);
		}

		private void addCircle(Attributes attributes) throws SAXException {
			String id = attributes.getValue("id");
			IArea area = areas.get(id);
			if (area == null) {
				if (attributes.getIndex("cx") == -1
						|| attributes.getIndex("cy") == -1
						|| attributes.getIndex("radius") == -1) {
					throw new SAXException(
							"Circle needs attributes cx, cy and radius.");
				}
				PolygonArea pa = new PolygonArea();
				pa.generateCircle(
					Double.parseDouble(attributes.getValue("cx")),
					Double.parseDouble(attributes.getValue("cy")),
					Double.parseDouble(attributes.getValue("radius")));
				if (id != null) {
					pa.setName(id);
				}
				areas.put(pa.getName(), pa);
				simpleAreas.add(pa);
				area = pa;
			} else {
				// if (!(area instanceof PolygonArea)
				// || ((PolygonArea) area).getShapeHint() !=
				// PolygonArea.Shape.CIRCLE) {
				throw new SAXException("Two conflicting ids: " + id);
				// }
			}
			addToElementOnTop(area);
			stack.push(area);
		}

		private void addCompound(String type, Attributes attributes)
				throws SAXException {
			String id = attributes.getValue("id");
			IArea area = areas.get(id);
			if (area == null) {
				ICompoundArea ca = new CompoundArea();
				ca.setType(type);
				if (id != null) {
					ca.setName(id);
				}
				areas.put(ca.getName(), ca);
				complexAreas.add(ca);
				area = ca;
			} else {
				// if (area instanceof ICompoundArea) {
				// if (!((ICompoundArea) area).getType()
				// .equalsIgnoreCase(type)) {
				// throw new SAXException("Two conflicting ids: " + id);
				// }
				// } else {
				throw new SAXException("Two conflicting ids: " + id);
				// }
			}
			addToElementOnTop(area);
			stack.push(area);
		}

		private void addExclude(Attributes attributes) throws SAXException {
			ICompoundArea ca;
			if (stack.peek() instanceof ICompoundArea) {
				ICompoundArea top = (ICompoundArea) stack.peek();
				ca = top.getExcludedArea();
				if (ca == null) {
					ca = new CompoundArea();
					ca.setType("exclude");
					try {
						top.setExcludedArea(ca);
					} catch (AreaException e) {
						ca = top.getExcludedArea();
						stack.push(ca);
						return;
					}
				}
			} else {
				throw new SAXException("Cannot exclude anything from element "
						+ stack.peek().toString());
			}
			areas.put(ca.getName(), ca);
			stack.push(ca);
		}

		private void addIntersection(Attributes attributes) throws SAXException {
			String id = attributes.getValue("id");
			IArea area = areas.get(id);
			if (area == null) {
				IIntersectArea ia = new IntersectArea();
				if (id != null) {
					ia.setName(id);
				}
				areas.put(ia.getName(), ia);
				complexAreas.add(ia);
				area = ia;
			} else {
				// if (!(area instanceof IIntersectArea)) {
				throw new SAXException("Two conflicting ids: " + id);
				// }
			}
			addToElementOnTop(area);
			stack.push(area);
		}

		private void addPath(Attributes attributes) throws SAXException {
			Path path = new Path();
			if (attributes.getIndex("id") != -1) {
				path.setName(attributes.getValue("id"));
			}
			String color = attributes.getValue("color");
			if (color != null) {
				try {
					color = translator.translateFrom(color);
				} catch (CannotTranslateException e) {
				}
				path.setColor(ColorTable.getColorFor(color));
			}
			String visible = attributes.getValue("visible");
			if (visible != null) {
				try {
					visible = translator.translateFrom(visible);
				} catch (CannotTranslateException e) {
					throw new SAXException("Cannot translate \"" + visible
							+ "\"!");
				}
				path.setVisible(Boolean.parseBoolean(visible));
			}
			String width = attributes.getValue("width");
			if (width != null) {
				path.setWidth(Double.parseDouble(width));
			}
			stack.push(path);
			marks.add(path);
			markMap.put(path.getName(), path);
		}

		private void addPathPoint(Attributes attributes) throws SAXException {
			if (attributes.getIndex("x") == -1
					|| attributes.getIndex("y") == -1) {
				throw new SAXException("PathPoint needs attributes x, y");
			}
			PathPoint point = new PathPoint();
			if (attributes.getIndex("id") != -1) {
				point.setName(attributes.getValue("id"));
			}
			double x = Double.parseDouble(attributes.getValue("x"));
			double y = Double.parseDouble(attributes.getValue("y"));
			double z = 0;
			if (attributes.getIndex("z") != -1) {
				z = Double.parseDouble(attributes.getValue("z"));
			}
			point.setLocation(new Location(x, y, z));
			addToElementOnTop(point);
			stack.push(point);
		}

		private void addPOI(Attributes attributes) throws SAXException {
			if (attributes.getIndex("x") == -1
					|| attributes.getIndex("y") == -1) {
				throw new SAXException("POI needs attributes x, y");
			}
			POI poi = new POI();
			if (attributes.getIndex("id") != -1) {
				poi.setName(attributes.getValue("id"));
			}
			double x = Double.parseDouble(attributes.getValue("x"));
			double y = Double.parseDouble(attributes.getValue("y"));
			double z = 0;
			if (attributes.getIndex("z") != -1) {
				z = Double.parseDouble(attributes.getValue("z"));
			}
			poi.setLocation(new Location(x, y, z));
			poi.setDescription(attributes.getValue("description"));
			poi.setSymbol(POI.Symbol.valueOf(attributes.getValue("symbol").toUpperCase()));
			String color = attributes.getValue("color");
			if (color != null) {
				try {
					color = translator.translateFrom(color);
				} catch (CannotTranslateException e) {
				}
				poi.setColor(ColorTable.getColorFor(color));
			}
			String visible = attributes.getValue("visible");
			if (visible != null) {
				try {
					visible = translator.translateFrom(visible);
				} catch (CannotTranslateException e) {
					throw new SAXException("Cannot translate \"" + visible
							+ "\"!");
				}
				poi.setVisible(Boolean.parseBoolean(visible));
			}
			stack.push(poi);
			marks.add(poi);
			markMap.put(poi.getName(), poi);
		}

		private void addPolygon(Attributes attributes) throws SAXException {
			String id = attributes.getValue("id");
			IArea area = areas.get(id);
			if (area == null) {
				if (attributes.getIndex("x1") == -1
						|| attributes.getIndex("y1") == -1
						|| attributes.getIndex("x2") == -1
						|| attributes.getIndex("y2") == -1
						|| attributes.getIndex("x3") == -1
						|| attributes.getIndex("y3") == -1) {
					throw new SAXException("Polygon needs at least 3 vertices!");
				}
				PolygonArea pa = new PolygonArea();
				pa.setShapeHint(PolygonArea.Shape.POLYGON);
				int i = 1;
				String x, y;
				while (true) {
					x = attributes.getValue("x" + i);
					y = attributes.getValue("y" + i);
					if (x == null || y == null) {
						break;
					}
					pa.addPoint(new Location(Double.parseDouble(x),
							Double.parseDouble(y)));
					++i;
				}
				if (attributes.getIndex("bottom") != -1
						&& attributes.getIndex("top") != -1) {
					pa.setZLimits(
						Double.parseDouble(attributes.getValue("bottom")),
						Double.parseDouble(attributes.getValue("top")));
				}
				if (id != null) {
					pa.setName(id);
				}
				areas.put(pa.getName(), pa);
				simpleAreas.add(pa);
				area = pa;
			} else {
				// if ((area instanceof PolygonArea)
				// || ((PolygonArea) area).getShapeHint() !=
				// PolygonArea.Shape.POLYGON) {
				throw new SAXException("Two conflicting ids: " + id);
				// }
			}
			addToElementOnTop(area);
			stack.push(area);
		}

		private void addRect(Attributes attributes) throws SAXException {
			String id = attributes.getValue("id");
			IArea area = areas.get(id);
			if (area == null) {
				if (attributes.getIndex("x1") == -1
						|| attributes.getIndex("y1") == -1
						|| attributes.getIndex("x2") == -1
						|| attributes.getIndex("y2") == -1) {
					throw new SAXException(
							"Rect needs attributes x1, y1, x2 and y2.");
				}
				PolygonArea pa = new PolygonArea();
				pa.setShapeHint(PolygonArea.Shape.RECT);
				double x1 = Double.parseDouble(attributes.getValue("x1"));
				double y1 = Double.parseDouble(attributes.getValue("y1"));
				double x2 = Double.parseDouble(attributes.getValue("x2"));
				double y2 = Double.parseDouble(attributes.getValue("y2"));
				pa.addPoint(new Location(x1, y1));
				pa.addPoint(new Location(x1, y2));
				pa.addPoint(new Location(x2, y2));
				pa.addPoint(new Location(x2, y1));
				if (id != null) {
					pa.setName(id);
				}
				areas.put(pa.getName(), pa);
				simpleAreas.add(pa);
				area = pa;
			} else {
				// if ((area instanceof PolygonArea)
				// || ((PolygonArea) area).getShapeHint() !=
				// PolygonArea.Shape.RECT) {
				throw new SAXException("Two conflicting ids: " + id);
				// }
			}
			addToElementOnTop(area);
			stack.push(area);
		}

		private void addSpace(Attributes attributes) throws SAXException {
			String id = attributes.getValue("id");
			IArea area = areas.get(id);
			if (area == null) {
				if (attributes.getIndex("x1") == -1
						|| attributes.getIndex("y1") == -1
						|| attributes.getIndex("z1") == -1
						|| attributes.getIndex("x2") == -1
						|| attributes.getIndex("y2") == -1
						|| attributes.getIndex("z2") == -1) {
					throw new SAXException(
							"Space needs attributes x1, y1, z1, x2, y2 and z2.");
				}
				PolygonArea pa = new PolygonArea();
				pa.setShapeHint(PolygonArea.Shape.SPACE);
				double x1 = Double.parseDouble(attributes.getValue("x1"));
				double y1 = Double.parseDouble(attributes.getValue("y1"));
				double x2 = Double.parseDouble(attributes.getValue("x2"));
				double y2 = Double.parseDouble(attributes.getValue("y2"));
				pa.addPoint(new Location(x1, y1));
				pa.addPoint(new Location(x1, y2));
				pa.addPoint(new Location(x2, y2));
				pa.addPoint(new Location(x2, y1));
				pa.setZLimits(Double.parseDouble(attributes.getValue("z1")),
					Double.parseDouble(attributes.getValue("z2")));
				if (id != null) {
					pa.setName(id);
				}
				areas.put(pa.getName(), pa);
				simpleAreas.add(pa);
				area = pa;
			} else {
				// if ((area instanceof PolygonArea)
				// || ((PolygonArea) area).getShapeHint() !=
				// PolygonArea.Shape.SPACE) {
				throw new SAXException("Two conflicting ids: " + id);
				// }
			}
			addToElementOnTop(area);
			stack.push(area);
		}

		protected void addToElementOnTop(IMapDataUnit data) throws SAXException {
			IMapDataUnit top = stack.peek();
			if (top instanceof IComplexArea) {
				IComplexArea ca = (IComplexArea) top;
				if (data instanceof IArea) {
					IArea area = (IArea) data;
					ca.add(area);
					area.setSuperior(ca);
				} else {
					throw new SAXException("Element " + top.toString()
							+ " doesn't support children elements of type"
							+ data.getClass().getName());
				}
			} else if (top instanceof Path) {
				Path path = (Path) top;
				if (data instanceof PathPoint) {
					path.addPoint((PathPoint) data);
				} else {
					throw new SAXException("Element " + top.toString()
							+ " doesn't support children elements of type"
							+ data.getClass().getName());
				}
			} else {
				throw new SAXException("Element " + top.toString()
						+ " doesn't support children elements");
			}
		}

		@Override
		public void endElement(String uri, String localName, String qName)
				throws SAXException {
			stack.pop();
		}

		@Override
		public void startElement(String uri, String localName, String qName,
				Attributes attributes) throws SAXException {
			String tag;
			try {
				tag = translator.translateFrom(qName);
			} catch (CannotTranslateException e) {
				tag = qName;
			}
			if (tag.equalsIgnoreCase("rect")) {
				addRect(attributes);
			} else if (tag.equalsIgnoreCase("space")) {
				addSpace(attributes);
			} else if (tag.equalsIgnoreCase("circle")) {
				addCircle(attributes);
			} else if (tag.equalsIgnoreCase("exclude")) {
				addExclude(attributes);
			} else if (tag.equalsIgnoreCase("intersection")) {
				addIntersection(attributes);
			} else if (tag.equalsIgnoreCase("polygon")) {
				addPolygon(attributes);
			} else if (tag.equalsIgnoreCase("POI")) {
				addPOI(attributes);
			} else if (tag.equalsIgnoreCase("path")) {
				addPath(attributes);
			} else if (tag.equalsIgnoreCase("point")) {
				addPathPoint(attributes);
			} else if (tag.equalsIgnoreCase("map")) {

			} else if (tag.equalsIgnoreCase("plan")) {
				if (plan == null || plan.isEmpty()) {
					plan = attributes.getValue("src");
					unrealId = attributes.getValue("unrealId");
					scale = Double.valueOf(attributes.getValue("scale"));
					originX = Double.valueOf(attributes.getValue("originX"));
					originY = Double.valueOf(attributes.getValue("originY"));
				} else {
					//ignore it
				}
				// in endElement() we do pop()
				stack.push(null);
			} else {
				addCompound(tag, attributes);
			}
		}
	}

	/**
	 * Writes the indentation
	 */
	public static void writeXMLIndent(XMLStreamWriter writer, int indent)
			throws XMLStreamException {
		String indentString = "\n";
		for (int i = 0; i < indent; ++i) {
			indentString += "    ";
		}
		writer.writeCharacters(indentString);
	}

	protected Map<String, IArea> areas = new HashMap<String, IArea>();
	protected List<IComplexArea> complexAreas = new ArrayList<IComplexArea>();
	protected List<IMark> marks = new ArrayList<IMark>();
	protected Map<String, IMark> markMap = new HashMap<String, IMark>();
	protected double originX = 0;
	protected double originY = 0;

	protected String plan = "";
	protected String unrealId ="";

	protected ICompoundArea rootArea = new CompoundArea();
	protected double scale = 1;

	protected List<IArea> simpleAreas = new ArrayList<IArea>();

	public void add(IArea area) {
		areas.put(area.getName(), area);
		if (area instanceof IComplexArea) {
			complexAreas.add((IComplexArea) area);
		} else {
			simpleAreas.add(area);
		}
		if (area.getSuperior() == null) {
			rootArea.add(area);
			area.setSuperior(rootArea);
		}
	}

	public Map<String, IArea> getAreas() {
		return Collections.unmodifiableMap(areas);
	}

	public List<IComplexArea> getComplexAreas() {
		return Collections.unmodifiableList(complexAreas);
	}

	public IMark getMark(String name) {
		return markMap.get(name);
	}
	
	public double getOriginX() {
		return originX;
	}

	public double getOriginY() {
		return originY;
	}

	public String getPlan() {
		return plan;
	}

	public ICompoundArea getRootArea() {
		return rootArea;
	}

	public double getScale() {
		return scale;
	}

	public List<IArea> getSimpleAreas() {
		return Collections.unmodifiableList(simpleAreas);
	}

	public void parseSource(InputStream source, ITranslator translator)
			throws ParserConfigurationException, SAXException, IOException {
		SAXParserFactory spf = SAXParserFactory.newInstance();
		SAXParser sp = spf.newSAXParser();
		sp.parse(source, new MapDataParser(translator));
	}

	public void parseSource(String uri, ITranslator translator)
			throws ParserConfigurationException, SAXException, IOException {
		SAXParserFactory spf = SAXParserFactory.newInstance();
		SAXParser sp = spf.newSAXParser();
		sp.parse(uri, new MapDataParser(translator));
	}

	public void remove(IArea area) {
		areas.remove(area.getName());
		if (area instanceof IComplexArea) {
			complexAreas.remove((IComplexArea) area);
		} else {
			simpleAreas.remove(area);
		}
		// superior shouldn't be null!
		if (area.getSuperior() == null) {
			throw new IllegalArgumentException("Area with no superior area!");
		}
		((IComplexArea) area.getSuperior()).remove(area);
	}

	public void storeData(OutputStream stream, ITranslator translator)
			throws MapException {
		XMLStreamWriter writer;
		try {
			writer = XMLOutputFactory.newInstance().createXMLStreamWriter(
				stream);
			writer.writeStartDocument();
			writer.writeCharacters("\n");
			writer.writeStartElement("map");
			writer.writeCharacters("\n    ");
			try {
				writer.writeStartElement(translator.translateTo("plan"));
			} catch (CannotTranslateException e) {
				writer.writeStartElement("plan");
			}
			writer.writeAttribute("src", plan);
			writer.writeAttribute("unrealId", unrealId);
			writer.writeAttribute("scale", String.valueOf(scale));
			writer.writeAttribute("originX", String.valueOf(originX));
			writer.writeAttribute("originY", String.valueOf(originY));
			writer.writeEndElement();
			writer.writeCharacters("\n");
			for (IArea area : rootArea.getSubareas()) {
				area.writeXML(writer, 1, translator);
			}
			for (IMark mark : marks) {
				mark.writeXML(writer, 1, translator);
			}
			writer.writeCharacters("\n");
			writer.writeEndElement();
			writer.writeEndDocument();
			writer.close();
		} catch (XMLStreamException e) {
			throw new MapException("Cannot write data.", e);
		} catch (FactoryConfigurationError e) {
			throw new MapException("Cannot write data.", e);
		}
	}

	public List<IMark> getMarks() {
		return Collections.unmodifiableList(marks);
	}

	public String getPlanUnrealId() {
		return unrealId;
	}
	
	public void setPlanUnrealId(String unrealId) {
		this.unrealId = unrealId;
	}

	public void setPlan(String src) {
		plan = src;		
	}

	public void rescale(IRescaler rescaler) {
		for (IArea a: areas.values()) {
			a.rescale(rescaler);
		}
	}

	public void setOriginX(double originX) {
		this.originX = originX;
	}

	public void setOriginY(double originY) {
		this.originY = originY;
	}

	public void setScale(double scale) {
		this.scale = scale;
	}

	public void addMark(IMark mark) {
		marks.add(mark);
		markMap.put(mark.getName(), mark);
	}

	public void removeMark(IMark mark) {
		marks.remove(mark);
		markMap.remove(mark.getName());
	}

	public void updateArea(IArea area) {
		areas.remove(area.getPreviousName());
		areas.put(area.getName(), area);
	}
	
	public void updateMark(IMark mark) {
		markMap.remove(mark.getPreviousName());
		markMap.put(mark.getName(), mark);
	}
}
