package cz.cuni.amis.pogamut.edu.map;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import cz.cuni.amis.pogamut.base3d.worldview.objects.Location;
import cz.cuni.amis.pogamut.edu.Scenario;
import cz.cuni.amis.pogamut.edu.agent.Agent;
import cz.cuni.amis.pogamut.edu.drools.RulesPackage;
import cz.cuni.amis.pogamut.edu.map.areas.CompoundArea;
import cz.cuni.amis.pogamut.edu.map.areas.IArea;
import cz.cuni.amis.pogamut.edu.map.areas.IComplexArea;
import cz.cuni.amis.pogamut.edu.map.areas.IIntersectArea;
import cz.cuni.amis.pogamut.edu.map.areas.IntersectArea;
import cz.cuni.amis.pogamut.edu.map.areas.ProvisoryProxy;
import cz.cuni.amis.pogamut.edu.map.exceptions.MapException;
import cz.cuni.amis.pogamut.edu.map.marks.IMark;
import cz.cuni.amis.pogamut.edu.utils.IRescaler;

/**
 * The main Map object. It contains MapData, MapGraph and AreaQuadTree.
 * Several methods used in DSL are written here.
 * 
 * @author Radim Vansa <radim.vansa@matfyz.cz>
 *
 */
public class Map {

	protected AreaQuadTree areaQuadTree = new AreaQuadTree();
	protected MapData data;
	protected MapGraph graph = new MapGraph(this);
	protected Scenario scenario = null;
	
	public Map(Scenario scenario) {
		this.scenario = scenario;
		this.data = new MapData();
	}

	/**
	 * Adds one area to another complex area registering the new rules for
	 * the complex one.
	 * @param area
	 * @param to
	 */
	public void addAreaTo(IArea area, IComplexArea to) {
		if (area == null) {
			throw new IllegalArgumentException("Cannot add null.");
		}
		if (to == null) {
			throw new IllegalArgumentException("Cannot add to null.");
		}
		to.add(area);
		to.removeRules();
		registerArea(to);
	}

	/**
	 * Loads MapData from stream.
	 * 
	 * @param stream
	 * @throws MapException
	 */
	public void addData(InputStream stream) throws MapException {
		try {
			data.parseSource(stream, scenario.getTranslator());
		} catch (Exception e) {
			throw new MapException("Unable to load data:\n"
					+ e.getLocalizedMessage());
		}
	}

	/**
	 * Loads MapData from file.
	 * @param uri
	 * @throws MapException
	 */
	public void addData(String uri) throws MapException {
		try {
			data.parseSource(uri, scenario.getTranslator());
		} catch (Exception e) {
			throw new MapException("Unable to load data:\n"
					+ e.getLocalizedMessage());
		}
	}

	/**
	 * Creates a new alias for specified area.
	 * 
	 * @param name
	 * @param area
	 * @return
	 */
	public IArea createAlias(String name, IArea area) {
		ProvisoryProxy alias = new ProvisoryProxy(area, scenario);
		alias.setName(name);
		registerArea(alias);
		return alias;
	}

	/**
	 * Creates a new provisory proxy which is intersection of area and the shortest path
	 * between from and to. 
	 * 
	 * @param name
	 * @param area
	 * @param from
	 * @param to
	 * @return
	 */
	public IArea createProvisoryArea(String name, IArea area, IArea from, IArea to) {
		IIntersectArea ia = new IntersectArea();
		ia.insertTo(scenario.getWorkingMemory());
		ia.add(area);
		List<IArea> path = graph.findPath(from, to);
		CompoundArea ca = new CompoundArea(path);
		ca.insertTo(scenario.getWorkingMemory());
		ia.add(ca);
		ProvisoryProxy proxy = new ProvisoryProxy(ia, scenario);
		proxy.setName(name);
		proxy.addMaintained(ia);
		proxy.addMaintained(ca);
		registerArea(proxy);
		return proxy;
	}
	
	/**
	 * Creates a new provisory proxy with empty compound area.
	 * 
	 * @param name
	 * @return
	 */
	public IArea createProvisoryCompoundArea(String name) {
		CompoundArea ca = new CompoundArea();
		ca.insertTo(scenario.getWorkingMemory());
		ProvisoryProxy proxy = new ProvisoryProxy(ca, scenario);
		proxy.setName(name);
		proxy.addMaintained(ca);
		// no register - currently empty
		return proxy;
	}

	/**
	 * Creates a provisory proxy with intersection of the two areas.
	 * @param name
	 * @param a1
	 * @param a2
	 * @return
	 */
	public IArea createProvisoryCrossing(String name, IArea a1, IArea a2) {
		IntersectArea ia = new IntersectArea(a1, a2);
		ia.insertTo(scenario.getWorkingMemory());
		ProvisoryProxy proxy = new ProvisoryProxy(ia, scenario);
		proxy.setName(name);
		proxy.addMaintained(ia);
		registerArea(proxy);
		return proxy;
	}

	/**
	 * Creates the AreaQuadTree and MapGraph after the data are loaded from file.
	 */
	public void fillStructures() {
		for (IArea a : data.getSimpleAreas()) {
			areaQuadTree.insert(a);
		}
		graph.construct(data.getSimpleAreas());
	}

	/**
	 * Starts the rules generation for areas loaded from the map file.
	 * 
	 * @param pkg
	 */
	public void generateRules(RulesPackage pkg) {
		pkg.addImport("cz.cuni.amis.pogamut.edu.agent.Agent");
		pkg.addImport("cz.cuni.amis.pogamut.edu.map.areas.CompoundArea");
		pkg.addImport("cz.cuni.amis.pogamut.edu.map.areas.IArea");
		pkg.addImport("cz.cuni.amis.pogamut.edu.map.areas.IntersectArea");
		data.getRootArea().generateRules(pkg);
	}

	public IArea getArea(String name) {
		return data.getAreas().get(name);
	}

	public MapData getData() {
		return data;
	}

	/**
	 * Sometimes the we need some area we are standing on - this picks up the
	 * smallest simple area (there could be more of them if the map is not
	 * constructed very well)
	 * 
	 * @param location
	 * @return
	 */
	public IArea getLeastArea(Location location) {
		IArea smallest = null;
		double min = Double.MAX_VALUE;
		for (IArea area : areaQuadTree.getAreasPossibleOn(location.getX(),
			location.getY())) {
			if (!(area instanceof IComplexArea)
					&& area.contains(location, false)) {
				double surface = area.getSurface();
				if (surface < min) {
					min = surface;
					smallest = area;
				}
			}
		}
		return smallest;
	}

	public IMark getMark(String name) {
		return data.getMark(name);
	}

	public AreaQuadTree getQuadTree() {
		return this.areaQuadTree;
	}

	/**
	 * Generates and registers area's rules.
	 * @param area
	 */
	public void registerArea(IComplexArea area) {
		RulesPackage pkg = scenario.getRuntimeRulesPackage();
		area.generateRules(pkg);
		scenario.registerRulesPackage(pkg);
	}

	/**
	 * Rescales all areas. Editor uses this method.
	 * @param rescaler
	 */
	public void rescale(IRescaler rescaler) {
		areaQuadTree = new AreaQuadTree();
		graph = new MapGraph(this);
		data.rescale(rescaler);
		fillStructures();
	}

	/**
	 * Scenario calls this to store the marks and areas in variables.
	 */
	public void registerMapDataUnitsAsVariables() {
		for (IArea a : data.getAreas().values()) {
			scenario.setVariable(a.getName(), a);
		}
		for (IMark m : data.getMarks()) {
			scenario.setVariable(m.getName(), m);
		}
	}

	/**
	 * Destroys rules dynamically created when registering the area. The method
	 * accepts any object to avoid instanceof checks in DSL.
	 * 
	 * @param area
	 */
	public void unregisterArea(Object area) {
		if (area == null) {
			return;
		}
		if (area instanceof IComplexArea) {
			// test if it's really only occasional
			if (getArea(((IComplexArea) area).getName()) == null) {
				((IComplexArea) area).removeRules();
			}
		}

	}

	/**
	 * Called by ControlServer, updates players' positions.
	 * 
	 * @param agent
	 * @param mark
	 */
	public void updateSimpleAreas(Agent agent, long mark) {
		List<IArea> suspicious = this.areaQuadTree.getAreasPossibleOn(
			agent.getLocation().getX(), agent.getLocation().getY());
		ArrayList<IArea> containing = new ArrayList<IArea>();
		for (IArea a : suspicious) {
			if (a.contains(agent.getLocation(), false)) {
				a.addAgent(agent, mark);
				containing.add(a);
			}
		}

		for (IArea a : agent.getAreas()) {
			a.removeAgent(agent, mark);
		}
		agent.setAreas(containing);

		// DEBUG:
		/*
		 * System.out.println("DEBUG"); for (IArea a:
		 * this.getData().getSimpleAreas()) { if (!a.getAgents().isEmpty() ) {
		 * System.out.println("simple " + a.getName()); } } for (IArea a:
		 * this.getData().getComplexAreas()) { if (!a.getAgents().isEmpty() ) {
		 * System.out.println("complex " + a.getName()); } }
		 */
	}
}
