package cz.cuni.amis.pogamut.edu.map;

import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import cz.cuni.amis.pogamut.base3d.worldview.objects.Location;
import cz.cuni.amis.pogamut.edu.map.areas.IArea;

/**
 * AreaQuadTree preprocesses the map and allows fast search for areas on
 * specific position. It starts with bounding square of the first inserted area.
 * If another area is inserted it computes whether it fits into current largest
 * square and if it does not it creates a new, larger one to satisfy new area's
 * needs. The information about new area is spread into sub-squares where the
 * area could range. If the number of areas in some square is larger than the
 * depth from top to this square (depth in tree) and the square has no children
 * it divides into four sub-squares hoping that the areas will code. The depth
 * for the sub-squares raises, of course. The AreaQuadTree can also suggest
 * which areas could collide with specified area. The information provided by
 * this class is not exact: It can always suggest some areas that do not satisfy
 * the condition but if there are such areas they are suggested.
 * 
 * @author Radim Vansa <radim.vansa@matfyz.cz>
 * 
 */
public class AreaQuadTree {

	private QTNode root = null;

	/**
	 * The one sub-square, called node (because it is in tree).
	 * 
	 * @author Radim Vansa <radim.vansa@matfyz.cz>
	 * 
	 */
	private class QTNode {

		private Rectangle2D rect = new Rectangle2D.Double();
		private QTNode n1, n2, n3, n4;
		private ArrayList<IArea> areas;
		private int position;

		/**
		 * Creates a QTNode enclosing the first area.
		 * 
		 * @param a
		 */
		public QTNode(IArea a) {
			Rectangle2D bounds = a.getBounds();
			if (bounds.getWidth() <= 0 || bounds.getHeight() <= 0) {
				throw new IllegalArgumentException(
						"Bounds of this area are empty!");
			}
			if (bounds.getHeight() > bounds.getWidth()) {
				this.rect.setFrame(bounds.getX(), bounds.getY(),
					bounds.getHeight(), bounds.getHeight());
			} else {
				this.rect.setFrame(bounds.getX(), bounds.getY(),
					bounds.getWidth(), bounds.getWidth());
			}
			this.areas = new ArrayList<IArea>();
			this.areas.add(a);
			this.position = 0;
		}

		private QTNode(int position, double x, double y, double size) {
			this.position = position;
			this.rect.setFrame(x, y, size, size);
			this.areas = new ArrayList<IArea>();
		}

		private QTNode(int position, double x, double y, double size,
				ArrayList<IArea> a) {
			this.position = position;
			this.rect.setFrame(x, y, size, size);
			this.areas = a;
		}

		/**
		 * If current QTNode is big enough to cover the whole bounding rectangle
		 * of inserted area, it adds the area into its sub-nodes, respectively
		 * splits itself into more sub-nodes, if it would lead to some
		 * optimisation. If this QTNode isn't big enough, it creates QTNodes in
		 * such way that the inserted area fits in the topmost - then it returns
		 * the topmost area.
		 */
		@SuppressWarnings("unchecked")
		public QTNode insert(IArea area) {
			Rectangle2D newRect = area.getBounds();
			if (newRect.getWidth() <= 0 || newRect.getHeight() <= 0) {
				throw new IllegalArgumentException(
						"Bounds of this area are empty!");
			}
			if (this.rect.contains(newRect)) {
				this.insert(area, newRect);
				return this;
			} else {
				QTNode node = this;
				do {
					if (newRect.getCenterX() >= node.rect.getCenterX()) {
						if (newRect.getCenterY() >= node.rect.getCenterY()) {
							QTNode temp = node;
							node = new QTNode(node.position - 1,
									node.rect.getX(), node.rect.getY(),
									node.rect.getWidth() * 2,
									(ArrayList<IArea>) node.areas.clone());
							node.n1 = temp;
							node.n2 = new QTNode(temp.position,
									node.rect.getX() + temp.rect.getWidth(),
									node.rect.getY(), temp.rect.getWidth());
							node.n3 = new QTNode(temp.position,
									node.rect.getX(), node.rect.getY()
											+ temp.rect.getWidth(),
									temp.rect.getWidth());
							node.n4 = new QTNode(temp.position,
									node.rect.getX() + temp.rect.getWidth(),
									node.rect.getY() + temp.rect.getWidth(),
									temp.rect.getWidth());
						} else {
							QTNode temp = node;
							node = new QTNode(node.position - 1,
									node.rect.getX(), node.rect.getY()
											- node.rect.getWidth(),
									node.rect.getWidth() * 2,
									(ArrayList<IArea>) node.areas.clone());
							node.n1 = new QTNode(temp.position,
									node.rect.getX(), node.rect.getY(),
									temp.rect.getWidth());
							node.n2 = new QTNode(temp.position,
									node.rect.getX() + temp.rect.getWidth(),
									node.rect.getY(), temp.rect.getWidth());
							node.n3 = temp;
							node.n4 = new QTNode(temp.position,
									node.rect.getX() + temp.rect.getWidth(),
									node.rect.getY() + temp.rect.getWidth(),
									temp.rect.getWidth());
						}
					} else {
						if (newRect.getCenterY() >= node.rect.getCenterY()) {
							QTNode temp = node;
							node = new QTNode(node.position - 1,
									node.rect.getX() - node.rect.getWidth(),
									node.rect.getY(), node.rect.getWidth() * 2,
									(ArrayList<IArea>) node.areas.clone());
							node.n1 = new QTNode(temp.position,
									node.rect.getX(), node.rect.getY(),
									temp.rect.getWidth());
							node.n2 = temp;
							node.n3 = new QTNode(temp.position,
									node.rect.getX(), node.rect.getY()
											+ temp.rect.getWidth(),
									temp.rect.getWidth());
							node.n4 = new QTNode(temp.position,
									node.rect.getX() + temp.rect.getWidth(),
									node.rect.getY() + temp.rect.getWidth(),
									temp.rect.getWidth());
						} else {
							QTNode temp = node;
							node = new QTNode(node.position - 1,
									node.rect.getX() - node.rect.getWidth(),
									node.rect.getY() - node.rect.getWidth(),
									node.rect.getWidth() * 2,
									(ArrayList<IArea>) node.areas.clone());
							node.n1 = new QTNode(temp.position,
									node.rect.getX(), node.rect.getY(),
									temp.rect.getWidth());
							node.n2 = new QTNode(temp.position,
									node.rect.getX() + temp.rect.getWidth(),
									node.rect.getY(), temp.rect.getWidth());
							node.n3 = new QTNode(temp.position,
									node.rect.getX(), node.rect.getY()
											+ temp.rect.getWidth(),
									temp.rect.getWidth());
							node.n4 = temp;
						}
					}

				} while (!node.rect.contains(newRect));
				node.insert(area, newRect);
				return node;
			}
		}

		private void insert(IArea area, Rectangle2D newRect) {
			areas.add(area);
			if (n1 != null) {// and n2 != null and so on...
				if (n1.rect.intersects(newRect)) {
					n1.insert(area, newRect);
				}
				if (n2.rect.intersects(newRect)) {
					n2.insert(area, newRect);
				}
				if (n3.rect.intersects(newRect)) {
					n3.insert(area, newRect);
				}
				if (n4.rect.intersects(newRect)) {
					n4.insert(area, newRect);
				}
			} else {
				if (test()) {
					split();
				}
			}
		}

		/*
		 * ! This method adds specified area into this node and tests, whether
		 * it shouldn't split itself into subareas. The node is splitted if it
		 * has more subareas (not counting these who are bigger than this node's
		 * rectangle) than the length of path from root to this node.
		 */
		private boolean test() {
			int count = 0;
			for (IArea a : areas) {
				Rectangle2D rr = a.getBounds();
				if (rr.contains(this.rect)) {
					continue;
				}
				count++;
			}
			return count > this.position - root.position;
		}

		private void split() {
			double size = this.rect.getWidth() / 2;
			n1 = new QTNode(this.position + 1, rect.getX(), rect.getY(), size);
			n2 = new QTNode(this.position + 1, rect.getX() + size, rect.getY(),
					size);
			n3 = new QTNode(this.position + 1, rect.getX(), rect.getY() + size,
					size);
			n4 = new QTNode(this.position + 1, rect.getX() + size, rect.getY()
					+ size, size);
			for (IArea a : areas) {
				// we could just split, but it would result into a lot of test
				Rectangle2D r = a.getBounds();
				if (n1.rect.intersects(r)) {
					n1.areas.add(a);
				}
				if (n2.rect.intersects(r)) {
					n2.areas.add(a);
				}
				if (n3.rect.intersects(r)) {
					n3.areas.add(a);
				}
				if (n4.rect.intersects(r)) {
					n4.areas.add(a);
				}
			}
			n1.testAndSplit();
			n2.testAndSplit();
			n3.testAndSplit();
			n4.testAndSplit();
		}

		private void testAndSplit() {
			if (test()) {
				split();
			}
		}

		public List<IArea> getAreasOn(double x, double y) {
			List<IArea> areas = new LinkedList<IArea>();
			for (IArea area : getAreasPossibleOn(x, y)) {
				if (area.contains(new Location(x, y), true)) {
					areas.add(area);
				}
			}
			return areas;
		}

		public List<IArea> getAreasPossibleOn(double x, double y) {
			if (n1 == null) {
				return Collections.unmodifiableList(this.areas);
			}
			if (y < rect.getCenterY()) {
				if (x < rect.getCenterX()) {
					return n1.getAreasPossibleOn(x, y);
				} else {
					return n2.getAreasPossibleOn(x, y);
				}
			} else {
				if (x < rect.getCenterX()) {
					return n3.getAreasPossibleOn(x, y);
				} else {
					return n4.getAreasPossibleOn(x, y);
				}
			}
		}

		public List<IArea> getAreasPossibleOn(Rectangle2D rect) {
			if (n1 != null) {// and n2 != null...
				if (n1.rect.contains(rect)) {
					return n1.getAreasPossibleOn(rect);
				} else if (n2.rect.contains(rect)) {
					return n2.getAreasPossibleOn(rect);
				} else if (n3.rect.contains(rect)) {
					return n3.getAreasPossibleOn(rect);
				} else if (n4.rect.contains(rect)) {
					return n4.getAreasPossibleOn(rect);
				}
			}
			return Collections.unmodifiableList(this.areas);
		}

		public void remove(IArea a) {
			if (areas.remove(a) && n1 != null) {
				n1.remove(a);
				n2.remove(a);
				n3.remove(a);
				n4.remove(a);
			}
		}
	}

	/**
	 * Adds a new area to the AreaQuadTree. This could take up to O(4^d) time
	 * where d is the maximum depth of the tree (which is limited by the number
	 * of areas in) so do it only in the pre-processing phase. Don't panic, the
	 * time in average cases for not completely overlapping areas is linear.
	 * Nevertheless the complete insertion would take O(n^2) time and therefore
	 * take the advice about pre-processing phase seriously. 
	 * 
	 * @param a
	 */
	public void insert(IArea a) {
		if (root == null) {
			root = new QTNode(a);
		} else {
			root = root.insert(a);
		}
	}
	
	/**
	 * Suggest areas on specified location.
	 * @param x
	 * @param y
	 * @return
	 */
	public List<IArea> getAreasPossibleOn(double x, double y) {
		if (root == null) {
			return new LinkedList<IArea>();
		} else {
			return root.getAreasPossibleOn(x, y);
		}
	}

	/**
	 * Suggest areas that could collide with this rectangle.
	 * @param rect
	 * @return
	 */
	public List<IArea> getAreasOn(Rectangle2D rect) {
		if (root == null) {
			return new LinkedList<IArea>();
		} else {
			return root.getAreasPossibleOn(rect);
		}
	}

	/**
	 * Removes the area from the AreaQuadTree. It has the same
	 * time cost as insertion.
	 * @param a
	 */
	public void remove(IArea a) {
		if (root != null) {
			root.remove(a);
		}
	}

}
