package cz.cuni.amis.pogamut.edu.drools;

import org.drools.FactHandle;
import org.drools.WorkingMemory;

/**
 * The class that actually manages being in the working memory. The logic was in
 * SimpleFact but some objects need the behaviour but cannot inherit the
 * SimpleFact. So all the code is here and SimpleFact just uses delegates the
 * method calls here.
 * 
 * @author Radim Vansa <radim.vansa@matfyz.cz>
 * 
 */
public class SimpleFactDelegate implements IFact {

	protected FactHandle factHandle;
	protected WorkingMemory workingMemory;
	protected int inserted = 0;
	protected IFact parent;

	/**
	 * Create new fact with generated name
	 */
	public SimpleFactDelegate(IFact parent) {
		this.parent = parent;
	}

	/**
	 * 
	 * @return fact handle of this object in its working memory
	 */
	protected FactHandle getFactHandle() {
		return factHandle;
	}

	/**
	 * 
	 * @return current working memory
	 */
	protected WorkingMemory getWorkingMemory() {
		return workingMemory;
	}

	/**
	 * All these methods must be synchronized over this delegate, because
	 * something should change the working memory between the null check and the
	 * beginning of synchronized block.
	 */
	public synchronized void insertTo(WorkingMemory workingMemory) {
		if (workingMemory != null) {
			this.workingMemory = workingMemory;
			synchronized (workingMemory) {
				this.factHandle = workingMemory.insert(parent);
				inserted++;
			}
		}
	}

	@Override
	public boolean isInserted() {
		return workingMemory != null;
	}

	public synchronized void retractFrom(WorkingMemory workingMemory) {
		if (workingMemory != this.workingMemory) {
			return;
		}
		if (workingMemory != null) {
			synchronized (workingMemory) {
				inserted--;
				if (inserted == 0) {
					workingMemory.retract(this.factHandle);
					this.workingMemory = null;
					this.factHandle = null;
				}
			}
		}
	}

	/**
	 * Sets fact handle
	 * 
	 * @param factHandle
	 */
	protected void setFactHandle(FactHandle factHandle) {
		this.factHandle = factHandle;
	}

	public synchronized void update() {
		if (this.workingMemory != null) {
			synchronized (this.workingMemory) {
				this.workingMemory.update(this.factHandle, parent);
			}
		}
	}

}
