package cz.cuni.amis.pogamut.edu.drools;

import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.logging.Logger;

import org.drools.RuleBase;
import org.drools.compiler.DroolsParserException;
import org.drools.compiler.PackageBuilder;

/**
 * The rules are not inserted into the rulebase one by one, they are gathered
 * into the rulebase and compiled together.
 * 
 * @author Radim Vansa <radim.vansa@matfyz.cz>
 * 
 */
public class RulesPackage extends Object {

	protected ArrayList<String> globals = new ArrayList<String>();
	protected ArrayList<String> imports = new ArrayList<String>();
	protected String pkg;
	protected HashMap<String, Rule> rules = new HashMap<String, Rule>();
	protected ArrayList<RuleBase> ruleBases = new ArrayList<RuleBase>();

	/**
	 * Creates a new rules package
	 */
	public RulesPackage() {

	}

	/**
	 * Creates a new rules package in specified package
	 * 
	 * @param pkg
	 */
	public RulesPackage(String pkg) {
		this.pkg = pkg;
	}

	/**
	 * Adds definition of global
	 * 
	 * @param class
	 * @param name
	 */
	public void addGlobal(String cls, String name) {
		globals.add(cls + " " + name);
	}

	/**
	 * Adds class to imported
	 * 
	 * @param import
	 */
	public void addImport(String imp) {
		imports.add(imp);
	}

	/**
	 * Adds another rule.
	 * 
	 * @param rule
	 */
	public void addRule(Rule rule) {
		rules.put(rule.getName(), rule);
	}

	/**
	 * Removes rule from this package as well as from rule bases where is this
	 * package registered
	 * 
	 * @param rule
	 */
	public void removeRule(Rule rule) {
		rule = rules.remove(rule.getName());
		if (rule != null) {
			for (RuleBase ruleBase : ruleBases) {
				ruleBase.removeRule(pkg, rule.getName());
			}
		}
	}

	/**
	 * Unregisters this package from specified rule base
	 * 
	 * @param ruleBase
	 */
	public void unregister(RuleBase ruleBase, Logger logger) {
		if (ruleBases.remove(ruleBase)) {
			ruleBase.removePackage(pkg);
		} else {
			logger
					.warning("This package isn't registered in specified RuleBase.");
		}
	}

	/**
	 * inserts all the rules to the rulebase
	 * 
	 * @param ruleBase
	 * @param logger -
	 *            output for possible errors
	 */
	public void registerAsDrl(RuleBase ruleBase, Logger logger) {
		ruleBases.add(ruleBase);
		PackageBuilder builder = new PackageBuilder();
		try {
			String string = this.toString();
			builder.addPackageFromDrl(new StringReader(string));
			ruleBase.addPackage(builder.getPackage());
		} catch (DroolsParserException e) {
			logger.severe("Cannot parse rules." + e.getMessage());
			return;
		} catch (IOException e) {
			logger.severe("Unable to read." + e.getMessage());
			return;
		} catch (Exception e) {
			logger.severe("Unable to add rules to existing rulebase."
					+ e.getMessage());
		}
	}

	/**
	 * Sets package for the rules
	 * 
	 * @param pkg
	 */
	public void setPackage(String pkg) {
		this.pkg = pkg;
	}

	/**
	 * 
	 * @return String with all the rules as if written in DRL file
	 */
	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();
		sb.append("package ");
		sb.append(this.pkg);
		sb.append(";\n\n");
		for (String imp : imports) {
			sb.append("import ");
			sb.append(imp);
			sb.append(";\n");
		}
		sb.append("\n");
		for (String global : globals) {
			sb.append("global ");
			sb.append(global);
			sb.append(";\n");
		}
		sb.append("\n");
		for (Rule rule : rules.values()) {
			sb.append(rule.toString());
			sb.append('\n');
		}
		return sb.toString();
	}

}
