package cz.cuni.amis.pogamut.edu.agent;

import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import cz.cuni.amis.pogamut.base.server.commands.IKeyEventListener;
import cz.cuni.amis.pogamut.edu.controlserver.IControlServer;
import cz.cuni.amis.pogamut.edu.controlserver.KeyEvent;
import cz.cuni.amis.pogamut.edu.drools.FactCollector;
import cz.cuni.amis.pogamut.edu.logging.Score;

/**
 * The agent that is controlled by human.
 * 
 * @author Radim Vansa <radim.vansa@matfyz.cz>
 * 
 */
public class Player extends Agent {
	/**
	 * Player's score (both bonuses and penalties).
	 */
	protected Score score = new Score();
	/**
	 * The listener for player's key events.
	 */
	protected IKeyEventListener keyEventListener;
	/**
	 * The agents whose position is available to this player (e.g. to be
	 * displayed on the map). These could be team-mates or such. The player
	 * itself must be also here to be able to know his position.
	 */
	protected Set<Agent> allowedAgents = new HashSet<Agent>();
	/**
	 * The name used in game. It is usually different from the name used in
	 * rules - there is for simplicity used "Player1", "Player2" etc.
	 */
	protected String inGameName;

	public Player(String name, String inGameName, IControlServer server) {
		super(name);
		setInGameName(inGameName);
		keyEventListener = new IKeyEventListener() {

			// the synchronisation is necessary because the fact has to be
			// added to the fact collector at the same while as inserted to
			// working memory to prohibit fireAllRules (in Scenario.run()),
			// insertTo, FactCollector.add(...), FactCollector.clear() (in
			// Scenario.run()) sequence
			@Override
			public void keyHeld(String key) {
				KeyEvent keyEvent = new KeyEvent(KeyEvent.Event.HOLD, key,
						Player.this);
				synchronized (getWorkingMemory()) {
					keyEvent.insertTo(getWorkingMemory());
					FactCollector.add(getWorkingMemory(), keyEvent);
				}
			}

			@Override
			public void keyPressed(String key) {
				KeyEvent keyEvent = new KeyEvent(KeyEvent.Event.PRESS, key,
						Player.this);
				synchronized (getWorkingMemory()) {
					keyEvent.insertTo(getWorkingMemory());
					FactCollector.add(getWorkingMemory(), keyEvent);
				}
			}

			@Override
			public void keyReleased(String key) {
				KeyEvent keyEvent = new KeyEvent(KeyEvent.Event.RELEASE, key,
						Player.this);
				synchronized (getWorkingMemory()) {
					keyEvent.insertTo(getWorkingMemory());
					FactCollector.add(getWorkingMemory(), keyEvent);
				}
			}

		};
		server.getPlayerCommunication().addKeyEventListener(getInGameName(),
			keyEventListener);
	}

	public Score getScore() {
		return this.score;
	}

	public void addAllowedAgent(Agent a) {
		allowedAgents.add(a);
	}

	public void removeAllowedAgent(Agent a) {
		allowedAgents.remove(a);
	}

	public void addAllowedAgents(Collection<Agent> as) {
		allowedAgents.addAll(as);
	}

	public void clearAllowedAgents() {
		allowedAgents.clear();
	}

	public boolean isAgentAllowed(Agent a) {
		return allowedAgents.contains(a);
	}

	@Override
	public String getInGameName() {
		return inGameName;
	}

	public void setInGameName(String inGameName) {
		this.inGameName = inGameName;
	}

	@Override
	public String toString() {
		return inGameName + '(' + getName() + ')';
	}

}
