package cz.cuni.amis.pogamut.edu.agent;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import cz.cuni.amis.pogamut.base.agent.IAgent;
import cz.cuni.amis.pogamut.base.communication.connection.socket.ISocketConnectionAddress;
import cz.cuni.amis.pogamut.base.communication.connection.socket.SocketConnectionAddress;
import cz.cuni.amis.pogamut.base.exceptions.PogamutException;
import cz.cuni.amis.pogamut.base.factory.IAgentFactory;
import cz.cuni.amis.pogamut.base.factory.guice.GuiceAgentFactory;
import cz.cuni.amis.pogamut.base.factory.guice.PogamutAgentModule;
import cz.cuni.amis.utils.Job;

/**
 * This class instantiates the bots.
 * 
 * @author Radim Vansa <radim.vansa@matfyz.cz>
 * 
 */
public class BotExecutor {

	/**
	 * Used to kill the agent in different thread (doesn't block the original
	 * thread).
	 * 
	 * @author Radim Vansa <radim.vansa@matfyz.cz>
	 * 
	 */
	protected static class AgentStopJob extends Job<Boolean> {

		private IAgent agent;

		public AgentStopJob(IAgent agent) {
			this.agent = agent;
		}

		@Override
		protected void job() throws Exception {
			agent.stop();
			setResult(true);
		}
	}

	public static Map<String, IAgent> instantiated = new HashMap<String, IAgent>();

	/**
	 * Creates a new Pogamut bot.
	 * 
	 * @param module
	 * @param name
	 * @param logger
	 * @return
	 */
	public static IAgent execute(PogamutAgentModule module, String name,
			Logger logger) {
		if (module == null) {
			logger.severe("Agent module not specified!");
		}

		IAgentFactory<ISocketConnectionAddress> factory = new GuiceAgentFactory<ISocketConnectionAddress>(
				module);

		IAgent agent = null;
		try {
			agent = factory.newAgent(new SocketConnectionAddress("localhost",
					3000));
		} catch (PogamutException e) {
			logger.severe("Can't create new agent." + e.getMessage());
			return null;
		}
		if (name != null) {
			Class<?> param[] = new Class<?>[1];
			param[0] = String.class;
			try {
				agent.getClass().getMethod("setName", param)
						.invoke(agent, name);
			} catch (Exception e) {
			}
		}

		agent.getLogger().addConsoleHandlersToAllCategories();
		agent.getLogger().setLevel(Level.WARNING);

		try {
			agent.start();
		} catch (PogamutException e) {
			logger.severe("Can't start the agent." + e.getMessage());
		}
		instantiated.put(agent.getName(), agent);
		return agent;
	}

	/**
	 * Actually doesn't work because the Class.forName method will not find the
	 * module in an OSGi environment.
	 * 
	 * @param agentClassName
	 * @param name
	 * @param logger
	 * @return
	 */
	public static IAgent execute(String agentClassName, String name,
			Logger logger) {
		try {
			PogamutAgentModule module = (PogamutAgentModule) Class.forName(
				agentClassName + "Module").newInstance();
			return execute(module, name, logger);
		} catch (InstantiationException e) {
			logger.severe("Cannot instantiate agent module " + e.getMessage());
		} catch (IllegalAccessException e) {
			logger.severe("Cannot instantiate agent module " + e.getMessage());
		} catch (ClassNotFoundException e) {
			logger.severe("Agent module class not found " + e.getMessage());
		}
		return null;
	}

	/**
	 * Kills the agent and removes it from the scenario.
	 * 
	 * @param agent
	 * @param logger
	 */
	public static void stop(IAgent agent, Logger logger) {
		if (agent == null) {
			return;
		}

		AgentStopJob stopBot = new AgentStopJob(agent);
		stopBot.startJob();
		try {
			stopBot.await(2000);
		} catch (InterruptedException e) {
			agent.kill();
			logger.warning("Interrupted during waiting for the agent to stop."
					+ e.getMessage());
		}

		if (!stopBot.isFinishedOk()) {
			agent.kill();
			logger.warning("Could not stop the agent in 2 secs.");
		}
	}

	/**
	 * When we want to wait until the bot already appears in the virtual reality
	 * we don't use the return value of execute method but after we notice some
	 * new agent we withdraw it here.
	 * 
	 * @param name
	 * @return
	 */
	public static IAgent yieldBot(String name) {
		return instantiated.remove(name);
	}
}
