package cz.cuni.amis.pogamut.edu;

import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Stack;
import java.util.Vector;
import java.util.logging.Logger;

import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.drools.RuleBase;
import org.drools.RuleBaseFactory;
import org.drools.WorkingMemory;
import org.drools.compiler.DroolsParserException;
import org.drools.compiler.PackageBuilder;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

import cz.cuni.amis.pogamut.base3d.worldview.objects.ILocated;
import cz.cuni.amis.pogamut.base3d.worldview.objects.Location;
import cz.cuni.amis.pogamut.edu.agent.Agent;
import cz.cuni.amis.pogamut.edu.agent.Bot;
import cz.cuni.amis.pogamut.edu.agent.Player;
import cz.cuni.amis.pogamut.edu.controlserver.IControlServer;
import cz.cuni.amis.pogamut.edu.drools.FactCollector;
import cz.cuni.amis.pogamut.edu.drools.IFact;
import cz.cuni.amis.pogamut.edu.drools.INamedFactHolder;
import cz.cuni.amis.pogamut.edu.drools.RulesPackage;
import cz.cuni.amis.pogamut.edu.drools.SimpleFact;
import cz.cuni.amis.pogamut.edu.drools.SimpleFactHolder;
import cz.cuni.amis.pogamut.edu.l10n.CannotTranslateException;
import cz.cuni.amis.pogamut.edu.l10n.ITranslator;
import cz.cuni.amis.pogamut.edu.l10n.NotSupportedLanguageException;
import cz.cuni.amis.pogamut.edu.l10n.TranslatorFactory;
import cz.cuni.amis.pogamut.edu.logging.Report;
import cz.cuni.amis.pogamut.edu.map.Map;
import cz.cuni.amis.pogamut.edu.map.areas.IComplexArea;
import cz.cuni.amis.pogamut.edu.map.exceptions.MapException;
import cz.cuni.amis.pogamut.edu.utils.IProvisory;
import cz.cuni.amis.pogamut.edu.utils.NamedObject;

/**
 * The basic class of Educational Scenarios. Holds all agents (players and
 * bots), the control server with link to Pogamut, logger and also the rulebase
 * and working memory. It can have four states, starting with the LOADING, where
 * no rules should be executed, switching to READY after it is loaded and
 * configured from the Java part and after initialization rules are executed it
 * switches to RUNNING state until it is forced to be FINISHED. It uses simple
 * loop where all affected rules are fired each 100ms. The scenario is loaded
 * from configuration file through ScenarioReader class.
 * 
 * @author Radim Vansa <radim.vansa@matfyz.cz>
 * 
 */
public abstract class Scenario extends SimpleFact {

	protected class ScenarioReader extends DefaultHandler {

		protected final static int NONE = 0, RULES = 1, SCENES = 2, MAPS = 3;
		protected Stack<Act> actStack = new Stack<Act>();
		protected Reader dsl;
		protected int phase = NONE;

		private void addAct(Attributes attributes) throws SAXException {
			String id = attributes.getValue("id");
			if (id != null) {
				Act a;
				if (actStack.isEmpty()) {
					a = new Act(id, null, Scenario.this);
					topActs.add(a);
					activeActs.add(null);
				} else {
					a = new Act(id, actStack.peek(), Scenario.this);
					actStack.peek().add(a);
				}
				actStack.push(a);
				actsByName.put(a.getName(), a);
			} else {
				throw new SAXException(
						"The attribute \"id\" is obligatory for act.");
			}
		}

		private void addRulePackage(Attributes attributes) throws SAXException {
			String src = attributes.getValue("src");
			if (src == null) {
				throw new SAXException(
						"The attribute \"src\" is obligatory for package.");
			} else if (dsl == null) {
				throw new SAXException("No DSL was specified!");
			} else {
				try {
					PackageBuilder builder = new PackageBuilder();
					builder.addPackageFromDrl(new FileReader(src), dsl);
					ruleBase.addPackage(builder.getPackage());
				} catch (FileNotFoundException e) {
					throw new SAXException(
							"File specified as rule package was not found:\n"
									+ e.getLocalizedMessage());
				} catch (DroolsParserException e) {
					throw new SAXException(
							"Problem occured when Drools parsed the rule package:\n"
									+ e.getLocalizedMessage());
				} catch (IOException e) {
					throw new SAXException("Cannot read the rule package:\n"
							+ e.getLocalizedMessage());
				} catch (Exception e) {
					throw new SAXException("Unable to add package \"" + src
							+ "\" to rulebase:\n" + e.getLocalizedMessage());
				}
			}
		}

		@Override
		public void endElement(String uri, String localName, String qName)
				throws SAXException {
			if (phase == SCENES) {
				if (actStack.isEmpty()) {
					phase = NONE;
				} else {
					actStack.pop();
				}
			}
		}

		@Override
		public void startElement(String uri, String localName, String qName,
				Attributes attributes) throws SAXException {
			if (translator == null) {
				if (qName.equalsIgnoreCase("scenario")) {
					String lang = attributes.getValue("lang");
					if (lang != null) {
						try {
							translator = TranslatorFactory
															.createTranslator(lang);
						} catch (NotSupportedLanguageException e) {
							throw new SAXException(
									"The selected language is not supported.");
						}
					} else {
						throw new SAXException(
								"Scenario has obligatory attribute \"lang\"");
					}
				} else {
					throw new SAXException(
							"The scenario file must start with root element \"scenario\"");
				}
			} else {
				String tag;
				try {
					tag = translator.translateFrom(qName);
				} catch (CannotTranslateException e) {
					tag = qName;
				}
				if (tag.equalsIgnoreCase("rules")) {
					phase = RULES;
					String dsl = attributes.getValue("dsl");
					if (dsl != null) {
						try {
							this.dsl = new FileReader(dsl);
						} catch (FileNotFoundException e) {
							throw new SAXException(
									"File specified as DSL was not found:\n"
											+ e.getLocalizedMessage());
						}
					} else {
						throw new SAXException(
								"The attribute \"dsl\" is obligatory for rules.");
					}
				} else if (tag.equalsIgnoreCase("scenes")) {
					phase = SCENES;
				} else if (tag.equalsIgnoreCase("maps")) {
					phase = MAPS;
				} else if (tag.equalsIgnoreCase("players")) {
					String min = attributes.getValue("min");
					String max = attributes.getValue("max");
					if (min != null) {
						minPlayers = Integer.parseInt(min);
					}
					if (max != null) {
						maxPlayers = Integer.parseInt(max);
					}
				} else {
					if (phase == RULES && tag.equalsIgnoreCase("package")) {
						addRulePackage(attributes);
					} else if (phase == SCENES && tag.equalsIgnoreCase("act")) {
						addAct(attributes);
					} else if (phase == MAPS && tag.equalsIgnoreCase("map")) {
						String src = attributes.getValue("src");
						if (src != null) {
							try {
								map.addData(src);
							} catch (MapException e) {
								throw new SAXException("Unable to load map:\n"
										+ e.getLocalizedMessage());
							}
						} else {
							throw new SAXException(
									"The attribute \"src\" is obligatory for map.");
						}
					}
				}
			}
		}
	}

	public enum State {
		LOADING, READY, RUNNING, FINISHED
	}

	/**
	 * The acts that are currently performed.
	 */
	protected ArrayList<Act> activeActs = new ArrayList<Act>();
	/**
	 * All acts.
	 */
	protected HashMap<String, Act> actsByName = new HashMap<String, Act>();
	/**
	 * All agents by their in game name.
	 */
	protected HashMap<String, Agent> agents = new HashMap<String, Agent>();
	/**
	 * The control server that is linked to Pogamut.
	 */
	protected IControlServer controlServer;
	/**
	 * Logger (for errors and debugging).
	 */
	public Logger logger = Logger.getAnonymousLogger();
	/**
	 * The Map object with MapData, MapGraph and AreaQuadTree.
	 */
	protected Map map = new Map(this);
	/**
	 * A number increased in each rule loop cycle. It is used when there are
	 * multiple possible changes that have priority but we cannot know the order
	 * of events causing the changes. The affected object remembers last mark of
	 * the change and will not allow the change with lesser priority if it has
	 * not greater mark.
	 */
	protected long mark = 0;
	/**
	 * Minimal and maximal number of players who can participate in this
	 * scenario.
	 */
	protected int minPlayers = 0, maxPlayers = Integer.MAX_VALUE;
	/**
	 * Players by their in game names.
	 */
	protected HashMap<String, Player> playerInGameNames = new HashMap<String, Player>();
	/**
	 * Players by identifiers "Player1", "Player2" etc.
	 */
	protected HashMap<String, Player> players = new HashMap<String, Player>();
	/**
	 * State in previous rule loop cycle - when we want to detect the change
	 * from one state to another, not just the actual state.
	 */
	protected State previous;
	/**
	 * Reports are logs about players behaviour stored after the scenario
	 * finishes.
	 */
	protected ArrayList<Report> reports = new ArrayList<Report>();
	/**
	 * Drools' object with all the rules.
	 */
	protected RuleBase ruleBase;
	/**
	 * The current state of the scenario.
	 */
	protected State state;
	/**
	 * A vector with the roots of all acts trees. Se Act for details.
	 */
	protected Vector<Act> topActs = new Vector<Act>();
	/**
	 * Translator used for localization.
	 */
	protected ITranslator translator;
	/**
	 * The variables used in the rules. It is mirrored in the working memory.
	 */
	protected java.util.Map<String, Object> variables = new HashMap<String, Object>();

	public Scenario(String source) {
		previous = State.LOADING;
		state = State.LOADING;
		SAXParserFactory spf = SAXParserFactory.newInstance();
		ruleBase = RuleBaseFactory.newRuleBase();
		try {
			SAXParser sp = spf.newSAXParser();
			if (source != null) {
				parse(sp, source);
			} else {
				throw new Exception("Cannot load acts, no source specified!");
			}
		} catch (Exception e) {
			e.printStackTrace();
			logger.severe("Cannot read acts!\n" + e.getMessage());
		}
		map.fillStructures();
		fillWorkingMemory();
	}

	public void addAgent(Agent agent) {
		agents.put(agent.getInGameName(), agent);
		variables.put(agent.getName(), agent);
		if (getWorkingMemory() != null) {
			agent.insertTo(getWorkingMemory());
		}
	}

	public void addBot(Bot bot) {
		addAgent(bot);
	}

	public void addPlayer(Player player) {
		players.put(player.getName(), player);
		playerInGameNames.put(player.getInGameName(), player);
		addAgent(player);
	}

	public void addReport(Report r) {
		this.reports.add(r);
	}

	public void destroyVariable(String name) {
		Object o = setVariable(name, null);
		if (o instanceof IFact) {
			((IFact) o).retractFrom(getWorkingMemory());
		}
	}

	/**
	 * Initializes globals in working memory and inserts all necessary objects
	 * there.
	 */
	protected void fillWorkingMemory() {
		{ // this may seem unnecessary, but both are needed
			this.insertTo(ruleBase.newStatefulSession());
			getWorkingMemory().setGlobal("scenario", this);
		}

		if (this.map != null) {
			map.registerMapDataUnitsAsVariables();
		}

		for (Act a : actsByName.values()) {
			a.insertTo(getWorkingMemory());
		}
		RulesPackage pkg = new RulesPackage();
		pkg.setPackage("cz.cuni.amis.pogamut.edu.generated");
		pkg.addGlobal(this.getClass().getCanonicalName(), "scenario");
		map.generateRules(pkg);
		pkg.registerAsDrl(ruleBase, logger);
	}

	public Agent getAgent(String name) {
		return agents.get(name);
	}

	public Collection<Agent> getAgentList() {
		return Collections.unmodifiableCollection(agents.values());
	}

	public Bot getBot(String name) {
		Agent a = agents.get(name);
		if (a instanceof Bot) {
			return (Bot) a;
		} else {
			return null;
		}
	}

	public IControlServer getControlServer() {
		return controlServer;
	}

	/**
	 * Computes the Euclidean distance between two objects.
	 * 
	 * @param o1
	 * @param o2
	 * @param ignoreZCoord
	 * @return
	 */
	public double getDistance(Object o1, Object o2, boolean ignoreZCoord) {
		Location l1, l2;
		if (o1 instanceof INamedFactHolder) {
			o1 = ((INamedFactHolder) o1).getData();
		}
		if (o1 instanceof ILocated) {
			l1 = ((ILocated) o1).getLocation();
		} else {
			throw new IllegalArgumentException(
					"Unable to resolve first argument's location");
		}
		if (o2 instanceof INamedFactHolder) {
			o2 = ((INamedFactHolder) o2).getData();
		}
		if (o2 instanceof ILocated) {
			l2 = ((ILocated) o2).getLocation();
		} else {
			throw new IllegalArgumentException(
					"Unable to resolve second argument's location");
		}
		double distance;
		if (ignoreZCoord) {
			distance = Location.getDistancePlane(l1, l2);
		} else {
			distance = Location.getDistance(l1, l2);
		}
		return distance;
	}

	public Logger getLogger() {
		return this.logger;
	}

	public Map getMap() {
		return map;
	}

	public long getMark() {
		return mark;
	}

	/**
	 * Returns the distance that should be considered as "nearby".
	 * 
	 * @return
	 */
	public abstract double getNearbyDistance();

	/**
	 * Returns player with the name as "Player1", "Player2" etc. If the
	 * parameter is null it returns the first player (or null if there is none).
	 * 
	 * @param name
	 * @return
	 */
	public Player getPlayer(String name) {
		if (name == null) {
			if (players.isEmpty()) {
				return null;
			}
			return players.values().iterator().next();
		} else {
			return players.get(name);
		}
	}

	public int getPlayerCount() {
		return players.size();
	}

	public Collection<Player> getPlayerList() {
		return Collections.unmodifiableCollection(players.values());
	}

	public Player getPlayerWithInGameName(String name) {
		return playerInGameNames.get(name);
	}

	public State getPrevious() {
		return previous;
	}

	/**
	 * Returns a random location that is in the specified distance from the
	 * origin. Used for hanging about or little shifts.
	 * 
	 * @param origin
	 * @param distance
	 * @return
	 */
	public Location getRandomCloseLocation(Location origin, double distance) {
		double angle = Math.random() * 2 * Math.PI;
		return new Location(origin.x + Math.sin(angle) * distance, origin.y
				+ Math.cos(angle) * distance, origin.z);
	}

	/**
	 * Creates a new RulesPackage. It has to be registered by
	 * registerRulesPackage(RulesPackage) after it is filled with rules.
	 * 
	 * @return
	 */
	public RulesPackage getRuntimeRulesPackage() {
		RulesPackage pkg = new RulesPackage();
		pkg.setPackage("cz.cuni.amis.pogamut.edu.generated.inruntime");
		pkg.addGlobal(this.getClass().getCanonicalName(), "scenario");
		return pkg;
	}

	public State getState() {
		return state;
	}

	public ITranslator getTranslator() {
		return translator;
	}

	public Object getVariable(String name) {
		return variables.get(name);
	}

	/**
	 * Exposes protected getWorkingMemory to public
	 */
	@Override
	public WorkingMemory getWorkingMemory() {
		return super.getWorkingMemory();
	}

	/**
	 * Parses the selected file and loads information from there. Overwrite if
	 * you want to specify another parser handler.
	 * 
	 * @param parser
	 * @param source
	 * @throws SAXException
	 * @throws IOException
	 */
	protected void parse(SAXParser parser, String source) throws SAXException,
			IOException {
		parser.parse(source, new ScenarioReader());
	}

	public void registerRulesPackage(RulesPackage pkg) {
		pkg.registerAsDrl(ruleBase, getLogger());
	}

	public Agent removeAgent(Agent agent) {
		agent.retractFrom(getWorkingMemory());
		variables.remove(agent.getName());
		return agents.remove(agent.getName());
	}

	public Bot removeBot(Bot bot) {
		return (Bot) removeAgent(bot);
	}

	public Bot removeBot(String name) {
		Agent a = agents.remove(name);
		if (a instanceof Bot) {
			variables.remove(a);
			a.retractFrom(getWorkingMemory());
			return (Bot) a;
		} else {
			// oops, accidentaly removed non-bot
			agents.put(a.getName(), a);
			return null;
		}
	}

	public Player removePlayer(Player player) {
		players.remove(player);
		return (Player) removeAgent(player);
	}

	public Object removeVariable(String name) {
		return variables.remove(name);
	}

	/**
	 * The rules loop.
	 * 
	 */
	public void run() {
		while (state != State.FINISHED) {
			synchronized (getWorkingMemory()) {
				getWorkingMemory().fireAllRules();
				FactCollector.clear(getWorkingMemory());
				++mark;
			}
			try {
				Thread.sleep(100);
			} catch (InterruptedException e) {
				logger.severe("Thread was interrupted." + e.getMessage());
				return;
			}
		}
	}

	/**
	 * Sets the current active act deactivating all the rules in the same acts
	 * tree but which are not superior to this act.
	 * 
	 * @param name
	 */
	public void setAct(String name) {
		++mark;
		Act act = this.actsByName.get(name);
		Act act2 = act;
		while (act2.getParent() != null) {
			act2.setActive(true, mark);
			act2 = act2.getParent();
		}
		act2.setActive(true, mark);
		int i = 0;
		for (Act a : topActs) {
			if (a == act2) {
				break;
			}
			++i;
		}
		act = this.activeActs.set(i, act);
		while (act != null && act.getMark() != mark) {
			act.setActive(false, mark);
			act = act.getParent();
		}
	}

	public void setControlServer(IControlServer controlServer) {
		this.controlServer = controlServer;
	}

	public void setLogger(Logger logger) {
		this.logger = logger;
	}

	public void setState(State state) {
		this.previous = this.state;
		this.state = state;
		update();
	}

	/**
	 * Sets current value of the variable, retracting the previous value from
	 * working memory and destroying it when it is IProvisory. If not and IFact
	 * is inserted it is enveloped by SimpleFactHolder.
	 * 
	 * @param name
	 * @param value
	 * @return
	 */
	public Object setVariable(String name, Object value) {
		Object o = variables.put(name, value);
		if (o == value) {
			return o;
		}
		if (o != null) {
			if (o instanceof IFact) {
				((IFact) o).retractFrom(getWorkingMemory());
			}
			if (o instanceof IProvisory) {
				((IProvisory) o).decreaseLinkCount();
			}
			// the variables and working memory would be not synchronized
			if (o instanceof NamedObject) {
				((NamedObject) o).generateNewName();
				if (o instanceof IFact) {
					((IFact) o).update();
				}
				if (o instanceof IComplexArea) {
					((IComplexArea) o).removeRules();
					getMap().registerArea((IComplexArea) o);
				}
			}
		}
		if (value instanceof IFact) {
			((IFact) value).insertTo(getWorkingMemory());
		} else {
			(new SimpleFactHolder(name, value)).insertTo(getWorkingMemory());
		}
		if (value instanceof IProvisory) {
			((IProvisory) value).increaseLinkCount();
		}
		return o;
	}

	/**
	 * Stores all the reports to the stream.
	 * 
	 * @param stream
	 */
	public void storeReports(OutputStream stream) {
		PrintWriter pw = new PrintWriter(stream);
		pw.write("Reports:\n\n");
		for (Report r : reports) {
			pw.write(r.toString());
			pw.write('\n');
		}
		pw.flush();
	}

	/**
	 * Stores all the reports to file with the specified filename.
	 * 
	 * @param fileName
	 */
	public void storeReports(String fileName) {
		try {
			FileOutputStream reports = new FileOutputStream(fileName);
			storeReports(reports);
			try {
				reports.close();
			} catch (IOException e) {
				logger.warning("Unable to close reports stream:\n"
						+ e.getLocalizedMessage());
			}
		} catch (FileNotFoundException e) {
			logger.severe("Unable to store reports to file \"" + fileName
					+ "\":\n" + e.getLocalizedMessage());
		}
	}

	/**
	 * Stores players' scores to the stream.
	 * 
	 * @param stream
	 */
	public void storeScores(OutputStream stream) {
		PrintWriter pw = new PrintWriter(stream);
		for (Player p : players.values()) {
			pw.write(p.toString());
			pw.write("\nBonus points: ");
			pw.write(String.valueOf(p.getScore().getBonuses()));
			pw.write("\nPenalty points: ");
			pw.write(String.valueOf(p.getScore().getPenalties()));
			pw.write("\nTotal score: ");
			pw.write(String.valueOf(p.getScore().getTotal()));
			pw.write("\n\n");
		}
		pw.flush();
	}

	/**
	 * Stores players' scores to file with specified filename.
	 * 
	 * @param fileName
	 */
	public void storeScores(String fileName) {
		try {
			FileOutputStream scores = new FileOutputStream(fileName);
			storeScores(scores);
			try {
				scores.close();
			} catch (IOException e) {
				logger.warning("Unable to close scores stream:\n"
						+ e.getLocalizedMessage());
			}
		} catch (FileNotFoundException e) {
			logger.severe("Unable to store scores to file \"" + fileName
					+ "\":\n" + e.getLocalizedMessage());
		}
	}
}
