package cz.cuni.amis.pogamut.edu.map.editor.visual.tools;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.MouseMoveListener;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;

import cz.cuni.amis.pogamut.base3d.worldview.objects.Location;
import cz.cuni.amis.pogamut.edu.map.areas.IArea;
import cz.cuni.amis.pogamut.edu.map.areas.PolygonArea;
import cz.cuni.amis.pogamut.edu.map.editor.Activator;
import cz.cuni.amis.pogamut.edu.map.editor.Constants;
import cz.cuni.amis.pogamut.edu.map.editor.utils.ICoordsTransformer;
import cz.cuni.amis.pogamut.edu.map.editor.visual.ToolKit;

public class PolygonTool extends AbstractPolyTool implements MouseListener,
		MouseMoveListener, KeyListener {

	static {
		Activator.getDefault().getImageRegistry().put("polytool",
			Activator.getImageDescriptor("icons/polytool.gif"));
	}

	public static boolean isCrossing(Location e1p1, Location e1p2,
			Location e2p1, Location e2p2) {
		double d1, d2;
		double diff1 = e1p2.x - e1p1.x;
		double diff2 = e2p2.x - e2p1.x;
		double crossX, crossY;
		if (diff1 == 0) {
			if (diff2 == 0) {
				return (e1p1.x == e2p1.x);
			}
			d2 = (e2p2.y - e2p1.y) / diff2;
			crossX = e1p1.x;
			crossY = (crossX - e2p1.x) * d2;
		} else {
			d1 = (e1p2.y - e1p1.y) / diff1;
			if (diff2 == 0) {
				crossX = e2p1.x;
				crossY = (crossX - e1p1.x) * d1;
			} else {
				d2 = (e2p2.y - e2p1.y) / diff2;
				if (d1 == d2) {
					return (e2p1.y - e1p1.y + e1p1.x * d1 - e2p1.x * d2 == 0);
				}
				crossX = (e2p1.y - e1p1.y + e1p1.x * d1 - e2p1.x * d2)
						/ (d1 - d2);
				crossY = d1 * (crossX - e1p1.x) + e1p1.y;
			}
		}
		// now we don't want to accept joint crosses as crosses
		return ((crossX > e1p1.x && crossX < e1p2.x) || (crossX > e1p2.x && crossX < e1p1.x))
				&& ((crossX > e2p1.x && crossX < e2p2.x) || (crossX > e2p2.x && crossX < e2p1.x))
				&& ((crossY > e1p1.y && crossY < e1p2.y) || (crossY > e1p2.y && crossY < e1p1.y))
				&& ((crossY > e2p1.y && crossY < e2p2.y) || (crossY > e2p2.y && crossY < e2p1.y));
	}

	protected PolygonArea active;

	protected PolygonArea constructed = new PolygonArea();
	protected int dragged = -1;
	private Point draggedNext;
	private Point draggedPrev;

	public PolygonTool(ToolKit toolKit) {
		super(toolKit, SWT.RADIO, "polytool", "Polygon tool");
	}

	protected void closePolygon() {
		constructed.setShapeHint(PolygonArea.Shape.POLYGON);
		toolKit.getEditor().addArea(constructed);
		toolKit.getEditor().setSelected(constructed);
		constructed = new PolygonArea();
	}

	@Override
	public void draw(GC gc, ICoordsTransformer coordsTransformer) {
		if (constructed.getNumPoints() == 0) {
			return;
		}
		int[] converted = new int[2 * constructed.getNumPoints()];
		gc.setForeground(Constants.BLACK);
		gc.setBackground(Constants.WHITE);
		// first draw lines, then corners
		Point tp1 = coordsTransformer.location2client(constructed.getPoint(0));
		converted[0] = tp1.x;
		converted[1] = tp1.y;
		Point tp2;
		for (int i = 0; i < constructed.getNumPoints(); ++i) {
			tp2 = coordsTransformer.location2client(constructed.getPoint(i));
			converted[2 * i] = tp2.x;
			converted[2 * i + 1] = tp2.y;
			gc.drawLine(tp1.x, tp1.y, tp2.x, tp2.y);
			tp1 = tp2;
		}
		for (int i = 0; i < converted.length; i += 2) {
			gc.fillRectangle(converted[i] - 3, converted[i + 1] - 3, 6, 6);
			gc.drawRectangle(converted[i] - 3, converted[i + 1] - 3, 6, 6);
		}
	}

	@Override
	public void drawDirect(GC gc, ICoordsTransformer coordsTransformer) {
		if (dragged != -1) {
			if (draggedPrev == null) {
				if (dragged != 0) {
					draggedPrev = coordsTransformer.location2client(
						active.getPoint(dragged - 1));
				} else if (active != constructed) {
					draggedPrev = coordsTransformer.location2client(
						active.getPoint(active.getNumPoints() - 1));
				}
			}
			if (draggedNext == null) {
				if (dragged != active.getNumPoints() - 1) {
					draggedNext = coordsTransformer.location2client(
						active.getPoint(dragged + 1));
				} else if (active != constructed) {
					draggedNext = coordsTransformer.location2client(
						active.getPoint(0));
				}
			}
			Point draggedPoint = coordsTransformer.location2client(
				active.getPoint(dragged));
			gc.setForeground(Constants.BLACK);
			gc.setBackground(Constants.WHITE);
			if (draggedPrev != null) {
				gc.drawLine(draggedPrev.x, draggedPrev.y, draggedPoint.x,
					draggedPoint.y);
				drawJoint(gc, draggedPrev);
			}
			if (draggedNext != null) {
				gc.drawLine(draggedNext.x, draggedNext.y, draggedPoint.x,
					draggedPoint.y);
				drawJoint(gc, draggedNext);
			}
			drawJoint(gc, draggedPoint);
		}
	}

	private void drawJoint(GC gc, Point p) {
		gc.fillRectangle(p.x - 3, p.y - 3, 6, 6);
		gc.drawRectangle(p.x - 3, p.y - 3, 6, 6);
	}

	/**
	 * Tests if the polygon would cross itself if continued with those points.
	 * Assumes that it already has some points
	 * 
	 * @param x
	 * @param y
	 * @return
	 */
	protected boolean isCrossing(Location n) {
		Location last = constructed.getPoint(constructed.getNumPoints() - 1);
		Location l1 = constructed.getPoint(0);
		Location l2;
		for (int i = 1; i < constructed.getNumPoints(); ++i) {
			l2 = constructed.getPoint(i);
			if (isCrossing(n, last, l1, l2)) {
				return true;
			}
			l1 = l2;
		}
		return false;
	}

	@Override
	public void keyPressed(KeyEvent e) {
		if (e.keyCode == SWT.DEL || e.keyCode == SWT.BS) {
			Object selected = toolKit.getEditor().getSelected();
			if (selected == null) {
				constructed.removePoint(constructed.getNumPoints() - 1);
			} else if (selected instanceof IArea) {
				toolKit.getEditor().removeArea((IArea) selected);
				toolKit.getEditor().setSelected(null);
			}
			setNeedsUpdate(true, true);
		} else if (e.keyCode == SWT.ESC) {
			constructed.removeAllPoints();
			setNeedsUpdate(true, true);
		}
	}

	@Override
	public void keyReleased(KeyEvent e) {
	}

	@Override
	public void mouseDoubleClick(MouseEvent e) {
	}

	@Override
	public void mouseDown(MouseEvent e) {
		Location l = new Location((double) e.x, (double) e.y);
		// test if it isn't a request to move something
		if (!processJointPress(e)) {
			if (active == null) {
				// test if it actually shouldn't be the constructed
				for (int i = 1; i < constructed.getNumPoints(); ++i) {
					if (toolKit.getEditor().isClose(l, constructed.getPoint(i))) {
						active = constructed;
						dragged = i;
						break;
					}
				}
			}
			if (dragged == -1) {
				if (constructed.getNumPoints() == 0) {
					constructed.addPoint(l);
					setNeedsUpdate(true, true);
				} else if (toolKit.getEditor().isClose(l,
					constructed.getPoint(0))
						&& constructed.getNumPoints() >= 3
						&& !isCrossing(constructed.getPoint(0))) {
					// it's important to not test crossing of l but the first
					// point!
					closePolygon();
					setNeedsUpdate(true, true);
				} else if (!isCrossing(l)) {
					constructed.addPoint(l);
					setNeedsUpdate(true, true);
				}
			}
		}
	}

	@Override
	public void mouseMove(MouseEvent e) {
		if (dragged != -1) {
			active.setPoint(dragged, e.x, e.y);
			setNeedsUpdate(true, false);
		}
	}

	@Override
	public void mouseUp(MouseEvent e) {
		if (dragged != -1) {
			if (active != constructed) {
				toolKit.getEditor().updateArea(active);
			}
			active = null;
			dragged = -1;
			draggedPrev = null;
			draggedNext = null;
			setNeedsUpdate(true, true);
		}
	}

	@Override
	public void jointPressed(PolygonArea poly, Location location,
			int jointNumber) {
		active = poly;
		dragged = jointNumber;		
	}
}
