package cz.cuni.amis.pogamut.edu.map.editor.visual;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.MouseMoveListener;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.ScrollBar;

import cz.cuni.amis.pogamut.base3d.worldview.objects.Location;
import cz.cuni.amis.pogamut.edu.map.editor.Constants;
import cz.cuni.amis.pogamut.edu.map.editor.utils.ICoordsTransformer;

public class WorkField extends CompositeHolder {

	protected class PaintListener implements Listener {
		public void handleEvent(Event e) {
			GC gc = e.gc;
			if (background == null) {
				gc.setBackground(Constants.WHITE);
				Rectangle canvasRect = canvas.getClientArea();
				gc.fillRectangle(canvasRect);
				gc.drawText("No image", canvasRect.width / 2 - 50,
					canvasRect.height / 2 - 25);
			} else {
				if (!checkBuffer()) {
					resetBuffer();
					redrawBuffer();
					syncScrollBars();
				}
				GC shortBufferGC = new GC(bufferShort);
				shortBufferGC.drawImage(bufferLong, 0, 0);
				editor.drawDataDirect(shortBufferGC, coordsTransformer);
				shortBufferGC.dispose();
				gc.drawImage(bufferShort, 0, 0);
			}
		}
	}

	protected static double ZOOM_COEFICIENT = 1.3;

	protected Image background = null;
	/** This buffer is updated only when scrolling or zooming. */
	protected Image bufferLong;
	/**
	 * This buffer provides double-buffering. It has the same dimension as
	 * bufferLong.
	 */
	protected Image bufferShort;

	protected Canvas canvas;
	protected ICoordsTransformer coordsTransformer = new ICoordsTransformer() {

		double originX = 0;
		double originY = 0;
		double scale = 1;

		@Override
		public Point client2image(Point p) {
			Point p2 = new Point(0, 0);
			Rectangle clientRect = canvas.getClientArea();
			Rectangle imageRect = background.getBounds();
			if (imageRect.width * zoom < clientRect.width) {
				p2.x = (int) ((p.x - (clientRect.width - imageRect.width * zoom) / 2) / zoom);
			} else {
				p2.x = (int) ((p.x + scroll.x) / zoom);
			}
			if (imageRect.height * zoom < clientRect.height) {
				p2.y = (int) ((p.y - (clientRect.height - imageRect.height
						* zoom) / 2) / zoom);
			} else {
				p2.y = (int) ((p.y + scroll.y) / zoom);
			}
			return p2;
		}

		@Override
		public Location client2location(Point p) {
			Location l = new Location();
			Rectangle clientRect = canvas.getClientArea();
			Rectangle imageRect = background.getBounds();
			if (imageRect.width * zoom < clientRect.width) {
				l.x = scale
						* (p.x - (clientRect.width - imageRect.width * zoom) / 2)
						/ zoom + originX;
			} else {
				l.x = scale * ((p.x + scroll.x) / zoom) + originX;
			}
			if (imageRect.height * zoom < clientRect.height) {
				l.y = scale
						* (p.y - (clientRect.height - imageRect.height * zoom) / 2)
						/ zoom + originY;
			} else {
				l.y = scale * (p.y + scroll.y) / zoom + originY;
			}
			return l;
		}

		@Override
		public double distance2client(double distance) {
			return distance / scale;
		}

		@Override
		public Point image2client(Point p) {
			Point p2 = new Point(0, 0);
			Rectangle clientRect = canvas.getClientArea();
			Rectangle imageRect = background.getBounds();
			if (imageRect.width * zoom < clientRect.width) {
				p2.x = (int) ((clientRect.width - imageRect.width * zoom) / 2 + p.x
						* zoom);
			} else {
				p2.x = (int) (p.x * zoom - scroll.x);
			}
			if (imageRect.height * zoom < clientRect.height) {
				p2.y = (int) ((clientRect.height - imageRect.height * zoom) / 2 + p.y
						* zoom);
			} else {
				p2.y = (int) (p.y * zoom - scroll.y);
			}
			return p2;
		}

		@Override
		public Point location2client(Location l) {
			Point p2 = new Point(0, 0);
			Rectangle clientRect = canvas.getClientArea();
			Rectangle imageRect = background.getBounds();
			if (imageRect.width * zoom < clientRect.width) {
				p2.x = (int) ((clientRect.width - imageRect.width * zoom) / 2 + (l.x - originX)
						/ scale * zoom);
			} else {
				p2.x = (int) ((l.x - originX) / scale * zoom - scroll.x);
			}
			if (imageRect.height * zoom < clientRect.height) {
				p2.y = (int) ((clientRect.height - imageRect.height * zoom) / 2 + (l.y - originY)
						/ scale * zoom);
			} else {
				p2.y = (int) ((l.y - originY) / scale * zoom - scroll.y);
			}
			return p2;
		}

		@Override
		public void setParams(double scale, double originX, double originY) {
			this.scale = scale;
			this.originX = originX;
			this.originY = originY;
		}

	};
	protected VisualEditor editor;
	protected Point scroll = new Point(0, 0);

	protected double zoom = 1;

	public WorkField(Composite parent, VisualEditor editor) {
		this.editor = editor;
		canvas = new Canvas(parent, SWT.BORDER | SWT.NO_BACKGROUND
				| SWT.NO_REDRAW_RESIZE | SWT.H_SCROLL | SWT.V_SCROLL);
		canvas.setCursor(new Cursor(null, SWT.CURSOR_CROSS));
		canvas.addListener(SWT.Paint, new PaintListener());
		canvas.addListener(SWT.RESIZE, new Listener() {

			@Override
			public void handleEvent(Event event) {
				resetBuffer();
				syncScrollBars();
				redrawBuffer();
				canvas.redraw();
			}

		});
		canvas.addMouseListener(new MouseListener() {

			@Override
			public void mouseDoubleClick(MouseEvent e) {
				transformCoords(e);
				getEditor().mouseDoubleClick(e);
				if (getEditor().getNeedsUpdate()) {
					update(getEditor().getNeedsFullUpdate());
				}
			}

			@Override
			public void mouseDown(MouseEvent e) {
				transformCoords(e);
				getEditor().mouseDown(e);
				if (getEditor().getNeedsUpdate()) {
					update(getEditor().getNeedsFullUpdate());
				}
			}

			@Override
			public void mouseUp(MouseEvent e) {
				transformCoords(e);
				getEditor().mouseUp(e);
				if (getEditor().getNeedsUpdate()) {
					update(getEditor().getNeedsFullUpdate());
				}
			}

		});
		canvas.addMouseMoveListener(new MouseMoveListener() {

			@Override
			public void mouseMove(MouseEvent e) {
				transformCoords(e);
				getEditor().mouseMove(e);
				if (getEditor().getNeedsUpdate()) {
					update(getEditor().getNeedsFullUpdate());
				}
			}

		});
		canvas.addKeyListener(new KeyListener() {

			@Override
			public void keyPressed(KeyEvent e) {
				getEditor().keyPressed(e);
				if (getEditor().getNeedsUpdate()) {
					update(getEditor().getNeedsFullUpdate());
				}
			}

			@Override
			public void keyReleased(KeyEvent e) {
				getEditor().keyReleased(e);
				if (getEditor().getNeedsUpdate()) {
					update(getEditor().getNeedsFullUpdate());
				}
			}

		});
		canvas.getHorizontalBar().addListener(SWT.Selection, new Listener() {

			@Override
			public void handleEvent(Event event) {
				scroll.x = canvas.getHorizontalBar().getSelection();
				update(true);
			}

		});
		canvas.getVerticalBar().addListener(SWT.Selection, new Listener() {

			@Override
			public void handleEvent(Event event) {
				scroll.y = canvas.getVerticalBar().getSelection();
				update(true);
			}

		});
	}

	public void dispose() {
		if (bufferLong != null) {
			bufferLong.dispose();
			bufferShort.dispose();
		}
		if (background != null) {
			background.dispose();
		}
	}

	@Override
	public Composite getComposite() {
		return canvas;
	}

	public ICoordsTransformer getCoordsTransformer() {
		return coordsTransformer;
	}

	public VisualEditor getEditor() {
		return editor;
	}

	protected boolean checkBuffer() {
		if (bufferLong == null) {
			return false;
		}
		Rectangle clientRect = canvas.getClientArea();
		Rectangle bufRect = bufferLong.getBounds();
		if (bufRect.width != clientRect.width
				|| bufRect.height != clientRect.height) {
			return false;
		}
		return true;
	}

	protected void redrawBuffer() {
		GC gc = new GC(bufferLong);
		gc.setBackground(Constants.BLACK);
		// to prevent situation when image is too small
		gc.fillRectangle(bufferLong.getBounds());
		Rectangle bgRect = background.getBounds();
		Rectangle client = canvas.getClientArea();
		Point p1 = coordsTransformer.client2image(new Point(0, 0));
		Point p2 = coordsTransformer.client2image(new Point(client.width,
				client.height));
		p1.x = Math.max(p1.x, 0);
		p1.y = Math.max(p1.y, 0);
		p2.x = Math.min(p2.x, bgRect.width);
		p2.y = Math.min(p2.y, bgRect.height);
		Point cp1 = coordsTransformer.image2client(p1);
		Point cp2 = coordsTransformer.image2client(p2);

		gc.drawImage(background, p1.x, p1.y, p2.x - p1.x, p2.y - p1.y, cp1.x,
			cp1.y, cp2.x - cp1.x, cp2.y - cp1.y);
		editor.drawData(gc, coordsTransformer);
		gc.dispose();
	}

	protected void resetBuffer() {
		if (bufferLong != null) {
			bufferLong.dispose();
			bufferShort.dispose();
		}
		Rectangle canvasRect = canvas.getClientArea();
		bufferLong = new Image(null, canvasRect.width, canvasRect.height);
		bufferShort = new Image(null, canvasRect.width, canvasRect.height);
	}

	public void setBackground(String path) {
		if (background != null) {
			background.dispose();
		}
		background = new Image(null, path);
	}

	protected void syncScrollBars() {
		ScrollBar vBar = canvas.getVerticalBar();
		ScrollBar hBar = canvas.getHorizontalBar();
		if (background == null) {
			vBar.setEnabled(false);
			hBar.setEnabled(false);
			return;
		}
		Rectangle bgRect = background.getBounds();
		Rectangle canvasRect = canvas.getClientArea();

		hBar.setIncrement((int) (canvasRect.width / 100));
		hBar.setPageIncrement(canvasRect.width);

		if (scroll.x < 0) {
			scroll.x = 0;
		}
		if (canvasRect.width < bgRect.width * zoom) {
			hBar.setEnabled(true);
			hBar.setMaximum((int) (bgRect.width * zoom));
			if (scroll.x > bgRect.width * zoom - canvasRect.width) {
				scroll.x = (int) (bgRect.width * zoom - canvasRect.width);
			}
		} else {
			hBar.setEnabled(false);
			scroll.x = 0;
		}
		hBar.setThumb(canvasRect.width);
		hBar.setSelection(scroll.x);

		vBar.setIncrement((int) (canvasRect.height / 100));
		vBar.setPageIncrement(canvasRect.height);
		if (scroll.y < 0) {
			scroll.y = 0;
		}
		if (canvasRect.height < bgRect.height * zoom) {
			vBar.setEnabled(true);
			vBar.setMaximum((int) (bgRect.height * zoom));
			if (scroll.y > bgRect.height * zoom - canvasRect.height) {
				scroll.y = (int) (bgRect.height * zoom - canvasRect.height);
			}
		} else {
			vBar.setEnabled(false);
			scroll.y = 0;
		}
		vBar.setThumb(canvasRect.height);
		vBar.setSelection(scroll.y);
	}

	protected void transformCoords(MouseEvent e) {
		if (background != null) {
			Location l = coordsTransformer.client2location(new Point(e.x, e.y));
			e.x = (int) l.x;
			e.y = (int) l.y;
		}
	}

	public void update(boolean redrawBuffer) {
		if (redrawBuffer) {
			syncScrollBars();
			redrawBuffer();
		}
		canvas.redraw();
		canvas.update();
	}

	public void zoomIn() {
		zoom *= ZOOM_COEFICIENT;
		Rectangle canvasRect = canvas.getBounds();
		scroll.x = (int) ((scroll.x + canvasRect.width / 2) * ZOOM_COEFICIENT - canvasRect.width / 2);
		scroll.y = (int) ((scroll.y + canvasRect.height / 2) * ZOOM_COEFICIENT - canvasRect.height / 2);
		update(true);
	}

	public void zoomOut() {
		zoom /= ZOOM_COEFICIENT;
		Rectangle canvasRect = canvas.getBounds();
		scroll.x = (int) ((scroll.x + canvasRect.width / 2) / ZOOM_COEFICIENT - canvasRect.width / 2);
		scroll.y = (int) ((scroll.y + canvasRect.height / 2) / ZOOM_COEFICIENT - canvasRect.height / 2);
		update(true);
	}
}
