package cz.cuni.amis.pogamut.edu.map.editor.visual;

import java.io.IOException;
import java.io.InputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.MouseMoveListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.dialogs.SaveAsDialog;
import org.eclipse.ui.part.EditorPart;
import org.eclipse.ui.part.FileEditorInput;

import com.sun.xml.internal.messaging.saaj.util.ByteInputStream;

import cz.cuni.amis.pogamut.base3d.worldview.objects.Location;
import cz.cuni.amis.pogamut.edu.l10n.ITranslator;
import cz.cuni.amis.pogamut.edu.l10n.NotSupportedLanguageException;
import cz.cuni.amis.pogamut.edu.l10n.TranslatorFactory;
import cz.cuni.amis.pogamut.edu.map.AreaQuadTree;
import cz.cuni.amis.pogamut.edu.map.MapData;
import cz.cuni.amis.pogamut.edu.map.areas.IArea;
import cz.cuni.amis.pogamut.edu.map.areas.IComplexArea;
import cz.cuni.amis.pogamut.edu.map.areas.ICompoundArea;
import cz.cuni.amis.pogamut.edu.map.areas.PolygonArea;
import cz.cuni.amis.pogamut.edu.map.editor.Constants;
import cz.cuni.amis.pogamut.edu.map.editor.utils.ColorConvertor;
import cz.cuni.amis.pogamut.edu.map.editor.utils.ICoordsTransformer;
import cz.cuni.amis.pogamut.edu.map.editor.visual.tools.Tool;
import cz.cuni.amis.pogamut.edu.map.exceptions.MapException;
import cz.cuni.amis.pogamut.edu.map.marks.IMark;
import cz.cuni.amis.pogamut.edu.map.marks.POI;
import cz.cuni.amis.pogamut.edu.map.marks.Path;
import cz.cuni.amis.pogamut.edu.map.marks.PathPoint;
import cz.cuni.amis.pogamut.edu.utils.AbstractRescaler;

public class VisualEditor extends EditorPart implements MouseListener,
		MouseMoveListener, KeyListener {
	private class Rescaler extends AbstractRescaler {
		private double newOX, newOY, newScale;
		private double prevOX, prevOY, prevScale;

		public Rescaler(double prevScale, double prevOX, double prevOY,
				double newScale, double newOX, double newOY) {
			this.prevOX = prevOX;
			this.prevOY = prevOY;
			this.prevScale = prevScale;
			this.newOX = newOX;
			this.newOY = newOY;
			this.newScale = newScale;
		}

		@Override
		public double rescaleX(double x) {
			return ((x - prevOX) / prevScale) * newScale + newOX;
		}

		@Override
		public double rescaleY(double y) {
			return ((y - prevOY) / prevScale) * newScale + newOY;
		}

		@Override
		public double rescaleZ(double z) {
			return z;
		}
	}

	public static final String ID = "cz.cuni.amis.pogamut.edu.edumapeditor.visualeditor";
	protected AreaQuadTree areaQuadTree = new AreaQuadTree();
	protected boolean dirty;
	protected IArea locked;
	protected MapData mapData;
	protected List<String> messages = new LinkedList<String>();
	protected boolean needsUpdate, needsFullUpdate;
	protected Object selected;
	protected Shell shell;
	protected ToolKit toolKit;
	protected ITranslator translator;

	protected WorkField workField;

	public VisualEditor() {
		try {
			translator = TranslatorFactory.createTranslator("english");
		} catch (NotSupportedLanguageException e) {
			e.printStackTrace();
		}
	}

	public void addArea(IArea area) {
		setDirty(true);
		mapData.add(area);
		if (!(area instanceof IComplexArea)) {
			areaQuadTree.insert(area);
		}
	}

	public void addMark(IMark mark) {
		mapData.addMark(mark);
		setDirty(true);
	}

	@Override
	public void createPartControl(Composite parent) {
		shell = parent.getShell();
		// some messages were thrown before the shell was set
		displayErrors();

		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		parent.setLayout(layout);
		{
			workField = new WorkField(parent, this);
			GridData workFieldGridData = new GridData();
			workFieldGridData.horizontalAlignment = SWT.FILL;
			workFieldGridData.verticalAlignment = SWT.FILL;
			workFieldGridData.grabExcessHorizontalSpace = true;
			workFieldGridData.grabExcessVerticalSpace = true;
			workField.getComposite().setLayoutData(workFieldGridData);
		}
		{
			toolKit = new ToolKit(parent, this);
			GridData toolBarGridData = new GridData();
			toolBarGridData.verticalAlignment = SWT.TOP;
			toolKit.getComposite().setLayoutData(toolBarGridData);
		}
		parent.addKeyListener(this);
		parent.pack();
		loadContent();
	}

	protected void displayErrors() {
		if (errorCanBeDisplayed()) {
			for (String message : messages) {
				errorMessageBox(message);
			}
			messages.clear();
		}
	}

	@Override
	public void dispose() {
		super.dispose();
		if (workField != null) {
			workField.dispose();
		}
	}

	@Override
	public void doSave(IProgressMonitor monitor) {
		if (monitor != null) {
			monitor.beginTask("Saving...", 1);
		}
		final PipedOutputStream output = new PipedOutputStream();
		PipedInputStream input = new PipedInputStream();
		try {
			output.connect(input);
		} catch (IOException e) {
			error("Problem occured when trying to store data.", e);
		}
		new Thread(new Runnable() {

			@Override
			public void run() {
				try {
					mapData.storeData(output, translator);
				} catch (MapException e) {
					error("Problem occured when trying to store data.", e);
				} finally {
					try {
						output.close();
					} catch (IOException e) {
						error("Couldn't close stream.", e);
					}
				}
			}

		}).start();
		IFile file = ((IFileEditorInput) getEditorInput()).getFile();
		try {
			file.setContents(input, false, true, monitor);
		} catch (CoreException e) {
			// some errors shouldn't been displayed in the other thread
			error("Problem occured when trying to store data.", e);
			return;
		} finally {
			displayErrors();
		}

		if (monitor != null) {
			monitor.worked(1);
			monitor.done();
		}
		setDirty(false);
	}

	@Override
	public void doSaveAs() {
		SaveAsDialog dialog = new SaveAsDialog(getShell());
		dialog.setOriginalFile(((IFileEditorInput) getEditorInput()).getFile());
		dialog.open();
		if (dialog.getResult() != null) {
			IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(
				dialog.getResult());
			setInput(new FileEditorInput(file));
			setPartName(file.getFullPath().lastSegment());
			if (!file.exists()) {
				try {
					file.create(new ByteInputStream(), false, null);
				} catch (CoreException e) {
					error("Cannot create file", e);
				}
			}
			doSave(null);
		}
	}

	/**
	 * Draws data on long-time buffer. Use for drawings that update only with
	 * scroll, zoom or serious action (adding a new point etc.).
	 * 
	 * @param gc
	 * @param coordsTransformer
	 */
	public void drawData(GC gc, ICoordsTransformer coordsTransformer) {
		if (mapData != null) {
			Tool active = toolKit.getActiveTool();
			drawSimpleAreas(gc, coordsTransformer);
			drawMarks(gc, coordsTransformer);
			if (active != null) {
				active.draw(gc, coordsTransformer);
			}
		}
	}

	/**
	 * Draws data directly on short-time buffer. Use for immediate drawing as
	 * resizing rectangles etc.
	 * 
	 * @param gc
	 * @param coordsTransformer
	 */
	public void drawDataDirect(GC gc, ICoordsTransformer coordsTransformer) {
		Tool active = toolKit.getActiveTool();
		if (active != null) {
			active.drawDirect(gc, coordsTransformer);
		}
	}

	private void drawJoint(GC gc, Point p) {
		gc.fillRectangle(p.x - 3, p.y - 3, 6, 6);
		gc.drawRectangle(p.x - 3, p.y - 3, 6, 6);
	}

	private void drawMarks(GC gc, ICoordsTransformer coordsTransformer) {
		gc.setAlpha(255);
		ArrayList<Point> converted = new ArrayList<Point>();
		for (IMark mark : mapData.getMarks()) {
			if (mark instanceof POI) {
				POI poi = (POI) mark;
				gc.setForeground(ColorConvertor.awt2swt(poi.getColor()));
				String symbol = "[" + poi.getSymbol().toString() + "]";
				Point p = gc.stringExtent(symbol);
				Point p2 = coordsTransformer.location2client(poi.getLocation());
				gc.drawText(symbol, p2.x - p.x / 2, p2.y - p.y - 5, true);
				p = gc.stringExtent(poi.getDescription());
				gc.drawText(poi.getDescription(), p2.x - p.x / 2, p2.y + 5,
					true);
				gc.setBackground(Constants.WHITE);
				gc.setForeground(Constants.BLACK);
				drawJoint(gc, p2);
			} else if (mark instanceof Path) {
				Path path = (Path) mark;
				gc.setAlpha(255);
				gc.setForeground(ColorConvertor.awt2swt(path.getColor()));
				gc.setLineCap(SWT.CAP_ROUND);
				gc.setLineWidth((int) Math.ceil(path.getWidth()));
				Point last = null;
				for (PathPoint p : path.getPoints()) {
					Point p2 = coordsTransformer
												.location2client(p
																	.getLocation());
					converted.add(p2);
					if (last != null) {
						gc.drawLine(last.x, last.y, p2.x, p2.y);
					}
					last = p2;
				}
				gc.setLineWidth(1);
				gc.setBackground(Constants.WHITE);
				gc.setForeground(Constants.BLACK);
				for (Point p2 : converted) {
					drawJoint(gc, p2);
				}
				converted.clear();
			}
		}
	}

	protected void drawPolygon(PolygonArea area, GC gc,
			ICoordsTransformer coordsTransformer) {
		Point tp1 = null, tp2;
		int[] converted = new int[2 * area.getNumPoints()];
		boolean skip = true;
		gc.setForeground(Constants.BLACK);
		gc.setBackground(Constants.WHITE);
		// first draw lines, then corners
		for (int i = 0; i < area.getNumPoints(); ++i) {
			if (skip) {
				skip = false;
				tp1 = coordsTransformer.location2client(area.getPoint(i));
				converted[2 * i] = tp1.x;
				converted[2 * i + 1] = tp1.y;
				continue;
			}
			tp2 = coordsTransformer.location2client(area.getPoint(i));
			converted[2 * i] = tp2.x;
			converted[2 * i + 1] = tp2.y;
			gc.drawLine(tp1.x, tp1.y, tp2.x, tp2.y);
			tp1 = tp2;
		}
		gc.drawLine(tp1.x, tp1.y, converted[0], converted[1]);
		gc.setAlpha(96);
		gc.setBackground(getAreaColor(area));
		gc.fillPolygon(converted);
		gc.setAlpha(255);
		gc.setBackground(Constants.WHITE);
		for (int i = 0; i < converted.length; i += 2) {
			gc.fillRectangle(converted[i] - 3, converted[i + 1] - 3, 6, 6);
			gc.drawRectangle(converted[i] - 3, converted[i + 1] - 3, 6, 6);
		}
		if (area.getShapeHint() == PolygonArea.Shape.CIRCLE) {
			drawJoint(gc, coordsTransformer.location2client(area.getCenter()));
		}
	}

	protected void drawSimpleAreas(GC gc, ICoordsTransformer coordsTransformer) {
		for (IArea area : mapData.getSimpleAreas()) {
			if (area instanceof PolygonArea) {
				drawPolygon((PolygonArea) area, gc, coordsTransformer);
			}
		}

	}

	private void error(String message, Throwable t) {
		String m = message + "\n" + (t != null ? t.getMessage() : "");
		if (errorCanBeDisplayed()) {
			// otherwise the messagebox would fail
			displayErrors();
			MessageBox mb = new MessageBox(shell, SWT.OK | SWT.ICON_ERROR);
			mb.setMessage(m);
			mb.setText("Error");
			mb.open();
		} else {
			messages.add(m);
		}
	}

	private boolean errorCanBeDisplayed() {
		return shell != null
				&& shell.getDisplay().getThread() == Thread.currentThread();
	}

	private void errorMessageBox(String message) {
		MessageBox mb = new MessageBox(shell, SWT.OK | SWT.ICON_ERROR);
		mb.setMessage(message);
		mb.setText("Error");
		mb.open();
	}

	protected void findLocationInfo(Location location) {
		selected = null;
		List<IArea> areas = areaQuadTree.getAreasPossibleOn(location.x,
			location.y);
		for (IArea area : areas) {
			if (area instanceof PolygonArea) {
				PolygonArea pa = (PolygonArea) area;
				if (pa.getShapeHint() == PolygonArea.Shape.CIRCLE) {
					if (isClose(location, pa.getCenter())) {
						selected = pa;
						return;
					}
				}
				for (int i = 0; i < pa.getNumPoints(); ++i) {
					if (isClose(location, pa.getPoint(i))) {
						selected = pa;
						return;
					}
				}
			}
			if (area.contains(location, true)) {
				selected = area;
				setNeedsUpdate(true, true);
			}
		}
		// this is O(n), but I hope that there will be not more
		// than hundreds of marks in each map
		for (IMark mark : mapData.getMarks()) {
			if (mark instanceof POI) {
				if (isClose(location, ((POI) mark).getLocation())) {
					selected = mark;
					return;
				}
			} else if (mark instanceof Path) {
				for (PathPoint p : ((Path) mark).getPoints()) {
					if (isClose(location, p.getLocation())) {
						selected = mark;
						return;
					}
				}
			}
		}
	}

	protected Color getAreaColor(PolygonArea area) {
		if (area == selected) {
			return Constants.BLUE_PALE;
		} else if (selected != null && selected instanceof ICompoundArea) {
			ICompoundArea ca = (ICompoundArea) selected;
			if (ca.containsExactly(area)) {
				return Constants.GREEN_PALE;
			} else if (ca.getExcludedArea().containsExactly(area)) {
				return Constants.RED_PALE;
			}
		}
		if (area == locked) {
			return Constants.CYAN;
		} else if (locked != null && locked instanceof ICompoundArea) {
			ICompoundArea ca = (ICompoundArea) locked;
			if (ca.containsExactly(area)) {
				return Constants.YELLOW;
			} else if (ca.getExcludedArea().containsExactly(area)) {
				return Constants.ORANGE;
			}
		}
		return Constants.GREY_PALE;
	}

	public IArea getLocked() {
		return locked;
	}

	public MapData getMapData() {
		return mapData;
	}

	public ICompoundArea getMapRootArea() {
		return mapData.getRootArea();
	}

	public IMark getMark(String name) {
		return mapData.getMark(name);
	}

	public boolean getNeedsFullUpdate() {
		boolean temp = needsFullUpdate;
		needsFullUpdate = false;
		return temp;
	}

	public boolean getNeedsUpdate() {
		boolean temp = needsUpdate;
		needsUpdate = false;
		return temp;
	}

	public Object getSelected() {
		return selected;
	}

	public Shell getShell() {
		return shell;
	}

	public ToolKit getToolKit() {
		return toolKit;
	}

	public WorkField getWorkField() {
		return workField;
	}

	@Override
	public void init(IEditorSite site, IEditorInput input)
			throws PartInitException {
		setSite(site);
		setInput(input);
		setPartName("EduMap Visual Editor");
		readFile();
	}

	public boolean isClose(Location p1, Location p2) {
		double distance2 = (p2.x - p1.x) * (p2.x - p1.x) + (p2.y - p1.y)
				* (p2.y - p1.y);
		double scale = mapData.getScale();
		return distance2 < 20 * scale * scale;
	}

	@Override
	public boolean isDirty() {
		return dirty;
	}

	public boolean isIdUsed(String id) {
		return (mapData.getAreas().get(id) != null) || (mapData.getMark(id) != null);
	}

	@Override
	public boolean isSaveAsAllowed() {
		return true;
	}

	@Override
	public void keyPressed(KeyEvent e) {
		Tool tool = getToolKit().getActiveTool();
		if (tool instanceof KeyListener) {
			((KeyListener) tool).keyPressed(e);
			setNeedsUpdate(tool.getNeedsUpdate(), tool.getNeedsFullUpdate());
		}
	}

	@Override
	public void keyReleased(KeyEvent e) {
		Tool tool = getToolKit().getActiveTool();
		if (tool instanceof KeyListener) {
			((KeyListener) tool).keyReleased(e);
			setNeedsUpdate(tool.getNeedsUpdate(), tool.getNeedsFullUpdate());
		}
	}

	private void loadContent() {
		if (mapData != null) {
			if (mapData.getPlan() != null) {
				IPath path = new org.eclipse.core.runtime.Path(
						mapData.getPlan());
				if (!path.isAbsolute()) {
					IFile file = ((IFileEditorInput) getEditorInput())
																		.getFile();
					path = file.getLocation().removeLastSegments(1)
								.append(path);
				}
				try {
					workField.setBackground(path.toString());
				} catch (Exception e) {
					error("Cannot open image \"" + path.toString() + "\"", null);
					return;
				}
			}
			workField.getCoordsTransformer().setParams(mapData.getScale(),
				mapData.getOriginX(), mapData.getOriginY());

			for (IArea a : mapData.getSimpleAreas()) {
				areaQuadTree.insert(a);
			}
		}
	}

	@Override
	public void mouseDoubleClick(MouseEvent e) {
		Tool tool = getToolKit().getActiveTool();
		if (tool instanceof MouseListener) {
			((MouseListener) tool).mouseDoubleClick(e);
			setNeedsUpdate(tool.getNeedsUpdate(), tool.getNeedsFullUpdate());
		}
	}

	@Override
	public void mouseDown(MouseEvent e) {
		findLocationInfo(new Location((double) e.x, (double) e.y));
		Tool tool = toolKit.getActiveTool();
		if (tool != null && tool instanceof MouseListener) {
			((MouseListener) tool).mouseDown(e);
			setNeedsUpdate(tool.getNeedsUpdate(), tool.getNeedsFullUpdate());
		}
	}

	@Override
	public void mouseMove(MouseEvent e) {
		Tool tool = getToolKit().getActiveTool();
		if (tool instanceof MouseMoveListener) {
			((MouseMoveListener) tool).mouseMove(e);
			setNeedsUpdate(tool.getNeedsUpdate(), tool.getNeedsFullUpdate());
		}
	}
	
	@Override
	public void mouseUp(MouseEvent e) {
		Tool tool = getToolKit().getActiveTool();
		if (tool instanceof MouseListener) {
			((MouseListener) tool).mouseUp(e);
			setNeedsUpdate(tool.getNeedsUpdate(), tool.getNeedsFullUpdate());
		}
	}

	protected void readFile() {
		try {
			IFile file;
			if (getEditorInput() instanceof IFileEditorInput) {
				file = ((IFileEditorInput) getEditorInput()).getFile();
				if (file.isSynchronized(IResource.DEPTH_INFINITE)) {
					setPartName(file.getFullPath().lastSegment());
					InputStream stream = file.getContents();
					mapData = new MapData();
					mapData.parseSource(stream, translator);
				} else {
					error(
						"Specified resource is out of sync with the file system.",
						null);
				}
			}
		} catch (CoreException e) {
			error("Problem occured during map file opening", e);
		} catch (Exception e) {
			error("Problem occured during map loading", e);
		}
	}

	public void removeArea(IArea area) {
		setDirty(true);
		if (!(area instanceof IComplexArea)) {
			areaQuadTree.remove(area);
		}
		mapData.remove(area);
	}

	public void removeMark(IMark mark) {
		mapData.removeMark(mark);
		setDirty(true);
	}

	public void setDirty() {
		setDirty(true);
	}

	protected void setDirty(boolean value) {
		dirty = value;
		firePropertyChange(PROP_DIRTY);
	}
	
	@Override
	public void setFocus() {
	}

	public void setLocked(IArea area) {
		this.locked = area;
	}

	public void setNeedsUpdate(boolean update, boolean full) {
		needsUpdate = needsUpdate | update;
		needsFullUpdate = needsFullUpdate | update & full;
	}

	public void setPlan(String src, double scale, double originX, double originY) {
		if (!src.isEmpty() && !src.equals(mapData.getPlan())) {
			mapData.setPlan(src);
			IPath path = new org.eclipse.core.runtime.Path(src);
			if (!path.isAbsolute()) {
				IFile file = ((IFileEditorInput) getEditorInput()).getFile();
				path = file.getLocation().removeLastSegments(1).append(path);
			}
			try {
				workField.setBackground(path.toString());
				workField.update(true);
			} catch (Exception e) {
				error("Cannot open image \"" + path.toString() + "\"", null);
				return;
			}
			setDirty(true);
		}
		if (scale != mapData.getScale() || originX != mapData.getOriginX()
				|| originY != mapData.getOriginY()) {
			Rescaler rescaler = new Rescaler(mapData.getScale(),
					mapData.getOriginX(), mapData.getOriginY(), scale, originX,
					originY);
			mapData.setScale(scale);
			mapData.setOriginX(originX);
			mapData.setOriginY(originY);
			mapData.rescale(rescaler);
			workField.getCoordsTransformer().setParams(scale, originX, originY);
			setDirty(true);
		}
	}

	public void setSelected(Object object) {
		selected = object;
	}
	
	/**
	 * This updates all information stored about area, e.g. those preprocessed
	 * for faster search.
	 * 
	 * @param area
	 */
	public void updateArea(IArea area) {
		mapData.updateArea(area);
		if (!(area instanceof IComplexArea)) {
			areaQuadTree.remove(area);
			areaQuadTree.insert(area);
		}
		setDirty(true);
	}
	
	public void updateMark(IMark mark) {
		mapData.updateMark(mark);
		setDirty(true);
	}
}
