package loquebot.util;

import java.util.ArrayList;

import cz.cuni.pogamut.MessageObjects.AddWeapon;
import cz.cuni.pogamut.MessageObjects.ItemType;
import loquebot.learn.WeaponStat;

/**
 * Enum of weapons with their valuable info. These values are used to tweak the
 * bot's behaviour based on weapons he carries and sees.
 *
 * <p>Note: Most of the values here were empirically settled on many combat
 * results. Therefore, the values are strongly dependent on this bot behaviour
 * and logic. You are welcome to try to use them for other bots, but be aware
 * that they do not represent real values from <i>UT.XEngine</i>.</p>
 *
 * <p>Also note that some mutators  (e.g. berserk, rulez modifier) can change
 * game rules (speed of projectiles, damage dealt, splash radius) and these
 * values would then no longer apply. All of these were settled empirically by
 * <i>brute-forcing</i> different values in many testing combats.</p>
 *
 * @author Juraj Simlovic [jsimlo@matfyz.cz]
 * @version Tested on Pogamut 2 platform version 1.0.5.
 */
public enum LoqueWeaponInfo
{
    /*========================================================================*/

    SHIELD_GUN ("XWeapons.ShieldGun", "hammer")
    {
        @Override protected void fill ()
        {
            //pickup is probably not going to change
            pickupWeaponAmount = 0;
            pickupAmmoAmount = 0;
            
        }

        
    },
    ASSAULT_RIFLE ("XWeapons.AssaultRifle", "handgun")
    {
        @Override protected void fill ()
        {
            //pickup is probably not going to change
            pickupWeaponAmount = 100;
            pickupAmmoAmount = 50;
        }

        
    },
    BIO_RIFLE ("XWeapons.BioRifle", "bio shit")
    {
        @Override protected void fill ()
        {
            //pickup is probably not going to change
            pickupWeaponAmount = 20;
            pickupAmmoAmount = 20;
            
        }

        
    },
    SHOCK_RIFLE ("XWeapons.ShockRifle", "purple laser")
    {
        @Override protected void fill ()
        {
            //pickup is probably not going to change
            pickupWeaponAmount = 20;
            pickupAmmoAmount = 10;
            
        }

        
    },
    MINIGUN ("XWeapons.Minigun", "mini-gun")
    {
        @Override protected void fill ()
        {
            //pickup is probably not going to change
            pickupWeaponAmount = 150;
            pickupAmmoAmount = 50;
           
        }

        
    },
    LINK_GUN ("XWeapons.LinkGun", "link shot")
    {
        @Override protected void fill ()
        {
            //pickup is probably not going to change
            pickupWeaponAmount = 70;
            pickupAmmoAmount = 50;
            
        }

        
    },
    FLAK_CANNON ("XWeapons.FlakCannon", "flak shrapnel")
    {
        @Override protected void fill ()
        {
            //pickup is probably not going to change
            pickupWeaponAmount = 15;
            pickupAmmoAmount = 10;
           
        }

        
    },
    ROCKET_LAUNCHER ("XWeapons.RocketLauncher", "rocketry")
    {
        @Override protected void fill ()
        {
            //pickup is probably not going to change
            pickupWeaponAmount = 12;
            pickupAmmoAmount = 9;
            
        }

        
    },
    LIGHTNING_GUN ("XWeapons.SniperRifle", "piece of lightning")
    {
        @Override protected void fill ()
        {
            //pickup is probably not going to change
            pickupWeaponAmount = 15;
            pickupAmmoAmount = 10;
            
        }

        
    },
    SNIPER_RIFLE ("UTClassic.ClassicSniperRifle", "classic one")
    {
        @Override protected void fill ()
        {
            //pickup is probably not going to change
            pickupWeaponAmount = 15;
            pickupAmmoAmount = 10;
            
        }

        
    },
    REDEEMER ("XWeapons.Redeemer", "RE-DEEM-ER")
    {
        @Override protected void fill ()
        {
            //pickup is probably not going to change
            pickupWeaponAmount = 1;
            pickupAmmoAmount = 1;
            
        }

    },
    // and a fail-safe value..
    UNKNOWN ("???", "gun")
    {
        @Override protected void fill () {}
        @Override public boolean altFire (double distance) { return false; }
        @Override public int inventoryScore (int skill, int ammo) { return 0; }
        @Override public int generalScore (int skill) { return 0; }
        @Override public int combatScore (int skill, double distance) { return 0; }
    },
    // used for calculation of score in weapon stat output
    TEMP ("temp", "temp")
    {
        @Override protected void fill () {}
        
    };

    /*========================================================================*/

    /** Name of the weapon in UT engine. */
    public String name;

    /** Name of the weapon in speech. */
    public String title;

    /**
     * Whether this weapon is a minor one. Should the bot be fighting with this
     * weapon, it should try to pick something better ASAP.
     */
    public boolean minorWeapon = true;

    /**
     * How much ammo this weapon adds upon pickup.
     */
    public int pickupWeaponAmount = 0;
    /**
     * How much ammo this weapon adds upon pickup.
     */
    public int pickupAmmoAmount = 0;

    
    public int priDamage = 0;
    public int priSelfDamage = 0;
    /**
     * Projectile speed of primary fire mode.
     * Defaults to value 9e+9 for speeding bullets.
     */
    public double priProjectileSpeed = 2000;
    /**
     * Max distance to which the weapon can shoot with primary fire mode.
     * Defaults to 9e+9, when there is no range limit.
     */
    public double priMaxRange = 9e+9;
    /**
     * Splash damage radius of primary fire mode. Zero if no splash damage.
     */
    public double priSplashRadius = 0;
    /**
     * Whether the weapon is better to be charged, even if no enemy is around.
     */
    public boolean priKeepShooting = false;

    /**
     * Amount of ammo, when the weapon is to be considered empty.
     */
    public int priLowAmmo = 3;

    /**
     * Most effective distance from the enemy while using primary fire mode.
     * This value is used to keep desired and working distance from enemies.
     */
    public double priIdealCombatRange = 0;
    /**
     * Most effective amount of strafing to side.
     */
    public double priStrafingAmount = 100;
    /**
     * Effective distance of primary fire mode. Beyond this distance, the
     * weapon might start to loose some glance. This value is used as cut-off
     * point for aim-ahead predictions based on enemy distance.
     */
    public double priEffectiveDistance = 9e+9;
    /**
     * How much time it is reasonable to let the projectile of primary fire
     * mode glide until expected impact. This value is used as cut-off point
     * for aim-ahead predictions based on enemy velocity and distance.
     */
    public double priReasonableFlight = .8;
    /**
     * How much static aim-ahead distance should be used for primary fire
     * mode. This value is used for aim-ahead predictions of speeding
     * bullets.
     */
    public double priStaticAimAhead = 10;
    /**
     * How much static z-axis correction should be used for primary fire
     * mode. This value is used for headshots and predictions.
     */
    public double priStaticZCorrection = 0;

    /**
     * Whether server may assist with aim corrections for primary fire mode.
     * This is reasonable only for speeding bullets, since aim correction
     * spoils precisely calculated aim-ahead predictions for projectiles.
     */
    public boolean priHelpAim = false;

    
    
    
    public int altDamage = 0;
    public int altSelfDamage = 0;
    /**
     * Projectile speed of alternate fire mode.
     * Defaults to value 9e+9 for speeding bullets.
     */
    public double altProjectileSpeed = 2000;
    /**
     * Max distance to which the weapon can shoot with alternate fire mode.
     * Defaults to 9e+9, when there is no range limit.
     */
    public double altMaxRange = 9e+9;
    /**
     * Splash damage radius of alternate fire mode. Zero if no splash damage.
     */
    public double altSplashRadius = 0;
    /**
     * Whether the weapon is better to be charged, even if no enemy is around.
     */
    public boolean altKeepShooting = false;

    /**
     * Amount of ammo, when the weapon is to be considered empty.
     */
    public int altLowAmmo = 3;

    /**
     * Most effective distance from the enemy while using primary fire mode.
     * This value is used to keep desired and working distance from enemies.
     */
    public double altIdealCombatRange = 0;
    /**
     * Most effective amount of strafing to side.
     */
    public double altStrafingAmount = 100;
    /**
     * Effective distance of alternate fire mode. Beyond this distance, the
     * weapon might start to loose some glance. This value is used as cut-off
     * point for aim-ahead predictions based on enemy distance.
     */
    public double altEffectiveDistance = 9e+9;
    /**
     * How much time it is reasonable to let the projectile of alternate fire
     * mode glide until expected impact. This value is used as cut-off point
     * for aim-ahead predictions based on enemy velocity and distance.
     */
    public double altReasonableFlight = .8;
    /**
     * How much static aim-ahead distance should be used for alternate fire
     * mode. This value is used for aim-ahead predictions of speeding
     * bullets.
     */
    public double altStaticAimAhead = 10;
    /**
     * How much static z-axis correction should be used for alternate fire
     * mode. This value is used for headshots and predictions.
     */
    public double altStaticZCorrection = 0;

    /**
     * Whether server may assist with aim corrections for alternate fire mode.
     * This is reasonable only for speeding bullets, since aim correction
     * spoils precisely calculated aim-ahead predictions for projectiles.
     */
    public boolean altHelpAim = false;

    /*========================================================================*/
    //combat weapon selection statistics
    public static int bestScoreTreshold = 35;
    public static int goodScoreTreshold = 20;
    public static int poorScoreTreshold = 10;
    
    //powerup statistics
    public static int reasonableInventoryScoreTreshold = 25;
    public static int enoughInventoryScoreTreshold = 50;
    
    //learn weapons threshold
    public static int minorTreshold = 30;
    
    /*========================================================================*/

    /**
     * Function used for filling-up enum item properties. Used in constructor.
     */
    protected abstract void fill ();

    /**
     * Enum item constructor.
     * @param name UT name of the item.
     * @param title Speech name of the item.
     */
    private LoqueWeaponInfo (String name, String title)
    {
        // set the name and title
        this.name = name;
        this.title = title;
        // fill in some values
        fill ();
    }

    /*========================================================================*/


       
    /**
     * Decides, whether to use alternate fire mode.
     * @param distance Distance at which the destination is located.
     * Might be -1 if no distance is to be considered while picking the gun.
     * @return Returns true, if alternate mode is recommended.
     */
    public boolean altFire (double distance){
        
        
        //check max distance
        boolean alt = distance < this.altMaxRange;
        boolean pri = distance < this.priMaxRange;
        
        //check if damage is 0
        if(alt && this.altDamage == 0) alt = false;
        if(pri && this.priDamage == 0) pri = false;
        
        
        
        
        //do we have a winner?
        if(alt != pri) return alt;
        
        
        
        
        //consider the ideal distance and damage delt
        double rangeDif = 0;
        rangeDif = this.altIdealCombatRange - this.priIdealCombatRange;
        if(Math.abs(rangeDif) < 150 && distance > -1){   //simmilar combat range
            //consider effective distance
            int aDamage = (this.altEffectiveDistance < distance && this.altDamage > 30) ? 30 : this.altDamage;
            int pDamage = (this.priEffectiveDistance < distance && this.priDamage > 30) ? 30 : this.priDamage;
            
            //consider splash damage
            if(this.altSplashRadius > distance) aDamage -= this.altSelfDamage;
            if(this.priSplashRadius > distance) pDamage -= this.priSelfDamage;
            
            alt = aDamage - 5 > pDamage;
            pri = aDamage + 5 < pDamage;
        }else if(Math.abs(rangeDif) > 150 && distance > -1){
            alt = Math.abs(this.altIdealCombatRange - distance) + 100 < Math.abs(this.priIdealCombatRange - distance);
            pri = Math.abs(this.altIdealCombatRange - distance) - 100 > Math.abs(this.priIdealCombatRange - distance);            
        }
        
        
        
        
        //do the choice
        if(alt != pri){
            return alt;
        }else{
            return Math.random()>0.5;
        }
    }

    /**
     * Determines inventory weapon score based on the skill and current ammo.
     * @param skill Skill level of the bot.
     * @param ammo Current amount of ammo.
     * @return Comparable score of the weapon.
     */
    public int inventoryScore (int skill, int ammo)
    {
        return ammo == 0 ? 0 : generalScore(skill);
    }

    /**
     * Determines general weapon score based only on the skill.
     * @param skill Skill level of the bot.
     * @return Comparable score of the weapon.
     */
    public int generalScore (int skill){
        return Math.max(this.altDamage - this.altSelfDamage/2, this.priDamage - this.priSelfDamage/2);
    }

    /**
     * Determines combat weapon score based on the skill and target distance.
     * @param skill Skill level of the bot.
     * @param distance Distance, for which the weapon is being chosen.
     * Might be -1 if no distance is to be considered while picking the gun.
     * @return Comparable score of the weapon.
     */
    public int combatScore (int skill, double distance){
        
        int damage = 0;
                
        //should we use alt or normal fire?
        boolean alt = this.altFire(distance);
        
        if(alt){
            
            //check max distance and if damage is 0
            if(distance > this.altMaxRange || this.altDamage == 0) return 0;
            
            //consider effective distance
            damage = (this.altEffectiveDistance < distance && this.altDamage > 30) ? 30 : this.altDamage;
            
            //consider splash damage
            if(this.altSplashRadius > distance) damage -= this.altSelfDamage;
            
        }else{
            
            
            //check max distance and if damage is 0
            if(distance > this.priMaxRange || this.priDamage == 0) return 0;
            
            //consider effective distance
            damage = (this.priEffectiveDistance < distance && this.priDamage > 30) ? 30 : this.priDamage;
            
            //consider splash damage
            if(this.priSplashRadius > distance) damage -= this.priSelfDamage;
            
            
        }
        
        return Math.max(damage, 0);
    }

    /*========================================================================*/

    /**
     * Chooses among given list of weapons. Considers skill level and distance.
     *
     * <h4>Future</h4>
     *
     * Decission based on enemy weapon is not implemented right now. The main
     * idea is to use (for example) a shield gun against a rocket launcher,
     * since these weapons require different fighting distances and the shield
     * gun might be better choice for engaging in this case, forcing the enemy
     * to rearm or hurt himself while hurting us.
     *
     * @param skill Skill level of the bot.
     * @param weapons List of weapons to choose from.
     * @param enemyDistance Distance, for which the weapon is being chosen.
     * Might be -1 if no distance is to be considered while picking the gun.
     * @param enemyWeapon Weapon, the enemy holds in his hands. Might be null.
     * @return Chosen weapon; or null upon no possible choice.
     */
    public static AddWeapon chooseWeapon (int skill, ArrayList<AddWeapon> weapons, double enemyDistance, String enemyWeapon)
    {
        AddWeapon chosen = null;
        LoqueWeaponInfo chosenInfo = null;

        // run through all weapons
        for (AddWeapon weapon : weapons)
        {
            // is there at least some ammo ready for this weapon?
            if (weapon.currentAmmo <= 1)
                continue;

            // is this a shield gun? no shieldguns here..
//            if (weapon.weaponType == ItemType.SHIELD_GUN)
//                continue;

            // retreive info about that weapon
            LoqueWeaponInfo weaponInfo = getInfo(weapon.weaponType);

            // is there enough ammo ready for this weapon?
            if (weapon.currentAmmo < weaponInfo.priLowAmmo)
                continue;

            // do we have a weapon to compare it with?
            if (chosen == null)
            {
                chosen = weapon;
                chosenInfo = weaponInfo;
            }
            else
            {
                // is the weapon better for combat?
                if (
                    weaponInfo.combatScore(skill, enemyDistance)
                    >= chosenInfo.combatScore(skill, enemyDistance)
                )
                {
                    // oukay then, choose this one..
                    chosen = weapon;
                    chosenInfo = weaponInfo;
                }
            }
        }

        // did we choose a weapon?
        if (chosen != null)
            return chosen;

        // run through all weapons again
        for (AddWeapon weapon : weapons)
        {
            LoqueWeaponInfo weaponInfo = getInfo(weapon.weaponType);

            // is there at least some ammo ready for this weapon?
            if (weapon.currentAmmo <= 1)
                continue;

            // do we have a weapon to compare it with?
            if (chosen == null)
            {
                chosen = weapon;
                chosenInfo = weaponInfo;
            }
            else
            {
                // is the weapon better for combat?
                if (
                    weaponInfo.combatScore(skill, enemyDistance)
                    > chosenInfo.combatScore(skill, enemyDistance)
                )
                {
                    // oukay then, choose this one..
                    chosen = weapon;
                    chosenInfo = weaponInfo;
                }
            }
        }

        // return chosen weapon; or null, if none..
        return chosen;
    }

    /*========================================================================*/

    /**
     * Retreives correct weapon info by given weapon item type.
     * @param itemType Item type of which info is requested.
     * @return Requested weapon info.
     */
    public static LoqueWeaponInfo getInfo (ItemType itemType)
    {
        // return by type
        switch (itemType)
        {
            case SHIELD_GUN: return SHIELD_GUN;
            case ASSAULT_RIFLE: return ASSAULT_RIFLE;
            case BIO_RIFLE: return BIO_RIFLE;
            case SHOCK_RIFLE: return SHOCK_RIFLE;
            case MINIGUN: return MINIGUN;
            case LINK_GUN: return LINK_GUN;
            case FLAK_CANNON: return FLAK_CANNON;
            case ROCKET_LAUNCHER: return ROCKET_LAUNCHER;
            case LIGHTNING_GUN: return LIGHTNING_GUN;
            case SNIPER_RIFLE: return SNIPER_RIFLE;
            case REDEEMER: return REDEEMER;
            default: return UNKNOWN;
        }
    }

    /**
     * Retreives correct weapon info by given weapon name.
     * @param weapon Weapon name of which info is requested.
     * @return Requested weapon info.
     */
    public static LoqueWeaponInfo getInfo (AddWeapon weapon)
    {
        // did we get a weapon?
        if (weapon == null)
            return UNKNOWN;

        // return by type
        return getInfo (weapon.weaponType);
    }

    /**
     * Retreives correct weapon info by given weapon name.
     * @param weapon Weapon name of which info is requested.
     * @return Requested weapon info.
     */
    public static LoqueWeaponInfo getInfo (String weapon)
    {
        // find the appropriate string..
        for (LoqueWeaponInfo info : values())
            if (info.name.compareTo(weapon) == 0)
                return info;

        // return unknown
        return UNKNOWN;
    }
    
    /**
     * Retreives correct weapon info by given weapon name.
     * @param weapon Weapon name of which info is requested.
     * @return Requested weapon info.
     */
    public static LoqueWeaponInfo getInfoEnum (String weapon)
    {
        // find the appropriate string..
        for (LoqueWeaponInfo info : values())
            if (info.toString().compareTo(weapon) == 0)
                return info;

        // return unknown
        return UNKNOWN;
    }

    /*========================================================================*/

    public static void Test ()
    {
        TestWeapon (ASSAULT_RIFLE, 400/*100*/);
        TestWeapon (BIO_RIFLE, 50/*20*/);
        TestWeapon (SHOCK_RIFLE, 50/*20*/);
        TestWeapon (MINIGUN, 300/*150*/);
        TestWeapon (LINK_GUN, 220/*70*/);
        TestWeapon (FLAK_CANNON, 35/*15*/);
        TestWeapon (ROCKET_LAUNCHER, 30/*12*/);
        TestWeapon (LIGHTNING_GUN, 40/*15*/);
        TestWeapon (SNIPER_RIFLE, 35/*15*/);
    }

    public static void TestWeapon (LoqueWeaponInfo w, int ammo)
    {
        int[] skills = {1,2,3,4,5,6,7};

        System.out.printf("%30s", w.name + ": ");

        for (int skill : skills)
            System.out.printf("%4d, ", w.inventoryScore(skill, ammo));

        System.out.println ("ammo " + ammo);
    }
    
    
    /*========================================================================*/
    
    
    
    public void readWeaponStat(WeaponStat stat){
        this.altDamage = stat.altDamage;
        this.altEffectiveDistance = stat.altEffectiveDistance;
        this.altIdealCombatRange = stat.altIdealCombatRange;
        this.altMaxRange = stat.altMaxRange;
        this.altSelfDamage = stat.altSelfDamage;
        this.altSplashRadius = stat.altSplashRadius;
        
        this.minorWeapon = stat.minorWeapon;
        
        this.priDamage = stat.priDamage;
        this.priEffectiveDistance = stat.priEffectiveDistance;
        this.priIdealCombatRange = stat.priIdealCombatRange;
        this.priMaxRange = stat.priMaxRange;
        this.priSelfDamage = stat.priSelfDamage;
        this.priSplashRadius = stat.priSplashRadius;
        
    }
    
    
    public static void loadWeaponInfo(String filename){
        try {
          //use buffering, reading one line at a time
          //FileReader always assumes default encoding is OK!
          java.io.BufferedReader dataFile =  new java.io.BufferedReader(new java.io.FileReader(filename)); 
          try {
            String line = null; //not declared within while loop
            WeaponStat stat = null;
            
            while (( line = dataFile.readLine()) != null){
                
                stat = WeaponStat.valueOf(line);
                
                LoqueWeaponInfo info = LoqueWeaponInfo.getInfoEnum(stat.name);
                
                info.readWeaponStat(stat);
                
            }
          }
          finally {
            dataFile.close();
          }
        }
        catch (java.io.IOException ex){
          filename = "";
        }
        
        
        
        //compute weapon statistics
        int maxScore = 0;
        int minScore = Integer.MAX_VALUE;
        int sumScore = 0;
        for(LoqueWeaponInfo wInfo:LoqueWeaponInfo.values()){
            int cScore = wInfo.generalScore(3);
            
            maxScore = maxScore < cScore ? cScore : maxScore;
            minScore = minScore > cScore && cScore > 0 ? cScore : minScore;
            sumScore += cScore;
        }
        
        int difScore = maxScore - minScore;
        
        LoqueWeaponInfo.bestScoreTreshold = minScore + 3 * difScore / 5;
        LoqueWeaponInfo.goodScoreTreshold = minScore + 2 * difScore / 5;
        LoqueWeaponInfo.poorScoreTreshold = minScore + 1 * difScore / 5;
        
        LoqueWeaponInfo.reasonableInventoryScoreTreshold = 1 * sumScore / 4;
        LoqueWeaponInfo.enoughInventoryScoreTreshold = 2 * sumScore / 4;
        
        LoqueWeaponInfo.minorTreshold = minScore + difScore / 2;
    }
    
    
    
    
    
    
    
    
}
