package loquebot.memory;

import java.util.logging.Logger;

import cz.cuni.pogamut.Client.AgentBody;

import cz.cuni.pogamut.MessageObjects.MessageObject;
import cz.cuni.pogamut.MessageObjects.MessageType;
import cz.cuni.pogamut.MessageObjects.Triple;

import cz.cuni.pogamut.MessageObjects.Self;
import cz.cuni.pogamut.MessageObjects.Spawn;
import cz.cuni.pogamut.MessageObjects.Player;
import cz.cuni.pogamut.MessageObjects.BotKilled;
import cz.cuni.pogamut.MessageObjects.PlayerKilled;

import loquebot.Main;
import loquebot.util.LoqueListener;

/**
 * Responsible for listening to the messages and managing main agent info.
 *
 * @author Juraj Simlovic [jsimlo@matfyz.cz]
 * @version Tested on Pogamut 2 platform version 1.0.5.
 */
public class LoqueSelf
{
    /**
     * Last received self message.
     */
    private Self selfMsg;

    /*========================================================================*/

    /**
     * Retreives unique ID of the agent in the game.
     * @return Agent ID.
     */
    public int getID ()
    {
        return selfMsg.ID;
    }

    /**
     * Compares given ID to the agent's one.
     * @param ID ID to be compared.
     * @return True, if the ID is the same as agent's.
     */
    public boolean hasID (int ID)
    {
        return selfMsg.ID == ID;
    }

    /*========================================================================*/

    /**
     * Retreives unique UnrealID of the agent in the game.
     * @return Agent UnrealID.
     */
    public String getUnrealID ()
    {
        return selfMsg.UnrealID;
    }

    /**
     * Compares given UnrealID to the agent's one.
     * @param UnrealID UnrealID to be compared.
     * @return True, if the UnrealID is the same as agent's.
     */
    public boolean hasUnrealID (String UnrealID)
    {
        return selfMsg.UnrealID.equals(UnrealID);
    }

    /*========================================================================*/

    /**
     * Retreives name of the agent in the game.
     * @return Agent name.
     */
    public String getName ()
    {
        return selfMsg.name;
    }

    /*========================================================================*/

    /** Team number of <i>red</i> team. */
    public static final int TEAM_RED = 0;
    /** Team number of <i>blue</i> team. */
    public static final int TEAM_BLUE = 0;
    /** Team number of <i>green</i> team. */
    public static final int TEAM_GREEN = 0;
    /** Team number of <i>gold</i> team. */
    public static final int TEAM_GOLD = 0;
    /** Team number of <i>no</i> team. */
    public static final int TEAM_NONE = 255;

    /**
     * Retreives team number of the agent in the game.
     * @see #TEAM_NONE
     * @see #TEAM_RED
     * @see #TEAM_BLUE
     * @see #TEAM_GREEN
     * @see #TEAM_GOLD
     * @return Agent team.
     */
    public int getTeam ()
    {
        return selfMsg.team;
    }

    /**
     * Compares given team to the agent's one.
     * @param team Team to be compared.
     * @return True, if the team is the same as agent's.
     */
    public boolean hasSameTeam (int team)
    {
        return (team != TEAM_NONE) && (team == selfMsg.team);
    }

    /*========================================================================*/

    /**
     * Retreives absolute agent location within the map.
     * @return Agent location within the map.
     */
    public Triple getLocation ()
    {
        return selfMsg.location;
    }

    /**
     * Computes space distance of given location from the agent.
     * @param location Location to be computed upon.
     * @return Space distance of given location from the agent.
     */
    public double getSpaceDistance (Triple location)
    {
        return Triple.distanceInSpace (location, selfMsg.location);
    }

    /**
     * Computes planar distance of given location from the agent.
     * @param location Location to be computed upon.
     * @return Planar distance of given location from the agent.
     */
    public double getPlanarDistance (Triple location)
    {
        return Triple.distanceInPlane (location, selfMsg.location);
    }

    /**
     * Computes vertical distance of given location from the agent.
     * @param location Location to be computed upon.
     * @return Vertical distance of given location from the agent.
     */
    public double getVerticalDistance (Triple location)
    {
        return (location.z >= selfMsg.location.z)
            ? (location.z - selfMsg.location.z)
            : (selfMsg.location.z - location.z);
    }

    /**
     * Computes vertical difference of given location from the agent.
     * @param location Location to be computed upon.
     * @return Vertical difference of given location from the agent. Value is
     * negative, if the location is beneath agent and positive, if it is above.
     */
    public double getVerticalDifference (Triple location)
    {
        return location.z - selfMsg.location.z;
    }

    /*========================================================================*/

    /**
     * Retreives absolute agent rotation as (x=yaw, y=roll, z=pitch).
     * @return Absolute agent rotation.
     */
    public Triple getRotation ()
    {
        return selfMsg.rotation;
    }

    /*========================================================================*/

    /**
     * Retreives current agent velocity as vector of movement.
     * @return Current agent velocity.
     */
    public Triple getVelocity ()
    {
        return selfMsg.velocity;
    }

    /*========================================================================*/

    /**
     * Retreives current agent health.
     * @return Current agent health. Ranges from 0 to 199.
     */
    public int getHealth ()
    {
        return selfMsg.health;
    }

    /**
     * Retreives current agent armor.
     * @return Current agent armor. Ranges from 0 to 150.
     */
    public int getArmor ()
    {
        return selfMsg.armor;
    }

    /**
     * Retreives current agent adrenaline.
     * @return Current agent adrenaline. Ranges from 0 to 100.
     */
    public int getAdrenaline ()
    {
        return (int) selfMsg.adrenaline;
    }

    /*========================================================================*/

    /**
     * Retreives UnrealID of weapon, which the agent is holding.
     * @return UnrealID of weapon, which the agent is holding.
     */
    public String getWeaponUnrealID ()
    {
        return selfMsg.weapon;
    }

    /*========================================================================*/

    /**
     * Retreives current status of ammo for the current weapon.
     * @return Current status of ammo for the current weapon.
     */
    public int getCurrentAmmo ()
    {
    	if(selfMsg == null) return -1;
        return selfMsg.currentAmmo;
    }

    /**
     * Retreives current status of alternative ammo for the current weapon.
     * @return Current status of alternative ammo for the current weapon.
     */
    public int getCurrentAltAmmo ()
    {
    	if(selfMsg == null) return -1;
        return selfMsg.currentAltAmmo;
    }

    /*========================================================================*/

    /**
     * Tells, wether the agent is shooting right now.
     * @return True, if the agent is shooting right now.
     */
    public boolean isShooting ()
    {
        return selfMsg.shooting;
    }

    /**
     * Tells, whether the agent is shooting with alternate fire mode.
     * @return True, if the agent is shooting with alternate fire mode.
     */
    public boolean isAltShooting ()
    {
        return (selfMsg.altFiring > 0);
    }

    /*========================================================================*/

    /**
     * How many times the agent scored a frag.
     */
    private int totalKills = 0;

    /**
     * Tells, how many times the agent scored a frag.
     * @return Total number of scored frags.
     */
    public int getTotalKills ()
    {
        return totalKills;
    }

    /**
     * How many times the agent got fragged.
     */
    private int totalDeaths = 0;

    /**
     * Tells, how many times the agent got fragged.
     * @return Total number of deaths, suicides not included.
     */
    public int getTotalDeaths ()
    {
        return totalDeaths;
    }

    /**
     * How many times the agent suicided.
     */
    private int totalSuicides = 0;

    /**
     * Tells, how many times the agent suicided.
     * @return Total number of suicides.
     */
    public int getTotalSuicides ()
    {
        return totalSuicides;
    }

    /*========================================================================*/

    /**
     * How many combats did the agent got into? This value is increased each
     * time the agent picks a new enemy to fight with. Loosing an enemy from
     * sight or respawning always cause a new combat and therefore increase
     * of this value.
     */
    private int totalCombats = 0;

    /**
     * Tells, how many times the agent got into combat.
     * @return Total number of combats.
     */
    public int getTotalCombats ()
    {
        return totalCombats;
    }

    /**
     * Increments total number of combats by one. Loosing an enemy from sight
     * or respawning should always cause a new combat and therefore increase.
     */
    public void incTotalCombats ()
    {
        totalCombats++;
    }

    /*========================================================================*/

    /**
     * Random number for the bot's name. Might be useful, when more than one
     * Loque is playing within the same game. See {@link updateName() }.
     */
    private int nameNumber = (int) (Math.random () * 9 + 1);

    /**
     * Updates bot name according to current values.
     */
    private void updateAgentName ()
    {
        // make the name
        String agentName = "player716";

        // add additional info?
        if (main.optionInfoInName)
        {
            // combats-to-kills rate
            if (totalCombats > 0)
                agentName += ", " + (((totalKills * 10000) / totalCombats) / 100.0) + "%";
            // kills vs. suicides vs. combats
            agentName += ", " + totalKills + "/" + totalSuicides + "/" + totalCombats;
        }

        // configure bot name - not for botprize
        //body.configureName(agentName);
    }

    /*========================================================================*/

    /**
     * Listening class for messages from engine.
     */
    private class Listener extends LoqueListener
    {
        /**
         * Periodic info about the agent.
         * @param msg Message to handle.
         */
        private void msgSelf (Self msg)
        {
            selfMsg = msg;
        }

        /**
         * Agent just spawned into the game.
         * @param msg Message to handle.
         */
        private void msgSpawn (Spawn msg)
        {
            updateAgentName ();
        }

        /**
         * Someone fragged somebody.
         * @param msg Message to handle.
         */
        private void msgPlayerKilled (PlayerKilled msg)
        {
            // did this agent fragged?
            if (hasID (msg.killerID))
            {
                // count it
                totalKills++;
                // get frag info
                Player info = memory.players.getPlayer (msg.playerID);
                String name = (info == null) ? "null" : info.name;
                // log the frag
                log.severe ("Self.Listener: FRAGED " + name + ", id " + msg.playerID + ", total kills " + totalKills);
            }
        }

        /**
         * Agent has died. Somehow.
         * @param msg Message to handle.
         */
        private void msgBotKilled (BotKilled msg)
        {
            // did the agent suicided?
            if (hasID (msg.killerID))
            {
                // count it
                totalSuicides++;
                // get kill info
                Player info = memory.players.getPlayer (msg.killerID);
                String name = (info == null) ? "null" : info.name;
                // log the suicide
                log.severe ("Self.Listener: SUICIDE by " + name + ", id " + msg.killerID + ", total suicides " + totalSuicides);
            }
            // no the agent got fragged
            else
            {
                // count it
                totalDeaths++;
                // get kill info
                Player info = memory.players.getPlayer (msg.killerID);
                String name = (info == null) ? "null" : info.name;
                // log the funeral
                log.severe ("Self.Listener: RIPPED by " + name + ", id " + msg.killerID + ", total deaths " + totalDeaths);
            }
        }

        /*========================================================================*/

        /**
         * Message switch.
         * @param msg Message to handle.
         */
        protected void processMessage (MessageObject msg)
        {
            switch (msg.type)
            {
                case SELF:
                    msgSelf ((Self) msg);
                    return;
                case SPAWN:
                    msgSpawn ((Spawn) msg);
                    return;
                case PLAYER_KILLED:
                    msgPlayerKilled ((PlayerKilled) msg);
                    return;
                case BOT_KILLED:
                    msgBotKilled ((BotKilled) msg);
                    return;
            }
        }

        /**
         * Constructor: Signs up for listening.
         */
        private Listener ()
        {
            body.addTypedRcvMsgListener (this, MessageType.SELF);
            body.addTypedRcvMsgListener (this, MessageType.SPAWN);
            body.addTypedRcvMsgListener (this, MessageType.PLAYER_KILLED);
            body.addTypedRcvMsgListener (this, MessageType.BOT_KILLED);
        }
    }

    /** Listener. */
    private LoqueListener listener;

    /*========================================================================*/

    /** Agent's main. */
    protected Main main;
    /** Loque memory. */
    protected LoqueMemory memory;
    /** Agent's body. */
    protected AgentBody body;
    /** Agent's log. */
    protected Logger log;

    /*========================================================================*/

    /**
     * Constructor.
     * @param main Agent's main.
     * @param memory Loque memory.
     */
    public LoqueSelf (Main main, LoqueMemory memory)
    {
        // setup reference to agent
        this.main = main;
        this.memory = memory;
        this.body = main.getBody ();
        this.log = main.getLogger ();

        // create listener
        this.listener = new Listener ();
    }
}