package loquebot.memory;

import java.util.logging.Logger;

import cz.cuni.pogamut.Client.AgentMemory;
import cz.cuni.pogamut.Client.AgentBody;

import cz.cuni.pogamut.MessageObjects.MessageObject;
import cz.cuni.pogamut.MessageObjects.MessageType;

import cz.cuni.pogamut.MessageObjects.Player;
import cz.cuni.pogamut.MessageObjects.Self;
import cz.cuni.pogamut.MessageObjects.Spawn;

import loquebot.Main;
import loquebot.util.LoqueListener;


/**
 * Responsible for listening to the messages and feeding agent senses.
 *
 * @author Juraj Simlovic [jsimlo@matfyz.cz]
 * @version Tested on Pogamut 2 platform version 1.0.5.
 */
public class LoqueMemory
{
    /**
     * Loque self info. Part of memory, which contains info about various agent
     * statuses, conditions and whereabouts.
     */
    public LoqueSelf self;
    /**
     * Loque senses. Part of memory, which drives senses about what the agent
     * actually sees, feels, hears, etc.
     */
    public LoqueSenses senses;
    /**
     * Loque inventory. Part of memory, which contains info about current agent
     * weaponry.
     */
    public LoqueInventory inventory;
    /**
     * Loque map items. Part of memory, which contains info about pickable items
     * all around the map.
     */
    public LoqueMapItems items;
    /**
     * Loque players. Part of memory, which contains info about other players.
     */
    public LoquePlayers players;

    /*========================================================================*/

    /**
     * Tells, whether the given player is an enemy or not.
     *
     * @param player The tested player.
     * @return Returns true, if the player is on other team.
     */
    public boolean isEnemy (Player player)
    {
        return !self.hasSameTeam (player.team);
    }

    /*========================================================================*/

    /**
     * Fades various memory info out by one step.
     * Should be called upon each agent logic loop.
     */
    public void fadeAllOut ()
    {
        senses.fadeAllOut ();

        synchronized (this)
        {
            // the logic is still running..
            logicTimout = 0;
        }
    }

    /*========================================================================*/

    /**
     * Counter checking, whether the logic is still running.
     */
    private int logicTimout = 0;

    /*========================================================================*/

    /**
     * Listening class for messages from engine.
     */
    private class Listener extends LoqueListener
    {
        /**
         * Agent just spawned into the game.
         * @param msg Message to handle.
         */
        private void msgSpawn (Spawn msg)
        {
            // break the logs
            log.severe ("Memory.Listener: SPAWNED ==================================================");
        }

        /**
         * Periodic info about the agent.
         * @param msg Message to handle.
         */
        private void msgSelf (Self msg)
        {
            synchronized (LoqueMemory.this)
            {
                // check whether the logic is still safe and sound
                if (logicTimout++ > main.logicFrequency * 10)
                {
                    // panic, panic, panic...
                    log.severe("Memory.Listener: LOGIC NOT RUNNING: isRunning() returns " + main.isRunning () + ", logicTimout is " + logicTimout);
                    // try to make a point about it
                    body.sendGlobalMessage("MY LOGIC'S DEAD !! Why, oh why? Dead..");
                    // make it visible on the bot?
                    body.getCommunicationStateFlag().setFlag(cz.cuni.pogamut.communication.CommunicationState.AWAITING_LOGIC);
                }
            }
        }

        /*========================================================================*/

        /**
         * Message switch.
         * @param msg Message to handle.
         */
        protected void processMessage (MessageObject msg)
        {
            switch (msg.type)
            {
                case SPAWN:
                    msgSpawn ((Spawn) msg);
                    return;
                case SELF:
                    msgSelf ((Self) msg);
                    return;
            }
        }

        /**
         * Constructor: Signs up for listening.
         */
        private Listener ()
        {
            body.addTypedRcvMsgListener (this, MessageType.SELF);
            body.addTypedRcvMsgListener (this, MessageType.SPAWN);
        }
    }

    /** Memory listener. */
    private LoqueListener listener;

    /*========================================================================*/

    /** Agent's main. */
    private Main main;
    /** Agent's memory. */
    private AgentMemory memory;
    /** Agent's body. */
    private AgentBody body;
    /** Agent's log. */
    private Logger log;
    /** Agent's platformLog. */
    private Logger platformLog;

    /*========================================================================*/

    /**
     * Constructor.
     * @param main Agent's main.
     */
    public LoqueMemory (Main main)
    {
        // setup reference to agent
        this.main = main;
        this.memory = main.getMemory ();
        this.body = main.getBody ();
        this.log = main.getLogger ();
        this.platformLog = main.getPlatformLog ();

        // create memory parts
        this.self = new LoqueSelf (main, this);
        this.senses = new LoqueSenses (main, this);
        this.items = new LoqueMapItems (main, this);
        this.inventory = new LoqueInventory (main, this);
        this.players = new LoquePlayers (main, this);

        // create listener
        this.listener = new Listener ();
    }
}
