package loquebot.memory;

import java.util.HashMap;
import java.util.ArrayList;
import java.util.logging.Logger;

import cz.cuni.pogamut.Client.AgentBody;

import cz.cuni.pogamut.MessageObjects.MessageObject;
import cz.cuni.pogamut.MessageObjects.MessageType;
import cz.cuni.pogamut.MessageObjects.ItemType;

import cz.cuni.pogamut.MessageObjects.Self;
import cz.cuni.pogamut.MessageObjects.Spawn;
import cz.cuni.pogamut.MessageObjects.AddAmmo;
import cz.cuni.pogamut.MessageObjects.AddWeapon;
import cz.cuni.pogamut.MessageObjects.BotKilled;

import loquebot.Main;
import loquebot.util.LoqueListener;
import loquebot.util.LoqueWeaponInfo;

/**
 * Responsible for listening to the messages and managing agent inventory.
 *
 * @author Juraj Simlovic [jsimlo@matfyz.cz]
 * @version Tested on Pogamut 2 platform version 1.0.5.
 */
public class LoqueInventory
{
    /**
     * Tells, whether specific weapon is in the agent's inventory.
     * @param UnrealID UnrealID of the weapon to be examined.
     * @return True, if the requested weapon is present; false otherwise.
     */
    public synchronized boolean hasWeapon (String UnrealID)
    {
        // search for the weapon
        return weaponryUnrealIDs.containsKey(UnrealID);
    }

    /**
     * Tells, whether specific weapon is in the agent's inventory.
     * @param itemType Type of the weapon to be retreived.
     * @return True, if the requested weapon is present; false otherwise.
     */
    public synchronized boolean hasWeapon (ItemType itemType)
    {
        // search for the weapon
        return weaponryTypes.containsKey (itemType);
    }

    /*========================================================================*/

    /**
     * Retreives specific weapon from the agent's inventory.
     * @param UnrealID UnrealID of the weapon to be retreived.
     * @return Requested weapon from inventory; or null upon no such weapon.
     * @note Returns copy of the weapon info to prevent outer modifications.
     */
    public synchronized AddWeapon getWeapon (String UnrealID)
    {
        // retreive weapon
        AddWeapon weapon = weaponryUnrealIDs.get (UnrealID);
        // return cloned weapon or null, upon none
        return (weapon == null) ? null : (AddWeapon) weapon.clone ();
    }

    /**
     * Retreives specific weapon from the agent's inventory.
     * @param itemType Type of the weapon to be retreived.
     * @return Requested weapon from inventory; or null upon no such weapon.
     * @note Returns copy of the weapon info to prevent outer modifications.
     */
    public synchronized AddWeapon getWeapon (ItemType itemType)
    {
        // retreive weapon
        AddWeapon weapon = weaponryTypes.get (itemType);
        // return cloned weapon or null, upon none
        return (weapon == null) ? null : (AddWeapon) weapon.clone ();
    }

    /*========================================================================*/

    /**
     * Retreives current weapon from the agent's inventory.
     * @return Current weapon from inventory; or null upon no current weapon.
     * @note Returns copy of the weapon info to prevent outer modifications.
     */
    public synchronized AddWeapon getCurrentWeapon ()
    {
        // retreive current weapon
        String current = memory.self.getWeaponUnrealID ();
        // retreive inventory weapon
        AddWeapon weapon = (current == null) ? null : weaponryUnrealIDs.get (current);
        // return cloned weapon or null, upon none
        return (weapon == null) ? null : (AddWeapon) weapon.clone ();
    }

    /*========================================================================*/

    /**
     * Retreives all weapons from the agent's inventory.
     * @return List of all available weapons from inventory.
     * @note Returns copies of the weapon infos to prevent outer modifications.
     */
    public synchronized ArrayList<AddWeapon> getWeapons ()
    {
        ArrayList<AddWeapon> result = new ArrayList<AddWeapon> (weaponry.size ());
        // make deep clone of the weaponry
        for (AddWeapon w : weaponry)
            result.add((AddWeapon) w.clone ());
        // return created list
        return result;
    }

    /*========================================================================*/

    /**
     * Tells, whether the agent has any loaded weapon in the inventory.
     *
     * <p>Note: <b>Shield guns</b> are never treated as loaded weapons, though
     * they are always <i>loaded</i>, i.e. ready to be fired.</p>
     *
     * <h4>Pogamut troubles</h4>
     *
     * There might be weapons with ammo falsely remembered as present while
     * there is none. Fortunatelly, right now all Loque drives are capable of
     * switching to all reported-as-non-empty weapons and thus request update
     * of their current ammo through SELF message mechanism. Ultimatelly, all
     * weapons get updated and this method then starts to work correctly again.
     *
     * <p>Warning: Do not use the manual respawn button. If respawned manualy,
     * the agent does not receive death message and does not clear inventory
     * correctly, keeping old weapons he does not posses anymore. This might
     * introduce currently unsolvable bugs in this method.</p>
     *
     * @return True, if there is a loaded weapon in the inventory.
     */
    public synchronized boolean hasLoadedWeapon ()
    {
        // run through all weaponry
        for (AddWeapon w : weaponry)
            if (
                // no shield guns please..
                (w.weaponType != ItemType.SHIELD_GUN)
                // is there any ammo in this weapon?
                && (w.currentAmmo > 0)
            )
                // great, we got one!
                return true;
        // got none
        return false;
    }

    /*========================================================================*/
    /*========================================================================*/
    /*========================================================================*/

    /**
     * All foraged weapons.
     */
    private ArrayList<AddWeapon> weaponry = new ArrayList<AddWeapon> ();
    /**
     * All foraged weapons mapped by their type.
     */
    private HashMap<ItemType, AddWeapon> weaponryTypes = new HashMap<ItemType, AddWeapon> ();
    /**
     * All foraged weapon mapped by their UnrealIDs.
     */
    private HashMap<String, AddWeapon> weaponryUnrealIDs = new HashMap<String, AddWeapon> ();
    /**
     * All foraged unassigned ammo (ammo without akin weapon) mapped by type.
     */
    private HashMap<ItemType, Integer> ammoOnly = new HashMap<ItemType, Integer> ();

    /*========================================================================*/

    /**
     * Adds new foraged weapon into the inventory.
     * @param weapon Weapon message to be relied upon.
     */
    private synchronized void addNewWeapon (AddWeapon weapon)
    {
        // clone new inventory weapon
        AddWeapon w = (AddWeapon) weapon.clone ();
        // retreive current ammo status..
        Integer currentAmmo = ammoOnly.remove(w.weaponType);
        // rest amounts of ammo
        w.currentAmmo = (currentAmmo == null) ? 0 : currentAmmo;
        w.currentAmmo = Math.min (w.currentAmmo, w.maxAmmo);
        w.currentAltAmmo = 0;
        // put the weapon to all maps
        weaponry.add(w);
        weaponryTypes.put (w.weaponType, w);
        weaponryUnrealIDs.put (w.UnrealID, w);

        log.config (
            "Memory.Inventory.addNewWeapon(): got first " + w.weaponType
            + ", initial ammo " + w.currentAmmo
        );
    }

    /**
     * Claims ammo from a foraged weapon, which is already in the inventory.
     * @param weapon Weapon message to be relied upon.
     */
    private synchronized void addExistingWeapon (AddWeapon weapon)
    {
        // do we have this kind of weapon already?
        AddWeapon w = weaponryTypes.get (weapon.weaponType);
        if (w != null)
        {
            // retreive ammo amounts
            int ammoAmount = LoqueWeaponInfo.getInfo(w.weaponType).pickupWeaponAmount;
            // update the amount of ammo
            w.currentAmmo += ammoAmount;
            w.currentAmmo = Math.min (w.currentAmmo, w.maxAmmo);

            log.config (
                "Memory.Inventory.addOldWeapon(): got another " + w.weaponType
                + ", ammo amount " + ammoAmount
                + ", current ammo " + w.currentAmmo
            );
        }
        else throw new RuntimeException ("don't have this kind of weapon yet!");
    }

    /**
     * Adds foraged ammo pack into inventory.
     * @param ammo Ammo message to be relied upon.
     */
    private synchronized void addAmmo (AddAmmo ammo)
    {
        // retreive ammo amounts
        int ammoAmount = LoqueWeaponInfo.getInfo(ammo.weaponType).pickupAmmoAmount;
        // do we have this kind of weapon already?
        AddWeapon w = weaponryTypes.get (ammo.weaponType);
        if (w != null)
        {
            // update the amount of ammo
            w.currentAmmo += ammoAmount;
            w.currentAmmo = Math.min (w.currentAmmo, w.maxAmmo);

            log.config (
                "Memory.Inventory.addAmmo(): got ammo into " + w.weaponType
                + ", ammo amount " + ammoAmount
                + ", current ammo " + w.currentAmmo
            );
        }
        // so we got no weapon like this yet
        else
        {
            // retreive current status..
            Integer current = ammoOnly.get(ammo.weaponType);
            // update the amount of ammo
            current = (current == null) ? ammoAmount : (current + ammoAmount);
            // put new status..
            ammoOnly.put(ammo.weaponType, current);

            log.config (
                "Memory.Inventory.addAmmo(): got ammo for " + ammo.weaponType
                + ", ammo amount " + ammoAmount
                + ", current ammo " + current
            );
        }
    }

    /**
     * Updates ammo in the current weapon according to self message.
     * @param self Self message to be relied upon.
     */
    private synchronized void updateCurrentAmmo (Self self)
    {
        // do we have this kind of weapon already?
        AddWeapon w = weaponryUnrealIDs.get (self.weapon);
        if (w != null)
        {
            // update it's ammo
            w.currentAmmo = self.currentAmmo;
            w.currentAltAmmo = self.currentAltAmmo;
        }
    }

    /**
     * Fixes initial ammo amounts after agent spawn.
     */
    private synchronized void fixInitialAmmo ()
    {
        // run through all weaponry
        for (AddWeapon w : weaponry)
            if (w.currentAmmo == 0)
            {
                // and fix ammo, if it's zero
                w.currentAmmo = LoqueWeaponInfo.getInfo(w.weaponType).pickupWeaponAmount;

                log.config (
                    "Memory.Inventory.fixInitialAmmo(): fixed ammo for " + w.weaponType
                    + ", current ammo " + w.currentAmmo
                );
            }
    }

    /**
     * Clears inventory. Used upon agent death.
     */
    private synchronized void clearWeaponry ()
    {
        log.config ("Memory.Inventory.clearWeaponry (): clearing inventory");

        weaponry.clear ();
        weaponryTypes.clear ();
        weaponryUnrealIDs.clear ();
        ammoOnly.clear ();
    }

    /*========================================================================*/

    /**
     * Listening class for messages from engine.
     */
    private class Listener extends LoqueListener
    {
        /**
         * Periodic info about the agent.
         * @param msg Message to handle.
         */
        private void msgSelf (Self msg)
        {
            updateCurrentAmmo (msg);
        }

        /**
         * Agent just spawned into the game.
         * @param msg Message to handle.
         */
        private void msgSpawn (Spawn msg)
        {
            fixInitialAmmo ();
        }

        /**
         * Agent picked up an ammo pack.
         * @param msg Message to handle.
         */
        private void msgAddAmmo (AddAmmo msg)
        {
            addAmmo (msg);
        }

        /**
         * Agent picked up a weapon.
         * @param msg Message to handle.
         */
        private void msgAddWeapon (AddWeapon msg)
        {
            // did we picked-up a gun we already carry?
            if (msg.ID == 0)
            {
                addNewWeapon (msg);
            }
            else addExistingWeapon (msg);
        }

        /**
         * Agent has died. Somehow.
         * @param msg Message to handle.
         */
        private void msgBotKilled (BotKilled msg)
        {
            clearWeaponry ();
        }

        /*========================================================================*/

        /**
         * Message switch.
         * @param msg Message to handle.
         */
        protected void processMessage (MessageObject msg)
        {
            switch (msg.type)
            {
                case SELF:
                    msgSelf ((Self) msg);
                    return;
                case SPAWN:
                    msgSpawn ((Spawn) msg);
                    return;
                case ADD_AMMO:
                    msgAddAmmo ((AddAmmo) msg);
                    return;
                case ADD_WEAPON:
                    msgAddWeapon ((AddWeapon) msg);
                    return;
                case BOT_KILLED:
                    msgBotKilled ((BotKilled) msg);
                    return;
            }
        }

        /**
         * Constructor: Signs up for listening.
         */
        private Listener ()
        {
            body.addTypedRcvMsgListener (this, MessageType.SELF);
            body.addTypedRcvMsgListener (this, MessageType.SPAWN);
            body.addTypedRcvMsgListener (this, MessageType.ADD_AMMO);
            body.addTypedRcvMsgListener (this, MessageType.ADD_WEAPON);
            body.addTypedRcvMsgListener (this, MessageType.BOT_KILLED);
        }
    }

    /** Listener. */
    private LoqueListener listener;

    /*========================================================================*/

    /** Agent's main. */
    protected Main main;
    /** Loque memory. */
    protected LoqueMemory memory;
    /** Agent's body. */
    protected AgentBody body;
    /** Agent's log. */
    protected Logger log;

    /*========================================================================*/

    /**
     * Constructor.
     * @param main Agent's main.
     * @param memory Loque memory.
     */
    public LoqueInventory (Main main, LoqueMemory memory)
    {
        // setup reference to agent
        this.main = main;
        this.memory = memory;
        this.body = main.getBody ();
        this.log = main.getLogger ();

        // create listener
        this.listener = new Listener ();
    }
}