package loquebot.drives;

import cz.cuni.pogamut.MessageObjects.MessageObject;
import cz.cuni.pogamut.MessageObjects.MessageType;
import cz.cuni.pogamut.MessageObjects.Triple;

import cz.cuni.pogamut.MessageObjects.Player;
import cz.cuni.pogamut.MessageObjects.Spawn;
import cz.cuni.pogamut.MessageObjects.PlayerKilled;

import loquebot.Main;
import loquebot.util.LoqueListener;
import loquebot.memory.LoqueMemory;
import loquebot.body.LoqueTravel;

/**
 * Responsible for pursuing a lost enemy.
 *
 * <p>Main purpose of this drive is to decide, whether a pursue is to be dealt
 * with and resolves the hunting of enemy upon oportunity. The pursue receives
 * hunting requests from other drives (e.g. {@link LoqueCombat}) and handles
 * the rest of the chase itself.</p>
 *
 * <p>Note: This is actually only a placeholder for a nice and smart drive. It
 * does not do anything reasonable right now and in current state it might be
 * better to be turned off. Well, it tries to follow the lost enmey, but that's
 * not always a good choice, is it? What if the enemy got lost from agent sight
 * because of falling into a lava pit? Then the agent would do the same..
 * <i>You know, there was a scene in Ice Age movie, with dodo birds getting
 * extinct. Well.. That's what I mean..</i></p>
 *
 * <h4>Future considerations</h4>
 *
 * <p>In order to pursue the enemy to the maximum available point, we might,
 * after running to the <i>enemy.location</i>, also try to run in the direction
 * of the <i>enemy.velocity</i>. Maybe even go and check further space..</p>
 *
 * <p>Since the enemy might got lost from sight because of falling into some
 * pit of lava, we might also want to check the <i>reachability and safety</i>
 * of <i>enemy.location</i>, whatever that might mean.</p>
 *
 * @author Juraj Simlovic [jsimlo@matfyz.cz]
 * @version Tested on Pogamut 2 platform version 1.0.5.
 */
public class LoquePursue extends LoqueDrive
{
    /**
     * Player that we're supoosed to hunt.
     */
    private Player lastTarget = null;

    /**
     * Id of the current travel ticket.
     */
    private int travelTicket = 0;

    /**
     * Whether to reset all pursue info.
     * Used in case of agent's or enemy death.
     */
    private boolean killPursue = false;

    /*========================================================================*/

    /**
     * Main logic of the pursue. Drives the hunting of given enemy.
     *
     * <h4>Cook book</h4>
     * <ul>
     * <li>Check, whether there is something to hunt. Hunting requests usually
     * come from {@link LoqueCombat} drive.</li>
     * <li>Travel to the last known location of the target. Hopefully, we can
     * find him there.. Ehm, waiting for us?</li>
     * </ul>
     *
     * @return True, if the drive decided what shall we do now. False otherwise.
     */
    @Override public boolean doLogic ()
    {
        if (!main.optionPursue)
            return false;

        // include pursue logs?
        enableLogs (main.optionPursueLogs);

        // do we kill current pursue?
        if (killPursue)
        {
            // reset info
            lastTarget = null;
            travelTicket = 0;
            // reset flag
            killPursue = false;
            // and do nothing
            return false;
        }

        // do we have a travel ticket already?
        if (travelTicket > 0)
        {
            // we have a ticket, travel to it..
            if (travel.keepTraveling (travelTicket))
                return true;

            // do not travel anymore..
            log.fine ("Pursue.doLogic(): lost target " + lastTarget.UnrealID + ", turning around a bit");

            // turn by the velocity of the lost enemy
            body.turnToLocation(Triple.add(lastTarget.location, lastTarget.velocity));

            // kill the pursue request
            lastTarget = null;
            travelTicket = 0;

            // we took the last action..
            return true;
        }
        else
        {
            // got a target to pursue?
            if (lastTarget == null)
                return false;

            log.info ("Pursue.doLogic(): starting new travel");

            // travel to the last known location
            travelTicket = travel.initTicketToPlayer (
                lastTarget, (int) (main.logicFrequency * main.optionPursueTimeout)
            );

            // are we traveling somewhere?
            return (travelTicket > 0);
        }
    }

    /*========================================================================*/

    /**
     * Sets new target to be pursued.
     *
     * @param enemy New target to hunt.
     */
    public void setTarget (Player enemy)
    {
        // setup new target..
        lastTarget = enemy;
        travelTicket = 0;
    }

    /*========================================================================*/

    /**
     * Kills all pursue info. Used in case of agent's death or enemy death.
     */
    private void killPursue ()
    {
        // setup kill flag
        killPursue = true;
    }

    /*========================================================================*/

    /**
     * Listening class for messages from engine.
     */
    private class Listener extends LoqueListener
    {
        /**
         * Agent just spawned into the game.
         * @param msg Message to handle.
         */
        private void msgSpawn (Spawn msg)
        {
            // reset pursue info
            killPursue ();
        }

        /**
         * Someone fragged somebody.
         * @param msg Message to handle.
         */
        private void msgPlayerKilled (PlayerKilled msg)
        {
            Player target = lastTarget;

            // did our own target died?
            if (
                (target != null)
                && (msg.playerID == target.ID)
            )
            {
                // kill pursue, since the hunt is over
                killPursue ();
            }
        }

        /**
         * Message switch.
         * @param msg Message to handle.
         */
        protected void processMessage (MessageObject msg)
        {
            switch (msg.type)
            {
                case SPAWN:
                    msgSpawn ((Spawn) msg);
                    return;
                case PLAYER_KILLED:
                    msgPlayerKilled ((PlayerKilled) msg);
                    return;
            }
        }

        /**
         * Constructor: Signs up for listening.
         */
        private Listener ()
        {
            body.addTypedRcvMsgListener(this, MessageType.SPAWN);
            body.addTypedRcvMsgListener(this, MessageType.PLAYER_KILLED);
        }
    }

    /** Listener. */
    private LoqueListener listener;

    /*========================================================================*/

    /** Loque pursue. */
    protected LoqueTravel travel;

    /*========================================================================*/

    /**
     * Constructor.
     * @param main Agent's main.
     * @param memory Loque memory.
     * @param travel Loque travel.
     */
    public LoquePursue (Main main, LoqueMemory memory, LoqueTravel travel)
    {
        super (main, memory);
        this.travel = travel;

        // create listener
        this.listener = new Listener ();
    }
}