package loquebot.drives;

import java.util.ArrayList;

import cz.cuni.pogamut.MessageObjects.Triple;

import cz.cuni.pogamut.MessageObjects.Player;
import cz.cuni.pogamut.MessageObjects.AddWeapon;

import loquebot.Main;
import loquebot.util.LoqueWeaponInfo;
import loquebot.memory.LoqueMemory;

/**
 * Necessary support for handling bloody combats. This class implements all
 * composite calculations and decission making routines used by the logic in
 * {@link LoqueCombat} to make {@link LoqueCombat} look more simple and
 * understandalbe on laic glance.
 *
 * @author Juraj Simlovic [jsimlo@matfyz.cz]
 * @version Tested on Pogamut 2 platform version 1.0.5.
 */
public abstract class LoqueCombatBase extends LoqueDrive
{
    /**
     * Where do we strafe? Left or right?
     */
    protected boolean strafingRight = false;

    /**
     * Which weapon do we currently use.
     */
    protected AddWeapon currentWeapon = null;
    /**
     * Which weapon do we currently use.
     */
    protected LoqueWeaponInfo currentWeaponInfo = null;

    /**
     * Tells, whether the bot should explicitly consider a better weapon.
     */
    protected boolean considerRearm = false;

    /**
     * Whether we shall use alternate fire mode.
     */
    protected boolean alternateFire = false;

    /**
     * Whether we shall benefit from aim correction.
     */
    protected boolean aimCorrection = false;

    /*========================================================================*/

    /**
     * Local fix variable for {@link #chooseWeaponToFire(Player)}.
     */
    private boolean changeToBestWeapon = false;

    
    
    
    /**
     * Decides, which weapon to use on the given enemy and whether to use the
     * alternate fire mode. Current agent inventory status is considered to
     * choose a reasonable weapon with enough of ammo. Current enemy location
     * is being considered as well as current enemy weapon.
     * enemy.
     *
     * @param enemy Enemy, for Whom the Bell Tolls. Might be null.
     */
    private void chooseWeaponToFire (Player enemy)
    {
        // and we are allowed to fire at will
        if (main.optionCombatSafetyOff)
        {
            // calculate enemy distance
            double enemyDistance = (enemy == null)
                ? -1 : memory.self.getSpaceDistance (enemy.location);

            // retreive agent skill level
            int agentSkill = body.initializer.getBotSkillLevel ();

            // retreive enemy weapon name
            String enemyWeapon = (enemy == null) ? null : enemy.weapon;

            // get the inventory
            ArrayList<AddWeapon> weapons = memory.inventory.getWeapons ();

            // log the weapons we choose from..
            for (AddWeapon w : weapons)
                log.finest(
                    "CombatBase.chooseWeaponToFire(): have " + w.weaponType
                    + ", ammo " + w.currentAmmo
                );
            
            //choose a weapon
            AddWeapon weapon;
            //use different algorithm if learning weapon stats (wee need enough ifo about each weapon)
            if(main.learnWeapons)
            {
            	if(main.timeSinceWeaponChanged > 50){
            		main.timeSinceWeaponChanged = 0;
            		weapon = main.Learn.chooseWeapon(weapons);
            		body.changeWeapon (weapon);
            	}
            }
            else
            {
                // choose a good weapon that is available..
                weapon = LoqueWeaponInfo.chooseWeapon (
                    agentSkill, weapons, enemyDistance, enemyWeapon
                );
            
                
                
                // did choosing weapon failed?
                if (weapon == null)
                {
                    log.finest ("CombatBase.chooseWeaponToFire(): choosing of weapon failed!");

                    // let the engine decide
                    // note: local fix for body.changeToBestWeapon()
                    if (!this.changeToBestWeapon)
                    {
                        body.changeToBestWeapon ();
                        this.changeToBestWeapon = true;
                    }
                    // do not use alternate fire
                    //alternateFire = false;
                    // use aim corection
                    //aimCorrection = true;
                    return;
                }
                // note: local fix for body.changeToBestWeapon()
                else this.changeToBestWeapon = false;

                // did we choose a different weapon?
                if (
                    (currentWeapon == null)
                    || (weapon.weaponType != currentWeapon.weaponType)
                )
                {
                    // what's the new weapon like?
                    LoqueWeaponInfo weaponInfo = LoqueWeaponInfo.getInfo (weapon.weaponType);

                    int weaponScore = weaponInfo.combatScore (agentSkill, enemyDistance);
                    int currentWeaponScore = currentWeaponInfo.combatScore (agentSkill, enemyDistance);
                    int currentWeaponAmmo = (currentWeapon == null) ? 999 : currentWeapon.currentAmmo;

                    // is the new weapon worth the change?
                    if (
                        // we're running low on ammo, the change is imminent anyway
                        (currentWeaponAmmo <= (currentWeaponInfo.priLowAmmo / 3))
                        || (
                            // or the current weapon is not one of the the best
                            (currentWeaponScore < LoqueWeaponInfo.bestScoreTreshold)
                            // and the new chosen weapon is better
                            && (weaponScore > currentWeaponScore)
                        )
                    )
                    {
                        log.config (
                            "CombatBase.chooseWeaponToFire(): changing from "
                            + currentWeaponInfo
                            + " (ammo " + currentWeaponAmmo + ")"
                            + " to " + weaponInfo
                        );

                        // we've got a new weapon
                        currentWeapon = weapon;
                        // with new weapon info
                        currentWeaponInfo = weaponInfo;

                        // change it
                        body.changeWeapon (currentWeapon);
                    }
                    else log.finest (
                        "CombatBase.chooseWeaponToFire(): changing from"
                        + " " + ((currentWeapon == null) ? "NONE" : currentWeapon.weaponType)
                        + " (ammo " + currentWeaponAmmo + ")"
                        + " to " + weapon.weaponType + " disapproved:"
                        + " score diff " + (weaponScore - currentWeaponScore)
                    );
                }
                else log.finest ("CombatBase.chooseWeaponToFire(): the " + currentWeaponInfo + " is just fine, ammo " + currentWeapon.currentAmmo);

                // do we use alternate fire?
                alternateFire = currentWeaponInfo.altFire (enemyDistance);

                // do we use aim corection?
                aimCorrection = alternateFire ? currentWeaponInfo.altHelpAim : currentWeaponInfo.priHelpAim;
            }
        }
    }

    /*========================================================================*/

    /**
     * Checks, whether the current weapon is a good choice for fighting.
     *
     * @param enemy Enemy, for Whom the Bell Tolls.
     * @return Whether the weapon is a good choice.
     */
    private boolean isWeaponEfficient (Player enemy)
    {
        // compute distance to the enemy
        double enemyDistance = memory.self.getSpaceDistance (enemy.location);

        // primary fire mode
        if (!alternateFire)
        {
            // is it too close?
            if (enemyDistance < currentWeaponInfo.priSplashRadius + 70)
            {
                log.finest("CombatBase.isWeaponEfficient(): target too close, " + enemyDistance);
                return false;
            }

            // is it too far?
            if (enemyDistance > currentWeaponInfo.priMaxRange)
            {
                log.finest("CombatBase.isWeaponEfficient(): target too far, " + enemyDistance);
                return false;
            }
        }
        // alternate fire mode
        else
        {
            // is it too close?
            if (enemyDistance < currentWeaponInfo.altSplashRadius + 70)
            {
                log.finest("CombatBase.isWeaponEfficient(): target too close, " + enemyDistance);
                return false;
            }

            // is it too far?
            if (enemyDistance > currentWeaponInfo.altMaxRange)
            {
                log.finest("CombatBase.isWeaponEfficient(): target too far, " + enemyDistance);
                return false;
            }
        }

        return true;
    }

    /*========================================================================*/
    /*========================================================================*/
    /*========================================================================*/

    /**
     * Checks, whether the current weapon is already shooting. Starts shooting
     * when necessary.
     *
     * @param enemy Enemy, for Whom the Bell Tolls.
     * @param location Location, where to shoot.
     */
    protected void checkWeaponShooting (Player enemy, Triple location)
    {
        if (
            main.optionCombatSafetyOff && !memory.self.isShooting ()
            && (
                memory.self.getSpaceDistance (enemy.location)
                < currentWeaponInfo.priMaxRange
            )
        )
        {
            
            if(main.learnWeapons){
                alternateFire = (Math.random() > 0.5);
//                if(alternateFire){
//                    alternateFire = (Math.random() > 0.25);
//                }else{
//                    alternateFire = (Math.random() > 0.75);
//                }
            }
            
            
            log.fine ("CombatBase.checkWeaponShooting(): fire at will, alternate " + alternateFire + ", aimCorrection " + aimCorrection);
            
            
            
            // start shooting alternate mode..
            if (alternateFire)
            {
                // enable/disable aim correction
                if (aimCorrection) body.shootAlternate (enemy);
                else{
                    body.shootAlternate (location);
                }
            }
            // or start shooting primary mode..
            else
            {
                // enable/disable aim correction
                if (aimCorrection) body.shoot (enemy);
                else body.shoot (location);
            }
            
            //learn about the result
            main.Learn.startShoot(alternateFire);
        }
        else if(Math.random() < 0.25)
        {
            body.stopShoot();
            main.Learn.stopShoot();
        }
    }

    /*========================================================================*/

    /**
     * Checks, whether the current weapon is a good choice for fighting. Rearms
     * when necessary.
     *
     * @param enemy Enemy, for Whom the Bell Tolls.
     */
    protected void checkWeaponEfficiency (Player enemy)
    {
        // what's the current weapon?
        if (null != (currentWeapon = memory.inventory.getCurrentWeapon ()))
        {
            // retreive weapon info
            currentWeaponInfo = LoqueWeaponInfo.getInfo(currentWeapon.weaponType);
        }
        // default to assault rifle upon no current weapon
        else currentWeaponInfo = LoqueWeaponInfo.ASSAULT_RIFLE;

        // should we reconsider the weapon for outer reasons?
        if (considerRearm || (enemy == null))
        {
            log.finest ("CombatBase.checkWeaponEfficiency(): forced change of weapon consideration");

            // reset the explicit flag
            considerRearm = false;
            // try to choose a better weapon
            chooseWeaponToFire (enemy);
        }
        // should we reconsider the weapon 'cos we're not shooting?
        else if (!memory.self.isShooting ())
        {
            log.finest ("CombatBase.checkWeaponEfficiency(): we're not shooting yet, change weapon before shooting");

            // try to choose a better weapon
            chooseWeaponToFire (enemy);
        }
        // should we reconsider the weapon 'cos we're out of ammo?
        else if (memory.self.getCurrentAmmo () <= currentWeaponInfo.priLowAmmo)
        {
            log.finest ("CombatBase.checkWeaponEfficiency(): low on ammo in current weapon, consider change of weapon");

            // try to choose a better weapon
            chooseWeaponToFire (enemy);
        }
        // should we reconsider the weapon 'cos the sky
        // just changed randomly from purple to blue
        else if (Math.random () < (1 / main.logicFrequency))
        {
            log.finest ("CombatBase.checkWeaponEfficiency(): change of weapon due to random");

            // try to choose a better weapon
            chooseWeaponToFire (enemy);
        }
        // should we reconsider the weapon 'cos its not efficient?
        else if (!isWeaponEfficient (enemy))
        {
            log.finest ("CombatBase.checkWeaponEfficiency(): change of weapon due to no efficiency");

            // are we shooting something?
            if (memory.self.isShooting ())
            {
                log.fine ("CombatBase.checkWeaponEfficiency(): stop wasting ammo");

                // stop shooting
                body.stopShoot ();
                //learn about the result
                main.Learn.stopShoot();
            }

            // then try to choose a better weapon
            chooseWeaponToFire (enemy);
        }
    }

    /*========================================================================*/

    /**
     * Computes the best aiming location that should be used with specified
     * weapon on the given enemy. Calculations include aim-ahead predictions
     * and head-shot opportunities.
     *
     * <h4>How does the aim-ahead work?</h4>
     *
     * <p> This method uses current enemy velocity and his distance from agent
     * to calculate most probable intersection of fired projectile and enemy.
     * This is done by applying the <i>Law of sines</i> in a triangle determined
     * by L(agent), E(enemy) and I(probable projected intersection).</p>
     *
     * <p>L and E are known locations, I shall be calculated. Further, angle B
     * in the vertex E is known. So, we first calculate angle A in the vertex L.
     * Second, we calculate size of the triangle side between vertices E and I
     * (probable enemy movement till intersection) from the angles and from the
     * enemy distance. Simple linear algebra that is, isn't it? :)</p>
     *
     * <h4>Static aim-ahead</h4>
     *
     * <p>To compensate for server communication lag, we also add static value
     * based on the enemy velocity to the aim-ahead vector.</p>
     *
     * @param enemy Enemy, for Whom the Bell Tolls.
     * @return Aiming location to where to aim to with the given weapon.
     */
    protected Triple getAimWeaponLocation (Player enemy)
    {
        double projectileVelocitySize;
        double weaponEffectiveDistance;
        double reasonableFlightRange;
        double staticAimAheadSize;
        double staticZCorrection;

        // primary fire mode
        if (!alternateFire)
        {
            projectileVelocitySize = currentWeaponInfo.priProjectileSpeed;
            weaponEffectiveDistance = currentWeaponInfo.priEffectiveDistance;
            reasonableFlightRange = currentWeaponInfo.priReasonableFlight;
            staticAimAheadSize = currentWeaponInfo.priStaticAimAhead;
            staticZCorrection = currentWeaponInfo.priStaticZCorrection;
        }
        // alternate fire mode
        else
        {
            projectileVelocitySize = currentWeaponInfo.altProjectileSpeed;
            weaponEffectiveDistance = currentWeaponInfo.altEffectiveDistance;
            reasonableFlightRange = currentWeaponInfo.altReasonableFlight;
            staticAimAheadSize = currentWeaponInfo.altStaticAimAhead;
            staticZCorrection = currentWeaponInfo.altStaticZCorrection;
        }

        // get agent location from memory
        Triple agentLocation = memory.self.getLocation ();

        // get location and velocity of enemy
        Triple enemyLocation = enemy.location;
        Triple enemyVelocity = enemy.velocity;

        // compute reversed direction to the enemy
        // howto: substract the two locations in reversed order
        Triple enemyDirection = Triple.subtract(agentLocation, enemyLocation);

        // get distance to the enemy and size of the enemy velocity
        double enemyDirectionSize = enemyDirection.vectorSize ();
        double enemyVelocitySize = enemyVelocity.vectorSize ();

        // if the enemy is not moving, this whole aim-ahead thing is somewhat
        // pointless, therefore, simply return the enemy location unmodified
        if (enemyVelocitySize == 0)
            return enemy.location;

        // compute angle between the direction to the enemy and the enemy velocity
        // howto: quotient of dot product of vectors and product of their sizes
        double enemyVelocityAngle =
            Math.acos(
                enemyDirection.dot (enemyVelocity)
                /
                (enemyDirectionSize * enemyVelocitySize)
            );

        // compute so-called "effective" distance to the enemy
        // howto: limit the distance by what the weapon considers effective
        double enemyDistance = Math.min(weaponEffectiveDistance, enemyDirectionSize);

        // compute angle between direction to the enemy and aim direction
        // howto: using modified Law of sines: sin(A) = (a/b) * sin(B)
        double sinOfAimAngle =
            Math.sin (enemyVelocityAngle)
            * (enemyVelocitySize / projectileVelocitySize);

        // compute length of predicted enemy movement (aim-ahead multiplier)
        // howto: using modified Law of sines: a = (c / sin(C)) * sin (A)
        // note: vector C is what's left in a triangle besides vectors A and B
        // therefore: a = (c / sin(PI - A - B)) * sin (A)
        double enemyMovement =
            (
                enemyDistance
                /
                Math.sin (Math.PI - enemyVelocityAngle - Math.asin(sinOfAimAngle))
            )
            * sinOfAimAngle;

        // limit the aim-ahead multiplier by the size of the enemy velocity
        // howto: this limit is vaguely based on risky assumtion that most
        // players change their movement at least every XX seconds
        if (enemyMovement > enemyVelocitySize * reasonableFlightRange)
            enemyMovement = enemyVelocitySize * reasonableFlightRange;

        // compute aim-ahead vector from the multiplier and enemy velocity
        // howto: normalize the enemy velocity vector before multiplying
        Triple aimAheadVector = Triple.multiplyByNumber(
            enemyVelocity, (enemyMovement / enemyVelocitySize)
        );

        // add static aim-ahead distance in the enemy velocity direction
        aimAheadVector = Triple.add(
            aimAheadVector,
            Triple.multiplyByNumber (
                enemyVelocity, (staticAimAheadSize / enemyVelocitySize)
            )
        );

        // aim at the head, or to the feet?
        aimAheadVector.z += staticZCorrection;

        //log.finest ("CombatBase.getAimWeaponLocation(): projectileVelocitySize " + projectileVelocitySize + " staticAimAheadSize " + staticAimAheadSize + ", staticZCorrection " + staticZCorrection);
        //log.finest ("CombatBase.getAimWeaponLocation(): aim-ahead size " + aimAheadVector.vectorSize () + " with " + currentWeaponInfo.name + ", vect " + aimAheadVector);

        // add multiplied aim-ahead vector to current enemy location
        enemyLocation = Triple.add(enemyLocation, aimAheadVector);

        // return the location
        return enemyLocation;
    }

    /*========================================================================*/

    /**
     * Checks, whether the current strafing decission is a good one. Considers
     * possible collisions, obstacles, etc.
     *
     * <h4>Pogamut troubles</h4>
     *
     * How about using autotrace rays for scanning the geometry around? Usage
     * of two autotraces pointed to the sides and a one to the ground to detect
     * incomming collisions as well as incomming falls would be nice. Well, if
     * the autotrace rays were working as they should. For now, it causes more
     * trouble than it helps.
     *
     * @param enemy Enemy, which to dance around.
     */
    protected void checkStrafingDirection (Player enemy)
    {
        // if colliding, strafe to other direction..
        if (memory.senses.isColliding ())
        {
            // flip the strafing direction
            log.fine ("CombatBase.checkStrafing(): changing strafe direction due to collision");
            strafingRight = !strafingRight;
        }
        // this might be a long journey.. consider changing strafe direction
        else if (Math.random () < (1 / (main.logicFrequency * 3)))
        {
            log.fine ("CombatBase.checkStrafing(): changing strafe direction due to random");
            strafingRight = !strafingRight;
        }
    }

    /*========================================================================*/

    /**
     * Computes the best location that should be used as strafing destination
     * while dancing around given enemy during a combat. Agent and enemy weapons
     * are considered to choose the location. Nearby health packs, vials and
     * armors are picked up along the way.
     *
     * <h4>Pogamut troubles</h4>
     *
     * How about using autotrace rays for scanning the ground around? Usage of
     * one autotrace ray pointed to the direction where the agent is aiming
     * might help prevent rocketry suicides. Well, it could, if the autotrace
     * were working as it was supposed to. For now, it causes more trouble than
     * it helps.
     *
     * <h4>Future</h4>
     *
     * What about foraging nearby healths? Check the perimeter and pick them up.
     * This could be done easily by comparing the calculated strafing vector
     * with the vectors of nearby reachable items. Should the angle between the
     * vectors be small enough, strafe to the item instead of the calculated
     * strafing point.
     *
     * <p>There is one pitfall to this however: The closer the items are to the
     * agent, the bigger might their angle-between-the-vectors be. Paradoxicaly:
     * the closer the item is, the more the angle starts to raise. And the vial
     * gets to be more attractive. In results, comparing the vectors only is
     * not good enough. Distance must be taken into consideration and tweaked
     * into a reasonable condition with the vectors angle.</p>
     *
     * @param enemy Enemy, which to dance around.
     * @return Strafing location to where to strafe to while wrestling.
     */
    protected Triple getStrafeAroundLocation (Player enemy)
    {
        // this is used for debugging purposes
        //if (main._DEBUGLocation != null) return main._DEBUGLocation;

        double desiredEnemyDistance;
        double strafingAmount;

        // primary fire mode
        if (!alternateFire)
        {
            desiredEnemyDistance = currentWeaponInfo.priIdealCombatRange;
            strafingAmount = currentWeaponInfo.priStrafingAmount;
        }
        // alternate fire mode
        else
        {
            desiredEnemyDistance = currentWeaponInfo.altIdealCombatRange;
            strafingAmount = currentWeaponInfo.altStrafingAmount;
        }

        // get agent location from memory
        Triple agentLocation = memory.self.getLocation ();

        // get location and velocity of enemy
        Triple enemyLocation = enemy.location;
        Triple enemyVelocity = enemy.velocity;

        // update the enemy location by its velocity
        enemyLocation = Triple.add(
            enemyLocation,
            Triple.multiplyByNumber(enemyVelocity, 1/main.logicFrequency)
        );

        // compute planar direction to the enemy
        // howto: substract the two locations, remove z-axis, normalize
        Triple enemyDirection = Triple.subtract(enemyLocation, agentLocation);
        // remove z-axis
        enemyDirection.z = 0;
        // and normalize it
        enemyDirection = enemyDirection.normalize ();

        // compute distance to the enemy
        double enemyDistance = Triple.distanceInSpace(enemyLocation, agentLocation);

        // compute orthogonal direction to the enemy
        Triple enemyOrthogonal = new Triple (enemyDirection.y, -enemyDirection.x, 0);

        // decide, how much to move forward
        double moveForward = enemyDistance - desiredEnemyDistance;

        // decide, how much and where to strafe
        double moveStrafe = strafingRight ? strafingAmount : -strafingAmount;

        // decide where to move..
        Triple moveDirection = moveDirection = Triple.add (
            // move forward/backward..
            Triple.multiplyByNumber (enemyDirection, moveForward),
            // and strafe to side along the way
            Triple.multiplyByNumber (enemyOrthogonal, moveStrafe)
        );

        // finally, add moving vector to current agent location
        return Triple.add(agentLocation, moveDirection);
    }

    /*========================================================================*/

    /**
     * Tries to compare two weapons and tell, which one is better..
     *
     * <p>The comparison is being done by substracting weapon scores returned
     * by {@link LoqueWeaponInfo#generalScore}. That means that value 100 means
     * a small difference, while a value 400 means a great difference.</p>
     *
     * @param w1 First weapon to compare.
     * @param w2 Second weapon to compare.
     * @return Returns positive number if the first weapon is better than the
     * second one. Returns negative number, if the second one is better.
     */
    protected int compareEnemyWeapons (String w1, String w2)
    {
        // we do not know enemy skill level, thus we use agent's
        int skill = body.initializer.getBotSkillLevel ();

        // compare the general score of the given weapons
        return
            LoqueWeaponInfo.getInfo(w1).generalScore(skill)
            - LoqueWeaponInfo.getInfo(w2).generalScore(skill);
    }

    /*========================================================================*/

    /**
     * Tries to refill agent's ammo to the current weapon, if the ammo is low.
     */
    protected void cheatRefillAmmo ()
    {
        // is there a need for ammo refill?
        if (
            (currentWeaponInfo != null)
            && (memory.self.getCurrentAmmo () < currentWeaponInfo.priLowAmmo)
        )
        {
            // add to the inventory..
            log.warning (
                "CombatBase.cheatRefillAmmo(): refreshing ammo for " + currentWeaponInfo.name
                + ", currently has " + memory.self.getCurrentAmmo ()
            );
            body.addInventory(currentWeaponInfo.name + "Pickup");
        }
    }

    /*========================================================================*/

    /**
     * Constructor.
     * @param main Agent's main.
     * @param memory Loque memory.
     */
    public LoqueCombatBase (Main main, LoqueMemory memory)
    {
        super (main, memory);
    }
}