package loquebot.body;

import java.util.ArrayList;
import java.util.logging.Logger;

import cz.cuni.pogamut.Client.AgentBody;

import cz.cuni.pogamut.MessageObjects.Triple;

import cz.cuni.pogamut.MessageObjects.Item;

import loquebot.Main;
import loquebot.util.LoqueUtils;
import loquebot.memory.LoqueMemory;

/**
 * Necessary support for issuing traveling tickets. This class implements all
 * composite calculations and decission making routines used by the logic in
 * {@link LoqueTravel} to make {@link LoqueTravel} look more simple and
 * understandalbe on laic glance.
 *
 * @author Juraj Simlovic [jsimlo@matfyz.cz]
 * @version Tested on Pogamut 2 platform version 1.0.5.
 */
public abstract class LoqueTravelBase
{
    /**
     * Id of the current travel ticket.
     */
    protected int ticketId = 0;

    /*========================================================================*/

    /**
     * Initializes new travel ticket to the items from the given list.
     *
     * @param items List of items to travel to.
     * @param loop Whether to loop other items after the first one.
     * @return Id of the travel ticket.
     */
    protected int initTicket (ArrayList<Item> items, boolean loop)
    {
        ArrayList<Triple> unreachable = new ArrayList<Triple> (items.size ());
        ArrayList<Triple> reachable = new ArrayList<Triple> (items.size ());

        // transform the list of items and get directly reachable items
        transformItems (items, unreachable, reachable);

        log.finest (
            "TravelBase.initTicket(): items divided:"
            + " unreachable " + unreachable.size ()
            + ", reachable " + reachable.size ()
        );

        // init the travel ticket..
        return initTicket (reachable, unreachable, loop);
    }

    /**
     * Initializes new travel ticket to the destinations from the given list.
     *
     * @param reachable List of directly reachable locations to travel to.
     * @param unreachable List of directly unreachable locations to travel to.
     * @param loop Whether to loop other destinations after the first one.
     * @return Id of the travel ticket.
     */
    private int initTicket (ArrayList<Triple> reachable, ArrayList<Triple> unreachable, boolean loop)
    {
        // are there any directly reachable locations?
        if (reachable.size() > 0)
        {
            // pick the nearest location
            Triple nearest = LoqueUtils.pickNearestLocation (memory.self.getLocation (), reachable, 300);

            // merge the directly reachable/unreachable locations
            unreachable.addAll(reachable);

            // init running ticket to that location
            return initTicket (nearest, unreachable, loop, -1);
        }

        // init running ticket to all locations
        return initTicket (null, unreachable, loop, -1);
    }

    /**
     * Initializes new travel ticket to the given destinations.
     *
     * @param dest Already chosen, directly reachable destination. Use null,
     * when no destination was chosen directly and the choice is to be made
     * from more destinations through the request to engine's path finding.
     * @param more List of other locations to travel to after the chosen,
     * directly reachable destination is reached/forsaken. This list may be
     * empty to indicate a travel only to the chosen, directly reachable
     * destination.
     * @param loop Whether to loop other destinations after the first one.
     * @param timeout Maximum amount of time to travel. Use -1 to for unlimited.
     * @return Id of the travel ticket.
     */
    protected int initTicket (Triple dest, ArrayList<Triple> more, boolean loop, int timeout)
    {
        // are there any locations to travel to?
        if (
            (dest == null)
            && (more.size() <= 0)
        )
        {
            // no items in there
            log.config ("TravelBase.initTicket(): failed: no locations to travel to");
            return -1;
        }

        // create new travel ticket
        ticketId++;

        log.config(
            "TravelBase.initTicket(): creating travel ticket " + ticketId
            + ", locations " + (more.size () + ((dest == null) ? 0 : 1))
            + ", timeout " + timeout
        );

        // init new travel request
        if (!manager.initRequest (dest, more, loop, timeout))
        {
            log.config ("TravelBase.initTicket(): failed: could not create travel ticket");
            return -1;
        }

        return ticketId;
    }

    /*========================================================================*/

    /**
     * Transforms the list of items into a list of locations. Filters the list
     * from forsaken and reachable locations during the process. While the
     * forsaken locations are trashed, the reachable ones are returned.
     *
     * @param items List of items to transform and filter.
     * @param unreachable List of directly unreachable locations.
     * @param reachable List of directly reachable locations.
     */
    private void transformItems (ArrayList<Item> items, ArrayList<Triple> unreachable, ArrayList<Triple> reachable)
    {
        // run thorugh the items
        for (Item item : items)
        {
            // is the item reachable?
            if (item.reachable)
            {
                // add to the reachables
                reachable.add(item.location);
            }
            // otherwise: add to the locations
            else unreachable.add(item.location);
        }
    }

    /*========================================================================*/

    /**
     * Loque Navigator.
     */
    protected LoqueTravelManager manager;

    /*========================================================================*/

    /** Agent's main. */
    protected Main main;
    /** Loque memory. */
    protected LoqueMemory memory;
    /** Agent's body. */
    protected AgentBody body;
    /** Agent's log. */
    protected Logger log;

    /*========================================================================*/

    /**
     * Constructor.
     * @param main Agent's main.
     * @param memory Loque memory.
     */
    public LoqueTravelBase (Main main, LoqueMemory memory)
    {
        // setup reference to agent
        this.main = main;
        this.memory = memory;
        this.body = main.getBody ();
        this.log = main.getLogger ();

        // create runner object
        this.manager = new LoqueTravelManager (main, memory);
    }
}