package loquebot;

import cz.cuni.pogamut.introspection.PogProp;
import cz.cuni.pogamut.exceptions.PogamutException;
import cz.cuni.pogamut.Client.Agent;

import cz.cuni.pogamut.MessageObjects.Triple;

import loquebot.memory.LoqueMemory;
import loquebot.body.LoqueTravel;
import loquebot.body.LoqueSpeech;
import loquebot.drives.LoquePursue;
import loquebot.drives.LoqueCombat;
import loquebot.drives.LoquePowerUp;
import loquebot.drives.LoqueWander;
import loquebot.learn.LoqueLearnWeapons;
import loquebot.util.LoqueWeaponInfo;

/**
 * Main class of the agnet. The agent consist of:
 * <ul>
 * <li>Options and status properties. Options can be modified during runtime
 * through introspection. These options control bot's behaviour and can be used
 * to closely examine specific parts of bot logic. Status properties provide
 * some useful info about agent sences and decission details.</li>
 * <li>Main bot drives, which shape the primary plan of bot's logic, ordered
 * by the <i>frag-them</i> and <i>outlive-them</i> priority. See each drive's
 * class for detailed info about how it works and when it fails to decide.
 * <ul>
 * <li>{@link LoqueCombat }, which drives combat, if there is an enemy.</li>
 * <li>{@link LoquePowerUp }, which drives picking up must-have items.</li>
 * <li>{@link LoquePursue }, which drives recent enemy pursue.</li>
 * <li>{@link LoqueWander }, which drives hiking around useful map items.</li>
 * </ul>
 * </li>
 * <li>{@link LoqueTravel } that provides <i>run-to-items</i> manager.</li>
 * <li>{@link LoqueMemory } that listens to msgs and feeds agent memory.</li>
 * <li>{@link util.LoqueWeaponInfo } that provides useful info about each
 * recognized weapon. Such info includes suitable combat ranges, weapon scores,
 * etc.</li>
 * </ul>
 *
 * <h4>Upbeat feature: Tweaked combat</h4>
 *
 * This agent is highly agile, aggresive and accurate in combats. This is due
 * to deeply tweaked fighting drive (see {@link LoqueCombat }) and adept weapon
 * knowledge (see {@link util.LoqueWeaponInfo }). Give it some weapon arsenal
 * and this agent will try to make the best use of it.
 *
 * <h4>Upbeat feature: Navigation and jumping</h4>
 *
 * This agent utilizes leveled navigation api, which emulates a travel agency.
 * Among others the travel api tweaks twi things:
 * <ul>
 * <li>Traveling along several items in sequence, according to their distance.
 * Though it doesn't solve Hamiltonian path, it helps forage quite quickly. See
 * {@link LoqueTravel } and {@link body.LoqueTravelManager }.</li>
 * <li>Safe-jumping over pitfalls and walls. Though the agent is almost blind,
 * he tries hard not to fall into every pit or bump into every wall. Small and
 * precise timeouts calculated separately per each path node help with those
 * troubles that could not be jumped over. See {@link body.LoqueNavigator } and
 * {@link body.LoqueRunner }.
 * </li>
 * </ul>
 *
 * <h4>Upbeat feature: Quick power-up</h4>
 *
 * This agent tries to make a difference between foraging the first goot weapon
 * for fighting and between foraging other items. This helps to rig him before
 * he gets into combats, rising the chances of success.
 *
 * <h4>Feature: Chatty</h4>
 *
 * This agent is quite chatty, especially in combats. Talking and sneering is
 * something this agent might abuse a bit.. ;)
 *
 * <h4>Feature: Personalized memory</h4>
 *
 * This agent utilizes its own memory (see {@link LoqueMemory }). The memory
 * resembles the pogamut's memory in most cases, though it fixes some bugs and
 * can provide either detailed or filtered info about things. This is essential
 * especially in retreiving known vs. visible items and in working with sences,
 * see {@link memory.LoqueMapItems } and {@link memory.LoqueSenses }.
 *
 * <h4>Downbeat: Presumable simplicity</h4>
 *
 * The rest of the agent logic is quite simple and can be <i>presumable</i>.
 * Skilled opponents could make a great use of this disadvantage. This agent
 * is mostly a berserk fighter rather than anything else. Well, it is tweaked
 * for death match only after all.. ;)
 *
 * @author Juraj Simlovic [jsimlo@matfyz.cz]
 * @version Tested on Pogamut 2 platform version 1.0.5.
 */
public class Main extends Agent
{
    /**
     * Fix: Check to force the bot give up on current travel request.
     * Should the bot stuck himself going endlessly and pointlessly to some
     * unreachable location, use this introspection "button" to fix him.. :))
     */
    @PogProp public boolean _fixFailNavigation = false;

    /*========================================================================*/

    /**
     * Option: Controls, whether the bot may try to engage in bloody combat.
     */
    @PogProp public boolean optionCombat = true;
    /**
     * Option: Controls, whether logs from combat are to be included.
     */
    @PogProp public boolean optionCombatLogs = true;
    /**
     * Option: Controls, whether the bot may try to shoot weapons.
     */
    @PogProp public boolean optionCombatSafetyOff = true;
    /**
     * Option: Controls, whether the bot may try to cheat and auto-refill ammo.
     */
    @PogProp public boolean optionCombatCheatAmmo = false;

    /*========================================================================*/

    /**
     * Option: Controls, whether the bot may try to power-up before combat.
     */
    @PogProp public boolean optionPowerUp = true;
    /**
     * Option: Controls, whether logs from power-up are to be included.
     */
    @PogProp public boolean optionPowerUpLogs = false;
    /**
     * Option: Controls, how much the agent feels greedy about powering-up.
     * Minimum reasonable value is 10, for obtaining at least one good weapon.
     * Higher values (above 10) allows the agent to think also about healing,
     * armors and udamage. Value of 50 means <i>at least two nicely loaded
     * top weapons</i>. Values above 100 are not easy to fulfil.
     *
     * <p>Note: The higher this value is, the longer the agent tries to drive
     * the power up. While powering up, other important drives like pursue are
     * locked-out. This might result in fully armed agent that will, however,
     * never try to hunt any bleeding enemy.</p>
     */
    @PogProp public int optionPowerUpWeaponTarget = 200;
    /**
     * Option: Controls, how much the agent feels freaked by low health status.
     * When the health is above this value, the agent does not feel the urge to
     * power-up health. This leaves space for other drives to take control,
     * specifically the pursue drive.
     */
    @PogProp public int optionPowerUpHealthTarget = 30;

    /*========================================================================*/

    /**
     * Option: Controls, whether the bot may try to pursue a lost enemy.
     */
    @PogProp public boolean optionPursue = true;
    /**
     * Option: Controls, whether logs from pursue are to be included.
     */
    @PogProp public boolean optionPursueLogs = false;
    /**
     * Option: Controls, for how long (in seconds) the bot should try to pursue.
     */
    @PogProp public double optionPursueTimeout = 6;

    /*========================================================================*/

    /**
     * Option: Controls, whether the bot may try to wander/search for an enemy.
     */
    @PogProp public boolean optionWander = true;
    /**
     * Option: Controls, whether logs from wander are to be included.
     */
    @PogProp public boolean optionWanderLogs = false;

    /*========================================================================*/

    /**
     * Option: Controls, how much the agent likes to talk.
     * Value of 1.0 is the default. Values below 1.0 decrease chance of speech,
     * values above 1.0 increase chance of speech.
     */
    @PogProp public double optionSpeech = 0.0;

    /*========================================================================*/

    /**
     * Option: Controls, whether the name should contain info about combats.
     */
    @PogProp public boolean optionInfoInName = false;

    /*========================================================================*/
    

    /*========================================================================*/
    /*========================================================================*/
    /*========================================================================*/

    /**
     * Status: Tells, what is our current weaponry score.
     */
    @PogProp public int statusInventoryScore = 0;

    /*========================================================================*/
    /*========================================================================*/
    /*========================================================================*/

    /** Instance of class responsible for memory. */
    protected LoqueMemory Memory;
    /** Instance of class responsible for agent speech. */
    protected LoqueSpeech Speech;
    /** Instance of class responsible for handling bloody combat. */
    protected LoqueCombat Combat;
    /** Instance of class responsible for powering-up. */
    protected LoquePowerUp PowerUp;
    /** Instance of class responsible for safe traveling to items. */
    protected LoqueTravel Travel;
    /** Instance of class responsible for pursuing a lost enemy. */
    protected LoquePursue Pursue;
    /** Instance of class responsible for searching for an enemy. */
    protected LoqueWander Wander;
    
    /*========================================================================*/
    /**
     * Option/Status: Should the bot collect information about weapons?
     */
    @PogProp public boolean learnWeapons = false;
    /** tells in what learning episode are we currently. -1 means, that learning is done. */
    private int learnEpisode = -1;
    /** Instance of class responsible for learning weapon stats. */
    public LoqueLearnWeapons Learn;
    /** counts the runs of doLogic() */
    private int learnTime = 0;
    /** how many doLogic runs goes into one record */
    private int learnFrequency = 8;
    private String learnOutputFolder;
    private int maxLearnTime;
    public int timeSinceWeaponChanged = 0;

    /*========================================================================*/
    /*========================================================================*/
    /*========================================================================*/

    /**
     * Main bol logic. Whatever the bot thinks and does starts here.
     */
    @Override protected void doLogic()
    {
        
        //if(Memory.self.isShooting() || Memory.self.isAltShooting())Learn.refresh();
        
        if(learnWeapons){
            if(learnTime % learnFrequency == 0){
            	++timeSinceWeaponChanged;
                Learn.refresh((int)(learnTime/learnFrequency));
            }
            if(learnTime % (learnFrequency*10) == 0){
                Combat.forceConsiderRearm();
            }
            if(learnTime  >= maxLearnTime * learnFrequency){
                Learn.processResults(learnEpisode);
                learnWeapons = false;
                stopAgentSoft();
            }
        }
        ++learnTime;
        
        log.finest ("---------------------------------------------------------");
        try
        {
            // tick some status info out..
            // note: this is not a real drive logic
            Memory.fadeAllOut ();

            // drive the combat logic..
            if (Combat.doLogic ())
            {
                log.finest ("Combat decided");
                return;
            }
            // drive the power-up logic..
            if (PowerUp.doLogic ())
            {
                log.finest ("PowerUp decided");
                return;
            }
            // drive the pursue logic..
            if (Pursue.doLogic ())
            {
                log.finest ("Pursue decided");
                return;
            }
            // drive the wander logic..
            if (Wander.doLogic ())
            {
                log.finest ("Wander decided");
                return;
            }
            // no more drives to drive
            log.severe ("DRIVES FAILED TO DECIDE");
            return;
        }
        catch (Exception e)
        {
            log.severe(cz.cuni.utils.ExceptionToString.process("Exception in logic...", e));
            stopAgentSoft ();
            return;
        }
    }

    /*========================================================================*/
    /*========================================================================*/
    /*========================================================================*/

    /**
     * Whatever's necessary to do before the bot connects to the server.
     * @throws cz.cuni.pogamut.exceptions.PogamutException
     */
    @Override protected void prePrepareAgent() throws PogamutException
    {
        // prepare initializer
        //body.initializer.setName("Loque");
        body.initializer.setBotSkillLevel(1);

        // setup speed.. note: speed command: SETGAMESPEED {Speed 0.5}
        logicFrequency = 8;

        // initialize helpers
        Memory = new LoqueMemory (this);
        Speech = new LoqueSpeech (this, Memory);
        Travel = new LoqueTravel (this, Memory);

        // initialize drives
        Pursue = new LoquePursue (this, Memory, Travel);
        Combat = new LoqueCombat (this, Memory, Travel, Pursue);
        PowerUp = new LoquePowerUp (this, Memory, Travel, Speech);
        Wander = new LoqueWander (this, Memory, Travel);
        
        //initialize learn modules
        Learn = new LoqueLearnWeapons(this, Memory, learnOutputFolder);
        
    }

    /**
     * Whatever's necessary to do after the bot connects to the server,
     * but before he spawns into the enviroment and starts thinking.
     * @throws cz.cuni.pogamut.exceptions.PogamutException
     */
    @Override protected void postPrepareAgent() throws PogamutException
    {
        // setup ray autotrace
        body.removeAllRaysFromAutoTrace ();
        body.addRayToAutoTrace (0, new Triple (1, 0, 0), 200.0, true, false);
    }

    /**
     * Whatever's necessary to do before the bot returns to cryostasis.
     * @throws cz.cuni.pogamut.exceptions.PogamutException
     */
    @Override protected void shutdownAgent() throws PogamutException
    {
        if(learnWeapons && learnTime < maxLearnTime * learnFrequency)Learn.processResults(learnEpisode);
    }

    /*========================================================================*/
    /*========================================================================*/
    /*========================================================================*/
    public void setLearningEpisode(int input, int output, String folder, int time){
        learnEpisode = output;
        learnWeapons = output > 0;
        optionCombatCheatAmmo = learnWeapons;
            
        if(input > 0)LoqueWeaponInfo.loadWeaponInfo(folder + "weapon_info" + input + ".txt");
        
        learnOutputFolder = folder;
        maxLearnTime = time;
        
    }
    
    /*========================================================================*/
    /*========================================================================*/
    /*========================================================================*/

    /**
     * Stupid netbeans requirement.
     * @param args
     */
    public static void main(String[] args) {}
}
