package studentguide;

import java.util.ArrayList;
import java.util.List;

import cz.cuni.amis.pogamut.base.agent.navigation.PathEventType;
import cz.cuni.amis.pogamut.base.agent.navigation.PathExecutorListener;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.annotation.EventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.annotation.ObjectClassEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.event.WorldObjectUpdatedEvent;
import cz.cuni.amis.pogamut.base.utils.math.DistanceUtils;
import cz.cuni.amis.pogamut.base3d.worldview.object.Location;
import cz.cuni.amis.pogamut.base3d.worldview.object.event.WorldObjectAppearedEvent;
import cz.cuni.amis.pogamut.base3d.worldview.object.event.WorldObjectDisappearedEvent;
import cz.cuni.amis.pogamut.ut2004.agent.module.utils.TabooSet;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.UTPathExecutor;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.floydwarshall.FloydWarshallPathPlanner;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.stuckdetectors.StupidStuckDetector;
import cz.cuni.amis.pogamut.ut2004.bot.impl.UT2004Bot;
import cz.cuni.amis.pogamut.ut2004.bot.impl.UT2004BotModuleController;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbcommands.Initialize;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.BotKilled;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.ConfigChange;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.EndMessage;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.GameInfo;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.GlobalChat;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.InitedMessage;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Item;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.NavPoint;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Player;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Self;
import cz.cuni.amis.pogamut.ut2004.utils.SingleUT2004BotRunner;
import cz.cuni.amis.utils.collections.MyCollections;
import cz.cuni.amis.utils.exception.PogamutException;

/**
 * Java Bot template.
 */
public class Guide extends UT2004BotModuleController {

    ///////
    //
    // CONSTANTS
    //
    ///////
    /**
     * Question to civilian whether he can see me?
     */
    public static final String QUESTION_SEE = "Can you see me?";
    /**
     * Civilian can see you.
     */
    public static final String ANSWER_SEE_YES = "Yes.";
    /**
     * Civilian can't see you.
     */
    public static final String ANSWER_SEE_NO = "No.";
    /**
     * Command the civilian to stop.
     */
    public static final String COMMAND_STOP = "Stop!";
    /**
     * Civilian stopped.
     */
    public static final String ANSWER_STOP_OK = "Okey!";
    /**
     * Command civilian to turn.
     */
    public static final String COMMAND_TURN = "Turn!";
    /**
     * Civilian is turning.
     */
    public static final String ANSWER_TURN_OK = "Turning!";
    /**
     * Command civilian to follow you.
     */
    public static final String COMMAND_FOLLOW_ME = "Follow me!";
    /**
     * Civilian is following you.
     */
    public static final String ANSWER_FOLLOW_ME_OK = "I'm right behind you!";
    /**
     * Civilian can't see you, it won't follow you.
     */
    public static final String ANSWER_FOLLOW_ME_CANT_SEE_YOU = "I can't see you, sorry.";
    /**
     * Civilian is angry with you, you've bossed him too much, it won't obey you for some time.
     */
    public static final String ANSWER_ANGRY = "Hey, don't boss me around!";
    /**
     * Message when civilian can't understand you, you're too far.
     */
    public static final String ANSWER_DONT_UNDERSTAND = "I don't understand you.";
    /**
     * Civilian is at factory, you're done!
     */
    public static final String AT_FACTORY_MSG = "Cool I'm at factory!";
    /**
     * How fast will bubble fade out over you.
     */
    public static final double BUBBLE_FADEOUT = 2;
    /**
     * When running to player we have to change the path after a while, this
     * constant is telling us how often we will recompute the path (based on the number of
     * navpoints we reach).
     */
    public static final int UPDATE_PATH_TO_PLAYER_AFTER_N_NAVPOINTS_REACHED = 3;
    /**
     * If the civilian is under this distance, it will hear you and respond to your commands.
     */
    public static final double TALKING_DISTANCE = 1000;
    /**
     * Defines what is taken as "be near civilian"
     */
    public static final double CIVILIAN_NEAR = 300;
    /**
     * How fast is the civilian moving - it is advised to leave this value alone.
     */
    public static final double CIVILIAN_MOVEMENT_SPEED = 0.65;
    /**
     * How fast the guide can move.
     */
    public static final double GUIDE_MOVEMENT_SPEED = 1;
    /**
     * Utility constant for sensor {@link Guide#isCivilianFollowing()}.
     */
    public static final int CIVILIAN_POSITION_HISTORY_LENGTH_MAX = 6;
    ///////
    //
    // VARIABLES
    //
    ///////
    /**
     * Here we store navpoints that we have already seen.
     */
    TabooSet<NavPoint> seenNavs;
    /**
     * Here we store all messages that comes from the civilian.
     */
    List<String> chat = new ArrayList<String>();
    /**
     * Civilian location in previous logic() (if civilian was visible).
     */
    Location previousCivilianLocation = null;
    /**
     * Instance of the civilian.
     */
    Player civilian = null;
    /**
     * Whether the civilian is moving.
     */
    boolean civilianIsMoving = false;
    /**
     * Whether you can see civilian.
     */
    boolean canSeeCivilian = false;
    /**
     * Position history of the civilian.
     */
    List<Location> civilianPositionHistory = new ArrayList<Location>();
    /**
     * Dunno, don't touch.
     */
    double civilianPositionHistoryTime = -1;
    /**
     * History of your (guide) locations.
     */
    List<Location> guidePositionHistory = new ArrayList<Location>();
    /**
     * Navpoint where you have to lead the civilian.
     */
    NavPoint factoryNavPoint;

    /////
    //
    // BOT INITIALIZING METHODS
    //
    /////
    /**
     * Initialize path planner as FloydWarshall...
     */
    @Override
    protected void initializePathFinding(UT2004Bot bot) {
        pathPlanner = new FloydWarshallPathPlanner(bot);
        pathExecutor = new UTPathExecutor(bot);
    }

    @Override
    public void prepareBot(UT2004Bot bot) {
        // bot has been instantiated and is being prepared to be launched into UT2004

        // uncomment to debug Pogamut platform
        //bot.getLogger().getCategory(world).setLevel(Level.ALL);

        seenNavs = new TabooSet<NavPoint>(bot);

        pathExecutor.addStuckDetector(new StupidStuckDetector(world, 3));
        pathExecutor.addPathListener(new PathExecutorListener() {

            @Override
            public void onEvent(PathEventType eventType) {
                switch (eventType) {
                    case BOT_STUCKED:
                        pathEventBotStuck(); // just recall the method to have clean interface
                        break;
                    case PATH_ELEMENT_REACHED:
                        pathEventNavPointReached(); // just recall the method to have clean interface
                        break;
                    case TARGET_REACHED:
                        pathEventTargetReached(); // just recall the method to have clean interface
                        break;
                }
            }
        });
    }

    @Override
    public Initialize getInitializeCommand() {
        return new Initialize().setName("Guide-Java");
    }

    @Override
    public void botInitialized(GameInfo gameInfo, ConfigChange currentConfig, InitedMessage init) {
        // bot has been initialized but is not spawned yet in the environment
        this.factoryNavPoint = DistanceUtils.getNearest(world.getAll(NavPoint.class).values(), world.getSingle(GameInfo.class).getFactoryLocation());
        this.config.setSpeedMultiplier(GUIDE_MOVEMENT_SPEED);
    }

    @Override
    public void botSpawned(GameInfo gameInfo, ConfigChange currentConfig, InitedMessage init, Self self) {
        // bot has been spawned for the first time into the environment
    }

    /////
    //
    // LISTENERS
    //
    /////
    /**
     * Listener that removes unexplored navpoints as the bot walks over them.
     * @param event
     */
    @ObjectClassEventListener(eventClass = WorldObjectUpdatedEvent.class, objectClass = Self.class)
    public void eventSelf(WorldObjectUpdatedEvent<Self> event) {
        NavPoint nav = DistanceUtils.getNearest(world.getAll(NavPoint.class).values(), info.getLocation());
        if (nav.getLocation().getDistance(info.getLocation()) < 100) {
            seenNavs.add(nav);
        }
        guidePositionHistory.add(event.getObject().getLocation());
        if (guidePositionHistory.size() > CIVILIAN_POSITION_HISTORY_LENGTH_MAX) {
            guidePositionHistory.remove(0);
        }
    }

    /**
     * Listener that listenes on globally said things and stores them inside {@link Guide#chat}.
     * @param event
     */
    @EventListener(eventClass = GlobalChat.class)
    public void eventGlobalChat(GlobalChat event) {
        chat.add(event.getText());
        if (event.getText().equalsIgnoreCase(ANSWER_FOLLOW_ME_OK)) {
            lastTimeCivilianFollowOk = info.getTime();
        }
    }

    /**
     * Listener that is called whenever some NavPoint appears in the field of view of the bot (becomes visible).
     * @param event
     */
    @ObjectClassEventListener(eventClass = WorldObjectAppearedEvent.class, objectClass = NavPoint.class)
    public void eventNavPointAppeared(WorldObjectAppearedEvent<NavPoint> event) {
        // put the navpoint into the set of seen navpoints - such navpoints should not need to be explored
        seenNavs.add(event.getObject());
    }

    /**
     * Listener that watch over the civilian instance, its location, history.
     * @param event
     */
    @ObjectClassEventListener(eventClass = WorldObjectAppearedEvent.class, objectClass = Player.class)
    public void eventPlayerAppeared(WorldObjectAppearedEvent<Player> event) {
        if (info.getSelf() == null) return;
        civilian = event.getObject();
        canSeeCivilian = true;
        previousCivilianLocation = null;
    }

    /**
     * Listener that watch over the civilian instance, its location, history.
     * @param event
     */
    @ObjectClassEventListener(eventClass = WorldObjectUpdatedEvent.class, objectClass = Player.class)
    public void eventPlayerUpdated(WorldObjectUpdatedEvent<Player> event) {
        if (info.getSelf() == null) return;
        if (previousCivilianLocation != null) {
            civilianIsMoving = previousCivilianLocation.getDistance(event.getObject().getLocation()) > 20;
        }
        previousCivilianLocation = event.getObject().getLocation();
        civilianPositionHistory.add(event.getObject().getLocation());
        if (civilianPositionHistory.size() > CIVILIAN_POSITION_HISTORY_LENGTH_MAX) {
            civilianPositionHistory.remove(0);
        }
        civilianPositionHistoryTime = info.getTime();
    }

    /**
     * Listener that watch over the civilian instance, its location, history.
     * @param event
     */
    @ObjectClassEventListener(eventClass = WorldObjectDisappearedEvent.class, objectClass = Player.class)
    public void eventPlayerDisappeared(WorldObjectDisappearedEvent<Player> event) {
        if (info.getSelf() == null) return;
        canSeeCivilian = false;
        civilianIsMoving = false;
        previousCivilianLocation = null;
    }

    /**
     * Listener that watch over the civilian instance, its location, history.
     * @param event
     */
    @EventListener(eventClass = EndMessage.class)
    public void eventEndMessage(EndMessage event) {
        if (civilianPositionHistoryTime != info.getTime()) {
            civilianPositionHistory.add(null);
            if (civilianPositionHistory.size() > CIVILIAN_POSITION_HISTORY_LENGTH_MAX) {
                civilianPositionHistory.remove(0);
            }
        }
    }

    /////
    //
    // BOT SENSORS
    //
    /////
    /**
     * True if the bot knows about unexplored navpoint.
     * @return Whether the bot knows about unexplored navpoint.
     */
    public boolean knowNavPointToExplore() {
        return getNavPointToExplore() != null;
    }

    /**
     * True if the bot can see a civilian.
     * @return whether the bot see a civilian
     */
    public boolean canSeePlayer() {
        return getNearestVisiblePlayer() != null;
    }

    /**
     * Returns random navpoint.
     * @return random navpoint, is always non-null
     */
    public NavPoint getRandomNavPoint() {
        return MyCollections.getRandom(world.getAll(NavPoint.class).values());
    }

    /**
     * Returns navpoint that is the nearest to the bot.
     * @return nearest navpoint to the bot, is always non-null
     */
    public NavPoint getNearestNavPoint() {
        return info.getNearestNavPoint();
    }

    /**
     * Returns NavPoint instance that should be explored (has not been explored before).
     * If all NavPoints have been explored, returns null
     * @return NavPoint to explore if such exists (otherwise returns null)
     */
    public NavPoint getNavPointToExplore() {
        NavPoint nav = DistanceUtils.getNearest(seenNavs.filter(world.getAll(NavPoint.class).values()), info.getLocation());
        if (nav != null) {
            user.info("navpoint to explore: " + nav.getId().getStringId());
            return nav;
        }
        // we have explored all navpoints
        user.info("navpoint to explore: NULL");
        return null;
    }

    /**
     * Returns nearest visible navpoint (if such exists, otherwise returns null).
     * @return nearest visible navpoint (or null if no such navpoint exists)
     */
    public NavPoint getNearestVisibleNavPoint() {
        return info.getNearestVisibleNavPoint();
    }

    /**
     * Returns Player instance in the case the bot can see Civilian.
     * @return if EmoHawk is visible, returns its instance
     */
    public Player getNearestVisiblePlayer() {
        return players.getNearestVisiblePlayer();
    }

    /**
     * Returns a distance to the nearest visible player. If no enemy is visible, returns Double.MAX_VALUE.
     * @return distance to the nearest visible enemy (or Double.MAX_VALUE if no enemy is visible)
     */
    public double getDistanceToNearestVisiblePlayer() {
        Player player = getNearestVisiblePlayer();
        return player == null ? Double.MAX_VALUE : player.getLocation().getDistance(info.getLocation());
    }

    /**
     * Tells whether the civilian will hear you if you say something.
     * @return whether the civilian will hear you
     */
    public boolean isPlayerInTalkingDistance() {
        return getDistanceToNearestVisiblePlayer() < TALKING_DISTANCE;
    }

    /**
     * Returns true if the bot is navigating through the environment (is running somewhere).
     * @return true if the bot is navigating through the environment
     */
    public boolean isMoving() {
        return moving;
    }

    /**
     * True if the bot is running to player.
     * @return whether the bot is running to player
     */
    public boolean isRunningToPlayer() {
        return runningToPlayer != null;
    }

    /**
     * True if the bot is running to some navpoint.
     * @return whether the bot is running to navpoint
     */
    public boolean isRunningToNavPoint() {
        return runningToNavPoint != null;
    }

    /**
     * If the bot is running somewhere (run has been issued via 'runToXXX' methods) than this method returns
     * how far is the destination now (changes as the bot moves towards the target).
     * @return how far is the distance to bot's target where it is running to
     */
    public double getDistanceToTarget() {
        if (runningToLocation == null) {
            return Double.MAX_VALUE;
        }
        return runningToLocation.getDistance(info.getLocation());
    }

    /**
     * Whether the bot can see the civilian.
     * @return
     */
    public boolean canSeeCivilian() {
        if (canSeeCivilian) {
            followTimeout = 1.5;
        }
        return canSeeCivilian;
    }

    /**
     * Whether the civilian is moving.
     * @return
     */
    public boolean isCivilianMoving() {
        return civilianIsMoving;
    }

    /**
     * Whether the civilian is near.
     */
    public boolean isCivilianNear() {
        return civilian == null ? false : (!civilian.isVisible() ? false : civilian.getLocation().getDistance(info.getLocation()) < CIVILIAN_NEAR);
    }

    /**
     * Returns current distance to civilian. The distance is returned only if the civilian is visible.
     * @return
     */
    public double getDistanceToCivilian() {
        Player player = getNearestVisiblePlayer();
        if (player == null) {
            return Double.MAX_VALUE;
        }
        return player.getLocation().getDistance(info.getLocation());
    }

    /**
     * Must be called together with {@link Guide#followCivilian()} to work properly.
     */
    public boolean canFollowCivilian() {
        if (canSeeCivilian()) {
            return true;
        }
        return followTimeout > 0;
    }

    //////
    //
    // KEY SENSOR - isCivilianFollowing
    //
    //////
    double lastTimeCivilianFollowOk = -100;

    /**
     * Returns whether the civilian is following you: 0 - absolutely not (or don't know), 1 - yes he is following you.
     * @return
     */
    public double isCivilianFollowing() {
        if (civilian == null) {
            return 0;
        }
        if (info.getTime() - lastTimeCivilianFollowOk < 10) {
            return 1;
        }
        double confidenceNulls = 1;
        List<Double> distances = new ArrayList<Double>();
        for (Location loc : civilianPositionHistory) {
            if (loc == null) {
                confidenceNulls -= 1 / civilianPositionHistory.size();
            } else {
                distances.add(loc.getDistance(guidePositionHistory.get(0)));
            }
        }
        if (confidenceNulls <= 0.01) {
            return 0;
        }
        if (distances.size() > 1) {
            double start = distances.get(0);
            double confidence = 0;
            for (int i = 1; i < distances.size(); ++i) {
                if (distances.get(i) + 10 < start) {
                    confidence += 1 / (CIVILIAN_POSITION_HISTORY_LENGTH_MAX - 1);
                }
                start = distances.get(i);
            }
            return confidence;
        } else {
            return 0.3;
        }
    }

    /////
    //
    // PATH EXECUTOR EVENTS HANDLING METHODS
    //
    /////
    /**
     * Called whenever the bot gets stucked.
     */
    public void pathEventBotStuck() {
        if (runningToNavPoint != null) {
            seenNavs.add(runningToNavPoint, 10);
        }
        stopMovement();
    }
    /**
     * Misc. variable - do not alter manually.
     */
    int updatePathToPlayerCounter = 3;

    /**
     * Called whenever another navpoint of the path is reached.
     */
    public void pathEventNavPointReached() {
        if (runningToPlayer != null) {
            // update path to the player when running towards him
            updatePathToPlayerCounter--;
            if (updatePathToPlayerCounter <= 0 && runningToPlayer.isVisible()) {
                runToPlayer(runningToPlayer);
            }
        }
    }

    /**
     * Called whenever the bot reaches its destination.
     */
    public void pathEventTargetReached() {
        if (runningToNavPoint != null) {
            seenNavs.add(runningToNavPoint);
        }
        stopMovement();
    }
    /////
    //
    // BOT MOVEMENT UTILITY VARIABLES
    //
    /////
    /**
     * Where the bot is running to - this variable is filled whenever on of 'runTo' methods is called
     * (either {@link Guide#runToItem(Item)} or {@link Guide#runToNavPoint(NavPoint)} or {@link Guide#runToPlayer(Player)}).
     * <p><p>
     * Do not alter manually (only in case you know what you're doing ;-)!
     */
    Location runningToLocation = null;
    /**
     * Whether the bot is moving or not.
     * <p><p>
     * Do not alter manually (only in case you know what you're doing ;-)!
     */
    boolean moving = false;
    /**
     * When the bot is running towards some player, the player instance is stored here.
     * <p><p>
     * Do not alter manually (only in case you know what you're doing ;-)!
     */
    Player runningToPlayer = null;
    /**
     * Whenever the bot is running towards some navpoint, the navpoint is stored here.
     * <p><p>
     * Do not alter manually (only in case you know what you're doing ;-)!
     */
    NavPoint runningToNavPoint = null;

    /////
    //
    // BOT ACTIONS
    //
    /////
    /**
     * Orders the bot to face the civilian - note that the bot may face civilian even if it is running somewhere!
     */
    public void faceCivilian() {
        if (civilian != null) {
            if (civilian.isVisible()) {
                user.info("Turning to civilian.");
                move.turnTo(civilian);
            } else {
                user.warning("Civilian is not visible, turning to last known position.");
                move.turnTo(civilian.getLocation());
            }
        } else {
            user.severe("Civilian has not been met, can't face it.");
        }
    }

    /**
     * Changes bot speed to match civilians speed.
     */
    public void setCivilianSpeed() {
        config.setSpeedMultiplier(CIVILIAN_MOVEMENT_SPEED);
    }

    /**
     * Changes bot speed to run at maximum speed.
     */
    public void setGuideSpeed() {
        config.setSpeedMultiplier(GUIDE_MOVEMENT_SPEED);
    }

    double followTimeout = 0;

    /**
     * The guide will try to follow the civilian (not the best implementation you can dream of).
     */
    public void followCivilian() {
        faceCivilian();
        if (canSeeCivilian()) {
            followTimeout = 1.5;
            if (getDistanceToCivilian() < 200) {
                stopMovement();
                move.moveTo(getNearestVisiblePlayer());
            } else {
                runToPlayer(getNearestVisiblePlayer());
            }
        } else {
            followTimeout -= senses.getTimeDelta();
            if (followTimeout > 0) {
                runToPlayer(civilian);
            }
        }
    }

    /**
     * Stops movement of the bot, clears bot's movement utility variables.
     */
    public void stopMovement() {
        pathExecutor.stop();
        move.stopMovement();

        moving = false;

        runningToLocation = null;
        runningToPlayer = null;
        runningToNavPoint = null;
    }

    /**
     * Starts moving to some navpoint.
     * <p><p>
     * RECALLABLE! You may repeatedly call the method with the same navpoint instance and it will work smoothly.
     * <p><p>
     * Sets bot's movement utility variables.
     * @param navpoint navpoint to run to
     */
    public void runToNavPoint(NavPoint navPoint) {
        if (runningToNavPoint == navPoint) {
            return;
        }
        user.info("Running to navpoint: " + navPoint.getId().getStringId());
        moving = true;
        runningToPlayer = null;
        runningToNavPoint = navPoint;
        runningToLocation = navPoint.getLocation();
        pathExecutor.followPath(pathPlanner.computePath(navPoint));
    }

    /**
     * Starts moving to some player.
     * <p><p>
     * RECALLABLE! You may repeatedly call the method with the same player instance and it will work smoothly.
     * <p><p>
     * Sets bot's movement utility variables.
     * @param player player to run to
     */
    public void runToPlayer(Player player) {
        if (runningToPlayer == player) {
            return;
        }
        user.info("Running to player: " + player.getId().getStringId());
        moving = true;
        runningToPlayer = player;
        runningToNavPoint = null;
        runningToLocation = player.getLocation();
        updatePathToPlayerCounter = UPDATE_PATH_TO_PLAYER_AFTER_N_NAVPOINTS_REACHED;
        pathExecutor.followPath(pathPlanner.computePath(player));
    }

    /**
     * Say something aloud - note that civilian recognize only a few messages defined in constants.
     * It is better to use commandXXX or questionXXX methods.
     * @param text
     */
    public void say(String text) {
        body.getCommunication().sendGlobalBubbleMessage(text, BUBBLE_FADEOUT);
    }

    /**
     * Commands civilian to stop.
     */
    public void commandCivilianStop() {
        user.info("Commanding civilian to stop.");
        say(COMMAND_STOP);
    }

    /**
     * Commands civilian to turn.
     */
    public void commandCivilianTurn() {
        user.info("Commanding civilian to turn.");
        say(COMMAND_TURN);
    }

    /**
     * Commands civilian to follow you.
     */
    public void commandCivilianFollowMe() {
        user.info("Commanding civilian to follow the guide.");
        say(COMMAND_FOLLOW_ME);
    }

    /**
     * Ask civilian whether he can see the guide.
     */
    public void questionCivilianCanSee() {
        user.info("Asking the civilian whether he can see the guide.");
        say(QUESTION_SEE);
    }

    ///////
    //
    // BOT UTILITY METHOD
    //
    ///////
    /**
     * Called whenever the bot is killed by the EmoHawk, reseting variables.
     */
    @Override
    public void botKilled(BotKilled event) {
        stopMovement();
        civilian = null;
        previousCivilianLocation = null;
        civilianIsMoving = false;
        civilianPositionHistory.clear();
        guidePositionHistory.clear();
    }

    /////
    //
    // CHAT HANDLING
    //
    /////
    
    /**
     * This method handles messages that comes from the civilian. It recalls 'answerXXX' methods,
     * handle respective answers there.
     */
    public void handleChat() {
        for (String text : chat) {
            if (text.equalsIgnoreCase(ANSWER_ANGRY)) {
                user.info("Civilian is angry.");
                answerAngry();
            } else if (text.equalsIgnoreCase(ANSWER_DONT_UNDERSTAND)) {
                user.info("Civilian don't understand.");
                answerDontUnderstand();
            } else if (text.equalsIgnoreCase(ANSWER_FOLLOW_ME_CANT_SEE_YOU)) {
                user.info("Civilian can't follow the guide, he does not see him.");
                answerCantFollowCantSee();
            } else if (text.equalsIgnoreCase(ANSWER_FOLLOW_ME_OK)) {
                user.info("Civilian starts to follow the guide.");
                answerFollowingOk();
            } else if (text.equalsIgnoreCase(ANSWER_SEE_NO)) {
                user.info("Civilian can't see the guide.");
                answerCantSee();
            } else if (text.equalsIgnoreCase(ANSWER_SEE_YES)) {
                user.info("Civilian can see the guide.");
                answerCanSee();
            } else if (text.equalsIgnoreCase(ANSWER_STOP_OK)) {
                user.info("Civilian stopped.");
                answerStopped();
            } else if (text.equalsIgnoreCase(ANSWER_TURN_OK)) {
                user.info("Civilian turning.");
                answerTurning();
            } else if (text.equalsIgnoreCase(AT_FACTORY_MSG)) {
                user.info("Civilian is at factory.");
                civilianReachedTheFactory();
            } else {
                // this never happens
                user.severe("Civilian responded with unknown message.");
            }
        }
        chat.clear();
    }

    /**
     * The civilian is angry with you - you're commanding him too often.
     * He will not respond to you for some time.
     */
    public void answerAngry() {
    }

    /**
     * The civilian does not recognized your message. This happens if you send custom
     * message that he does not know or when you are talking to him from greater distance
     * than is indicated by {@link Guide#isPlayerInTalkingDistance()}.
     */
    public void answerDontUnderstand() {
    }

    /**
     * The civilian can't follow you because he can't see you. You must either get nearer
     * or make him turn your way.
     */
    public void answerCantFollowCantSee() {
    }

    /**
     * The civilian is following you.
     */
    public void answerFollowingOk() {
    }

    /**
     * The civilian can't see you (response to {@link Guide#questionCivilianCanSee()}). He won't
     * follow you if you order him to do it.
     */
    public void answerCantSee() {
    }

    /**
     * The civilian can see you (resopnse to {@link Guide#questionCivilianCanSee()}). He will
     * follow you if you order him to do it.
     */
    public void answerCanSee() {
    }

    /**
     * The civilian has stopped walking. Note that he won't stand still forever.
     */
    public void answerStopped() {
    }

    /**
     * The civilian has turned a bit. Note that he won't stand still forever.
     */
    public void answerTurning() {
    }

    /**
     * The civilian has been successfully lead to the factory.
     */
    public void civilianReachedTheFactory() {
    }

    @Override
    public void logic() throws PogamutException {
        // LOG TRIVIAL STUFF (DO NOT ALTER)
        user.severe("---=== LOGIC ===---");
        user.info("Adrenalines:         " + info.getAdrenaline());
        user.info("Weapon:              " + weaponry.getCurrentWeapon());
        user.info("Ammo:                " + weaponry.getCurrentAmmo());
        user.info("Can see player:      " + canSeePlayer());
        if (isRunningToPlayer()) {
            user.info("Running to player:   " + runningToPlayer.getId().getStringId());
        }
        if (isRunningToNavPoint()) {
            user.info("Running to navpoint: " + runningToNavPoint.getId().getStringId());
        }
        if (isMoving()) {
            user.info("Distance:            " + getDistanceToTarget());
        }

        // PERFORM ACTION-SELECTION
        user.info("--- ACTION SELECTION---");

        //
        // EXAM TASK - CHAT HANDLING
        //

        // first - handle the messages from the civilian
        //         implement methods 'answerXXX()'
        handleChat();

        // EXAM TASK - IMPLEMENTATION
        
    }

    /**
     * This method is called when the bot is started either from IDE or from command line.
     * It connects the bot to the game server.
     * @param args
     */
    public static void main(String args[]) throws PogamutException {
        new SingleUT2004BotRunner<UT2004Bot>(Guide.class, "Guide").startAgent();
    }
}
