package studentemodoc;

import java.util.ArrayList;
import java.util.List;

import cz.cuni.amis.pogamut.base.agent.navigation.PathEventType;
import cz.cuni.amis.pogamut.base.agent.navigation.PathExecutorListener;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.annotation.EventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.annotation.ObjectClassEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.event.WorldObjectFirstEncounteredEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.object.event.WorldObjectUpdatedEvent;
import cz.cuni.amis.pogamut.base.utils.math.DistanceUtils;
import cz.cuni.amis.pogamut.base3d.worldview.object.Location;
import cz.cuni.amis.pogamut.base3d.worldview.object.event.WorldObjectAppearedEvent;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensomotoric.Weapon;
import cz.cuni.amis.pogamut.ut2004.agent.module.utils.TabooSet;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.UTPathExecutor;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.floydwarshall.FloydWarshallPathPlanner;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.stuckdetectors.StupidStuckDetector;
import cz.cuni.amis.pogamut.ut2004.bot.impl.UT2004Bot;
import cz.cuni.amis.pogamut.ut2004.bot.impl.UT2004BotModuleController;
import cz.cuni.amis.pogamut.ut2004.communication.messages.ItemType;
import cz.cuni.amis.pogamut.ut2004.communication.messages.UnrealId;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbcommands.Initialize;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.BotKilled;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.ConfigChange;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.EndMessage;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.GameInfo;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.InitedMessage;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Item;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.ItemPickedUp;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.NavPoint;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Player;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.PlayerKilled;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Self;
import cz.cuni.amis.pogamut.ut2004.utils.SingleUT2004BotRunner;
import cz.cuni.amis.utils.IFilter;
import cz.cuni.amis.utils.collections.MyCollections;
import cz.cuni.amis.utils.exception.PogamutException;

/**
 * Java Bot template.
 */
public class EmoDocJava extends UT2004BotModuleController {

    ///////
    //
    // CONSTANTS
    //
    ///////
    /**
     * Adrenaline respawn period in seconds.
     */
    public static final int ADRENALINE_RESPAWN_TIME = 60;
    /**
     * Other items' respawns period in seconds.
     */
    public static final int ITEM_RESPAWN_TIME = 20;
    /**
     * When running to player we have to change the path after a while, this
     * constant is telling us how often we will recompute the path (based on the number of
     * navpoints we reach).
     */
    public static final int UPDATE_PATH_TO_PLAYER_AFTER_N_NAVPOINTS_REACHED = 3;
    /**
     * Whenever an enemy EmoHawk gets under this distance,
     * method {@link EmoDoc#isEnemyInShootingDistance()} starts to report true.
     */
    public static final double SHOOTING_DISTANCE = 500;
    ///////
    //
    // VARIABLES
    //
    ///////
    /**
     * Here we store navpoints that we have already seen.
     */
    TabooSet<NavPoint> seenNavs;
    /**
     * Here we will store all adrenalines we encounter in the world.
     * Whenever an adrenaline is visible for the first time, it is added into this list.
     */
    List<Item> adrenalines = new ArrayList<Item>();
    /**
     * This taboo set counts respawn times of adrenalines,
     * whenever we pick up an adrenaline, we put it into taboo set for 60secs.
     * (that means if the adrenaline is in this taboo set than we know it is not spawned for sure)
     */
    TabooSet<Item> tabooAdrenalines;
    /**
     * Here we will store all flak cannons we encounter in the world.
     * Whenever a flak cannon is visible for the first time, it is added into this list.
     */
    List<Item> flakCannons = new ArrayList<Item>();
    /**
     * this taboo set couns respawn times for flak cannons and ammos
     * whenever we pick up a flak cannon or ammo, we put it into taboo set for 20secs
     * (that means if the item is in this taboo set than we know it is not spawned for sure)
     */
    TabooSet<Item> tabooItems;
    /**
     * Here we will store all flak cannonammos we encounter in the world.
     * Whenever a flak cannon ammo is visible for the first time, it is added into this list.
     */
    List<Item> flakCannonAmmos = new ArrayList<Item>();

    /////
    //
    // BOT INITIALIZING METHODS
    //
    /////
    @Override
    protected void initializePathFinding(UT2004Bot bot) {
        pathPlanner = new FloydWarshallPathPlanner(bot);
        pathExecutor = new UTPathExecutor(bot);
    }

    @Override
    public void prepareBot(UT2004Bot bot) {
        // bot has been instantiated and is being prepared to be launched into UT2004

        // uncomment to debug Pogamut platform
        //bot.getLogger().getCategory(world).setLevel(Level.ALL);

        seenNavs = new TabooSet<NavPoint>(bot);
        tabooAdrenalines = new TabooSet<Item>(bot);
        tabooItems = new TabooSet<Item>(bot);

        pathExecutor.addStuckDetector(new StupidStuckDetector(world, 3));
        pathExecutor.addPathListener(new PathExecutorListener() {

            @Override
            public void onEvent(PathEventType eventType) {
                switch (eventType) {
                    case BOT_STUCKED:
                        pathEventBotStuck(); // just recall the method to have clean interface
                        break;
                    case PATH_ELEMENT_REACHED:
                        pathEventNavPointReached(); // just recall the method to have clean interface
                        break;
                    case TARGET_REACHED:
                        pathEventTargetReached(); // just recall the method to have clean interface
                        break;
                }
            }
        });
    }

    @Override
    public Initialize getInitializeCommand() {
        return new Initialize().setName("EmoDoc");
    }

    @Override
    public void botInitialized(GameInfo gameInfo, ConfigChange currentConfig, InitedMessage init) {
        // bot has been initialized but is not spawned yet in the environment
    }

    @Override
    public void botSpawned(GameInfo gameInfo, ConfigChange currentConfig, InitedMessage init, Self self) {
        // bot has been spawned for the first time into the environment
    }

    /////
    //
    // LISTENERS
    //
    /////
    /**
     * Listener that removes unexplored navpoints as the bot walks over them.
     * @param event
     */
    @ObjectClassEventListener(eventClass = WorldObjectUpdatedEvent.class, objectClass = Self.class)
    public void eventSelf(WorldObjectUpdatedEvent<Self> event) {
        NavPoint nav = DistanceUtils.getNearest(world.getAll(NavPoint.class).values(), info.getLocation());
        if (nav.getLocation().getDistance(info.getLocation()) < 100) {
            seenNavs.add(nav);
        }
    }

    /**
     * Listener for encountered items in the world. This listener is called whenever the bot sees some
     * item for the first time.
     * @param event
     */
    @ObjectClassEventListener(eventClass = WorldObjectFirstEncounteredEvent.class, objectClass = Item.class)
    public void eventItemEncountered(WorldObjectFirstEncounteredEvent<Item> event) {
        // check whether an item is respawning / or has been dropped by the dead EmoHawk
        if (event.getObject().isDropped()) {
            // do not process dropped items as they won't respawn when picked up
            return;
        }
        // at this point we know that the item is "regular respawning" item, therefore it must lie on some navpoint,
        // find that navpoint
        NavPoint nav = DistanceUtils.getNearest(world.getAll(NavPoint.class).values(), event.getObject());

        // cross-link item with the navpoint and vice versa
        nav.setItem(event.getObject());
        event.getObject().setNavPoint(nav);

        if (event.getObject().getType() == ItemType.ADRENALINE_PACK) {
            // we have found an adrenaline - put it into the adrenalines list
            user.info("Adrenaline encountered: " + event.getObject().getId());
            adrenalines.add(event.getObject());
        } else if (event.getObject().getType() == ItemType.FLAK_CANNON) {
            // we have found the flak cannon - put it into the flakCannons list
            user.info("Flak cannon encountered: " + event.getObject().getId());
            flakCannons.add(event.getObject());
        } else if (event.getObject().getType() == ItemType.FLAK_CANNON_AMMO) {
            // we have found the ammo for the flak cannon - put it into flakCannonAmmos list
            user.info("Ammo encountered: " + event.getObject().getId());
            flakCannonAmmos.add(event.getObject());
        }
    }

    /**
     * Listener that is called whenever some NavPoint appears in the field of view of the bot (becomes visible).
     * @param event
     */
    @ObjectClassEventListener(eventClass = WorldObjectAppearedEvent.class, objectClass = NavPoint.class)
    public void eventNavPointAppeared(WorldObjectAppearedEvent<NavPoint> event) {
        // put the navpoint into the set of seen navpoints - such navpoints should not need to be explored
        seenNavs.add(event.getObject());
    }

    /**
     * Listener that is called whenever we pick up some item.
     * @param event
     */
    @EventListener(eventClass = ItemPickedUp.class)
    public void eventItemPickedUp(ItemPickedUp event) {
        if (event.getType() == ItemType.ADRENALINE_PACK) {
            // we've picked up an adrenaline
            user.info("Adrenaline picked up!");
            // place the adrenaline into adrenalines taboo set for a specified respawn time
            tabooAdrenalines.add(getItem(event.getId()), ADRENALINE_RESPAWN_TIME);
        } else {
            // we've picked up some other item then adrenaline
            // place it into item's taboo set for a specified respawn time
            tabooItems.add(getItem(event.getId()), ITEM_RESPAWN_TIME);
            if (event.getType() == ItemType.FLAK_CANNON) {
                // we've picked up flak cannon!
                user.info("Flak cannon picked up.");
                // rearm!
                weaponry.changeWeapon(ItemType.FLAK_CANNON);
            } else if (event.getType() == ItemType.FLAK_CANNON_AMMO) {
                // we've picked up ammo for the flak cannon
                user.info("Ammo picked up!");
            }
            // obtain an item instance we've just picked up
            Item item = world.getAll(Item.class).get(event.getId());
            if (item != null && item.isDropped()) {
                // it's dropped item! that means we won't be able ever to pick it up again
                // place it into taboo set forever
                tabooItems.add(item);
            }
        }
    }

    /**
     * Listener for the message that marks the end of bot's sensory info export.
     * @param event
     */
    @EventListener(eventClass = EndMessage.class)
    public void eventEndMessage(EndMessage event) {
        // we're going to check, whether the bot should not see some item
        // it it does (bot should see the item), but the item is still not visible
        // then it means that the item is not spawned yet, place it into taboo set
        // for a short period of time
        for (NavPoint nav : world.getAllVisible(NavPoint.class).values()) {
            if (nav.getItem() != null && !world.getAllVisible(Item.class).containsKey(nav.getItem())) {
                // examined navpoint should contain an item (first condition)
                // but the item is not visible (second condition)

                // obtain item instance
                Item item = nav.getItemInstance();
                if (item == null) {
                    // item instance is still null - we have never seen that item yet
                    return;
                }
                // check whether the item is not on the taboo list already
                if (tabooItems.isTaboo(item)) {
                    continue;
                }
                if (tabooAdrenalines.isTaboo(item)) {
                    continue;
                }

                //  NO! The item is still not spawned! Put it into correct taboo list again.
                if (item.getType() == ItemType.ADRENALINE_PACK) {
                    tabooAdrenalines.add(item, 20);
                } else {
                    tabooItems.add(item, 20);
                }
            }
        }
    }

    /**
     * Listener for EmoHawk death, called whenever the bot kills the EmoHawk.
     * @param event
     */
    @EventListener(eventClass = PlayerKilled.class)
    public void eventPlayerKilled(PlayerKilled event) {
        user.info("EmoHawk killed!");
        stopShooting();
    }

    /////
    //
    // BOT SENSORS
    //
    /////
    /**
     * True if the bot knows about unexplored navpoint.
     * @return Whether the bot knows about unexplored navpoint.
     */
    public boolean knowNavPointToExplore() {
        return getNavPointToExplore() != null;
    }

    /**
     * Returns true if the bot has discovered at least one adrenaline pickup point.
     * @return true if the bot has discovered at least one adrenaline pickup point
     */
    public boolean knowAdrenalineSpawningPoint() {
        return adrenalines.size() > 0;
    }

    /**
     * Returns true if the bot has discovered at least one flak cannon pickup point.
     * @return true if the bot has discovered at least one flak cannon pickup point
     */
    public boolean knowWeaponSpawningPoint() {
        return flakCannons.size() > 0;
    }

    /**
     * Returns true if the bot has discovered at least one flak cannon ammo pickup point.
     * @return true if the bot has discovered at least one flak cannon ammo pickup point
     */
    public boolean knowAmmoSpawningPoint() {
        return flakCannonAmmos.size() > 0;
    }

    /**
     * Returns true if the bot has discovered at least one flak cannon or flak cannon ammo pickup point.
     * @return true if the bot has discovered at least one flak cannon or flak cannon ammo pickup point
     */
    public boolean knowWeaponOrAmmoSpawningPoint() {
        return knowWeaponSpawningPoint() || knowAmmoSpawningPoint();
    }

    /**
     * True if the bot knows about spawned adrenaline.
     * @return Whether the bot knows about spawned adrenaline.
     */
    public boolean knowSpawnedWeapon() {
        return getNearestSpawnedWeapon() != null;
    }

    /**
     * True if the bot know about spawned adrenaline.
     * @return Whether the bot knows about spawned adrenaline.
     */
    public boolean knowSpawnedAdrenaline() {
        return getNearestSpawnedAdrenaline() != null;
    }

    /**
     * True if the bot knows about spawned ammo.
     * @return whether the bot knows about spawned ammo
     */
    public boolean knowSpawnedAmmo() {
        return getNearestSpawnedAmmo() != null;
    }

    /**
     * True if the bot knows aboud spawned weapon or ammo.
     * @return whether the bot knows about spawned weapon or ammo
     */
    public boolean knowSpawnedWeaponOrAmmo() {
        return knowSpawnedWeapon() || knowSpawnedAmmo();
    }

    /**
     * True if the bot can see any adrenaline that can be picked up.
     * @return Whether the bot see any adrenaline that can be picked up.
     */
    public boolean canSeeAdrenaline() {
        return getNearestVisibleAdrenaline() != null;
    }

    /**
     * True if the bot can see any weapon that can be picked up.
     * @return Whether the bot see any weapon that can be picked up.
     */
    public boolean canSeeWeapon() {
        return getNearestVisibleWeapon() != null;
    }

    /**
     * True if the bot can see any ammo that can be picked up.
     * @return Whether the bot see any ammo that can be picked up.
     */
    public boolean canSeeAmmo() {
        return getNearestVisibleAmmo() != null;
    }

    /**
     * True if the bot can see any ammo or weapon that can be picked up.
     * @return Whether the bot see any ammo or weapon that can be picked up.
     */
    public boolean canSeeAmmoOrWeapon() {
        return canSeeWeapon() || canSeeAmmo();
    }

    /**
     * True if the bot can see any enemy (EmoHawk).
     * @return whether the bot see any enemy
     */
    public boolean canSeeEnemy() {
        return getNearestVisibleEnemy() != null;
    }

    /**
     * Returns random navpoint.
     * @return random navpoint, is always non-null
     */
    public NavPoint getRandomNavPoint() {
        return MyCollections.getRandom(world.getAll(NavPoint.class).values());
    }

    /**
     * Returns navpoint that is the nearest to the bot.
     * @return nearest navpoint to the bot, is always non-null
     */
    public NavPoint getNearestNavPoint() {
        return info.getNearestNavPoint();
    }

    /**
     * Returns NavPoint instance that should be explored (has not been explored before).
     * If all NavPoints have been explored, returns null
     * @return NavPoint to explore if such exists (otherwise returns null)
     */
    public NavPoint getNavPointToExplore() {
        NavPoint nav = DistanceUtils.getNearest(seenNavs.filter(world.getAll(NavPoint.class).values()), info.getLocation());
        if (nav != null) {
            user.info("navpoint to explore: " + nav.getId().getStringId());
            return nav;
        }
        // we have explored all navpoints
        user.info("navpoint to explore: NULL");
        return null;
    }

    /**
     * Returns nearest visible navpoint (if such exists, otherwise returns null).
     * @return nearest visible navpoint (or null if no such navpoint exists)
     */
    public NavPoint getNearestVisibleNavPoint() {
        return info.getNearestVisibleNavPoint();
    }

    /**
     * Returns nearest visible adrenaline (if such exists). If no visible adrenalines exists
     * then the method returns null.
     * @return nearest visible adrenaline (or null if no adrenaline is visible)
     */
    public Item getNearestVisibleAdrenaline() {
        Item item = DistanceUtils.getNearestFiltered(world.getAllVisible(Item.class).values(), info.getLocation(), new IFilter<Item>() {

            @Override
            public boolean isAccepted(Item object) {
                return object.getType() == ItemType.ADRENALINE_PACK;
            }
        });
        if (item != null) {
            user.info("nearest visible adrenaline: " + item.getId().getStringId());
            return item;
        }
        user.info("nearest visible adrenaline: NULL");
        return null;
    }

    /**
     * Returns nearest visible weapon (if such exists). If no visible weapons exists
     * then the method returns null.
     * @return nearest visible weapon (or null if no weapon is visible)
     */
    public Item getNearestVisibleWeapon() {
        Item item = DistanceUtils.getNearestFiltered(world.getAllVisible(Item.class).values(), info.getLocation(), new IFilter<Item>() {

            @Override
            public boolean isAccepted(Item object) {
                return object.getType().getCategory() == ItemType.Category.WEAPON;
            }
        });
        if (item != null) {
            user.info("nearest visible weapon: " + item.getId().getStringId());
            return item;
        }
        user.info("nearest visible weapon: NULL");
        return null;
    }

    /**
     * Returns nearest visible ammo (if such exists). If no visible ammo exists
     * then the method returns null.
     * @return nearest visible ammo (or null if no ammo is visible)
     */
    public Item getNearestVisibleAmmo() {
        Item item = DistanceUtils.getNearestFiltered(world.getAllVisible(Item.class).values(), info.getLocation(), new IFilter<Item>() {

            @Override
            public boolean isAccepted(Item object) {
                return object.getType().getCategory() == ItemType.Category.AMMO;
            }
        });
        if (item != null) {
            user.info("nearest visible ammo: " + item.getId().getStringId());
            return item;
        }
        user.info("nearest visible ammo: NULL");
        return null;
    }

    /**
     * Returns nearest visible ammo or weapon (if such exists). If no visible ammo/weapon exists
     * then the method returns null.
     * @return nearest visible ammo/weapon (or null if no ammo/weapon is visible)
     */
    public Item getNearestVisibleAmmoOrWeapon() {
        Item item = DistanceUtils.getNearestFiltered(world.getAllVisible(Item.class).values(), info.getLocation(), new IFilter<Item>() {

            @Override
            public boolean isAccepted(Item object) {
                return object.getType().getCategory() == ItemType.Category.AMMO || object.getType().getCategory() == ItemType.Category.WEAPON;
            }
        });
        if (item != null) {
            user.info("nearest visible weapon or ammo: " + item.getId().getStringId());
            return item;
        }
        user.info("nearest visible weapon or ammo: NULL");
        return null;
    }

    /**
     * Returns Player instance in the case the bot can see EmoHawk.
     * @return if EmoHawk is visible, returns its instance
     */
    public Player getNearestVisibleEnemy() {
        return players.getNearestVisibleEnemy();
    }

    /**
     * Returns nearest spawned adrenaline (if such exists). If no spawned adrenalines exists
     * (either because the bot has not seen one yet or every discovered adrenalines have been picked)
     * then the method returns null.
     * @return nearest adrenaline that can be picked up (or null if such adrenaline does not exist)
     */
    public Item getNearestSpawnedAdrenaline() {
        Item item = DistanceUtils.getNearest(tabooAdrenalines.filter(adrenalines), info.getLocation());
        if (item != null) {
            user.info("nearest spawned adrenaline: " + item.getId().getStringId());
            return item;
        }
        user.info("nearest spawned adrenaline: NULL");
        return null;
    }

    /**
     * Returns nearest spawned weapon (if such exists). If no spawned weapon exists
     * (either because the bot has not seen one yet or every discovered weapons have been picked)
     * then the method returns null.
     * @return nearest weapon that can be picked up (or null if such adrenaline does not exist)
     */
    public Item getNearestSpawnedWeapon() {
        Item item = DistanceUtils.getNearestFiltered(world.getAll(Item.class).values(), info.getLocation(), new IFilter<Item>() {

            @Override
            public boolean isAccepted(Item object) {
                return object.getType().getCategory() == ItemType.Category.WEAPON && !tabooItems.isTaboo(object);
            }
        });
        if (item != null) {
            user.info("nearest spawned weapon: " + item.getId().getStringId());
            return item;
        }
        user.info("nearest spawned weapon: NULL");
        return null;
    }

    /**
     * Returns nearest spawned ammo (if such exists). If no spawned ammo exists
     * (either because the bot has not seen one yet or every discovered ammo have been picked)
     * then the method returns null.
     * @return nearest ammo that can be picked up (or null if such ammo does not exist)
     */
    public Item getNearestSpawnedAmmo() {
        Item item = DistanceUtils.getNearestFiltered(world.getAll(Item.class).values(), info.getLocation(), new IFilter<Item>() {

            @Override
            public boolean isAccepted(Item object) {
                return object.getType().getCategory() == ItemType.Category.AMMO && !tabooItems.isTaboo(object);
            }
        });
        if (item != null) {
            user.info("nearest spawned ammo: " + item.getId().getStringId());
            return item;
        }
        user.info("nearest spawned ammo: NULL");
        return null;
    }

    /**
     * Returns nearest spawned weapon or ammo (if such exists). If no spawned weapon or ammo exists
     * (either because the bot has not seen one yet or every discovered weapon and ammo have been picked)
     * then the method returns null.
     * @return nearest weapon or ammo that can be picked up (or null if such ammo does not exist)
     */
    public Item getNearestSpawnedWeaponOrAmmo() {
        Item item = DistanceUtils.getNearestFiltered(world.getAll(Item.class).values(), info.getLocation(), new IFilter<Item>() {

            @Override
            public boolean isAccepted(Item object) {
                return (object.getType().getCategory() == ItemType.Category.AMMO || object.getType().getCategory() == ItemType.Category.WEAPON) && !tabooItems.isTaboo(object);
            }
        });
        if (item != null) {
            user.info("nearest spawned weapon or ammo: " + item.getId().getStringId());
            return item;
        }
        user.info("nearest spawned weapon or ammo: NULL");
        return null;
    }

    /**
     * Returns an instance of the item of a given item's ID.
     * @param id item's ID, you must be sure it's an ID of the item
     * @return if ID is item's ID then it returns an Item instance, otherwise it returns null
     */
    public Item getItem(UnrealId id) {
        return world.getAll(Item.class).get(id);
    }

    /**
     * Returns instance of the weapon the bot is currently weilding. If the bot has not weapon, returns null.
     * @return weapon the bot is currently weilding (or null if bot does not have any weapon)
     */
    public Weapon getWeapon() {
        return weaponry.getCurrentWeapon();
    }

    /**
     * Returns true if the bot is armed with a weapon (flak cannon).
     * @return true if the bot is armed
     */
    public boolean hasWeapon() {
        return getWeapon() != null;
    }

    /**
     * Returns current number of the flak cannon ammo the bot has.
     * @return number of ammo for the flak cannon
     */
    public int getAmmo() {
        return weaponry.getAmmo(ItemType.FLAK_CANNON_AMMO);
    }

    /**
     * Returns true if the bot has at least one ammo for the flak cannon
     * @return true if the bot has at least one ammo
     */
    public boolean hasAmmo() {
        return weaponry.getAmmo(ItemType.FLAK_CANNON_AMMO) > 0;
    }

    /**
     * Returns true if the bot can shoot (has loaded weapon).
     * @return true if the bot can shoot
     */
    public boolean canShoot() {
        return hasWeapon() && hasAmmo();
    }

    /**
     * Returns a distance to the nearest visible enemy. If no enemy is visible, returns Double.MAX_VALUE.
     * @return distance to the nearest visible enemy (or Double.MAX_VALUE if no enemy is visible)
     */
    public double getDistanceToNearestVisibleEnemy() {
        Player enemy = getNearestVisibleEnemy();
        return enemy == null ? Double.MAX_VALUE : enemy.getLocation().getDistance(info.getLocation());
    }

    /**
     * Returns true if the nearest enemy gets into the shooting distance.
     * @return
     */
    public boolean isEnemyInShootingDistance() {
        return getDistanceToNearestVisibleEnemy() < SHOOTING_DISTANCE;
    }

    /**
     * Returns current level of bot's adrenaline.
     * @return current level of bot's adrenaline
     */
    public int getAdrenaline() {
        return info.getAdrenaline();
    }

    /**
     * Returns true if the bot is shooting.
     * @return true if the bot is shooting
     */
    public boolean isShooting() {
        return info.isShooting();
    }

    /**
     * Returns true if the bot is navigating through the environment (is running somewhere).
     * @return true if the bot is navigating through the environment
     */
    public boolean isMoving() {
        return moving;
    }

    /**
     * True if the bot is running to item.
     * @return whether the bot is running for the item
     */
    public boolean isRunningToItem() {
        return isMoving() && runningToItem != null;
    }

    /**
     * True if the bot is running to player.
     * @return whether the bot is running to player
     */
    public boolean isRunningToPlayer() {
        return runningToPlayer != null;
    }

    /**
     * True if the bot is running to some navpoint.
     * @return whether the bot is running to navpoint
     */
    public boolean isRunningToNavPoint() {
        return runningToNavPoint != null;
    }

    /**
     * If the bot is running somewhere (run has been issued via 'runToXXX' methods) than this method returns
     * how far is the destination now (changes as the bot moves towards the target).
     * @return how far is the distance to bot's target where it is running to
     */
    public double getDistanceToTarget() {
        if (runningToLocation == null) {
            return Double.MAX_VALUE;
        }
        return runningToLocation.getDistance(info.getLocation());
    }

    /**
     * Returns true if bot needs 'itemToSwitchTo' more than 'runningToItem'.
     * <p><p>
     * IT IS ADVISED THAT YOU ADJUST THIS METHOD TO SUIT YOUR NEEDS!
     * <p><p>
     * It compares 'itemToSwitchTo' with item the bot is currently pursuing. If the bot
     * is not pursuing any item then method returns true.
     *
     * @param itemToSwitchTo item the bot might need
     */
    public boolean wantToSwitchToItem(Item itemToSwitchTo) {
        if (runningToItem == null) {
            return true;
        }
        if (runningToItem.getType() == ItemType.FLAK_CANNON) {
            return false;
        }
        if (runningToItem.getType() == ItemType.FLAK_CANNON_AMMO) {
            if (itemToSwitchTo.getType() == ItemType.FLAK_CANNON) {
                return true;
            }
        }
        return false;
    }

    /////
    //
    // PATH EXECUTOR EVENTS HANDLING METHODS
    //
    /////
    /**
     * Called whenever the bot gets stucked.
     */
    public void pathEventBotStuck() {
        stopMovement();
    }
    /**
     * Misc. variable - do not alter manually.
     */
    int updatePathToPlayerCounter = 3;

    /**
     * Called whenever another navpoint of the path is reached.
     */
    public void pathEventNavPointReached() {
        if (runningToPlayer != null) {
            // update path to the player when running towards him
            updatePathToPlayerCounter--;
            if (updatePathToPlayerCounter <= 0 && runningToPlayer.isVisible()) {
                runToPlayer(runningToPlayer);
            }
        }
    }

    /**
     * Called whenever the bot reaches its destination.
     */
    public void pathEventTargetReached() {
        if (runningToItem != null) {
            // perform additional step forward to take the item
            move.moveTo(runningToItem.getLocation());
        }
        stopMovement();
    }
    /////
    //
    // BOT MOVEMENT UTILITY VARIABLES
    //
    /////
    /**
     * Where the bot is running to - this variable is filled whenever on of 'runTo' methods is called
     * (either {@link EmoDoc#runToItem(Item)} or {@link EmoDoc#runToNavPoint(NavPoint)} or {@link EmoDoc#runToPlayer(Player)}).
     * <p><p>
     * Do not alter manually (only in case you know what you're doing ;-)!
     */
    Location runningToLocation = null;
    /**
     * Whether the bot is moving or not.
     * <p><p>
     * Do not alter manually (only in case you know what you're doing ;-)!
     */
    boolean moving = false;
    /**
     * When the bot is running for some item, the item instance is stored here.
     * <p><p>
     * Do not alter manually (only in case you know what you're doing ;-)!
     */
    Item runningToItem = null;
    /**
     * When the bot is running towards some player, the player instance is stored here.
     * <p><p>
     * Do not alter manually (only in case you know what you're doing ;-)!
     */
    Player runningToPlayer = null;
    /**
     * Whenever the bot is running towards some navpoint, the navpoint is stored here.
     * <p><p>
     * Do not alter manually (only in case you know what you're doing ;-)!
     */
    NavPoint runningToNavPoint = null;

    /////
    //
    // BOT ACTIONS
    //
    /////
    /**
     * Stops movement of the bot, clears bot's movement utility variables.
     */
    public void stopMovement() {
        pathExecutor.stop();
        move.stopMovement();

        moving = false;

        runningToLocation = null;

        runningToItem = null;
        runningToPlayer = null;
        runningToNavPoint = null;
    }

    /**
     * Starts moving to some item.
     * <p><p>
     * RECALLABLE! You may repeatedly call the method with the same item instance and it will work smoothly.
     * <p><p>
     * Sets bot's movement utility variables.
     * @param item item to run to
     */
    public void runToItem(Item item) {
        if (runningToItem == item) {
            return;
        }
        user.info("Running for item: " + item.getId().getStringId());
        moving = true;
        runningToItem = item;
        runningToPlayer = null;
        runningToNavPoint = null;
        runningToLocation = item.getLocation();
        pathExecutor.followPath(pathPlanner.computePath(item));
    }

    /**
     * Starts moving to some navpoint.
     * <p><p>
     * RECALLABLE! You may repeatedly call the method with the same navpoint instance and it will work smoothly.
     * <p><p>
     * Sets bot's movement utility variables.
     * @param navpoint navpoint to run to
     */
    public void runToNavPoint(NavPoint navPoint) {
        if (runningToNavPoint == navPoint) {
            return;
        }
        user.info("Running to navpoint: " + navPoint.getId().getStringId());
        moving = true;
        runningToItem = null;
        runningToPlayer = null;
        runningToNavPoint = navPoint;
        runningToLocation = navPoint.getLocation();
        pathExecutor.followPath(pathPlanner.computePath(navPoint));
    }

    /**
     * Starts moving to some player.
     * <p><p>
     * RECALLABLE! You may repeatedly call the method with the same player instance and it will work smoothly.
     * <p><p>
     * Sets bot's movement utility variables.
     * @param player player to run to
     */
    public void runToPlayer(Player player) {
        if (runningToPlayer == player) {
            return;
        }
        user.info("Running to player: " + player.getId().getStringId());
        moving = true;
        runningToItem = null;
        runningToPlayer = player;
        runningToNavPoint = null;
        runningToLocation = player.getLocation();
        updatePathToPlayerCounter = UPDATE_PATH_TO_PLAYER_AFTER_N_NAVPOINTS_REACHED;
        pathExecutor.followPath(pathPlanner.computePath(player));
    }

    /**
     * Bot will start shooting at the player.
     * <p><p>
     * RECALLABLE! You may repeatedly call the method with the same player instance and it will work smoothly.
     *
     * @param player player to shoot at
     */
    public void shoot(Player player) {
        if (!canShoot()) {
            user.warning("CAN NOT SHOOT! But trying nevertheless...");
        } else {
            user.info("Shooting at: " + player.getId().getStringId());
        }
        shoot.shoot(player);
    }

    /**
     * Bot will stop shooting.
     */
    public void stopShooting() {
        shoot.stopShooting();
    }

    /////
    //
    // BOT LOGIC
    //
    /////
    @Override
    public void logic() throws PogamutException {
        // LOG TRIVIAL STUFF (DO NOT ALTER)
        user.severe("---=== LOGIC ===---");
        user.info("Adrenalines:        " + info.getAdrenaline());
        user.info("Weapon:             " + weaponry.getCurrentWeapon());
        user.info("Ammo:               " + weaponry.getCurrentAmmo());
        user.info("Can see enemy:      " + (getNearestVisibleEnemy() != null));
        if (isRunningToItem()) {
            user.info("Running for item:   " + runningToItem.getId().getStringId());
        }
        if (isRunningToPlayer()) {
            user.info("Running to enemy:   " + runningToPlayer.getId().getStringId());
        }
        if (isRunningToNavPoint()) {
            user.info("Running to navpoint: " + runningToNavPoint.getId().getStringId());
        }
        if (isMoving()) {
            user.info("Distance:           " + getDistanceToTarget());
        }

        // PERFORM ACTION-SELECTION
        user.info("--- ACTION SELECTION---");

        //
        // EXAM TASK SOLUTION
        //

		// START TO CODE HERE
		// Be sure to check 'wantToSwitchToItem' method.
        
    }

    /**
     * Called whenever the bot is killed by the EmoHawk, reseting variables.
     */
    @Override
    public void botKilled(BotKilled event) {
        stopMovement();
        stopShooting();
    }

    /**
     * This method is called when the bot is started either from IDE or from command line.
     * It connects the bot to the game server.
     * @param args
     */
    public static void main(String args[]) throws PogamutException {
        new SingleUT2004BotRunner<UT2004Bot>(EmoDocJava.class, "EmoDoc").startAgent();
    }
}
