package raycastingbot;

import cz.cuni.pogamut.Client.Agent;
import cz.cuni.pogamut.Client.RcvMsgEvent;
import cz.cuni.pogamut.Client.RcvMsgListener;
import cz.cuni.pogamut.MessageObjects.AutoTraceRay;
import cz.cuni.pogamut.MessageObjects.MessageType;
import cz.cuni.pogamut.MessageObjects.Triple;
import cz.cuni.pogamut.exceptions.PogamutException;
import java.util.HashMap;
import java.util.Map;

/**
 * This class may serve as an example how to create a raycasting bot
 * that defines a several rays.
 * 
 * Currently there are some problem with the platform and the coding is not
 * as clear as we would like...
 */
public class Main extends Agent {

    /** Creates a new instance of agent. */
    public Main() {
    }
    
    /**
     * First we will create an enum that will contain all the rays we are
     * going to use.
     * 
     * Every ray is described as triple:
     *      ID ... number under which it will be returned from GB
     *      Vector ... where the vector is going from the bot as 0,0,0
     *             ... 1, 0, 0 go straight ahead from the bot
     *             ... 0, 1, 0 go to the left from the bot
     *             ... 0,-1, 0 go to the right from the bot
     *             ... note that the vector serves only as direction, length is not taken into account
     *      Length ... how far we're tracing
     */
    public enum Ray {
        
        /**
         * Ray that is going straight ahead before the bot and is 200 long.
         */
        STRAIGHT_AHEAD_200(1, new Triple(1, 0, 0), 200),
        
        /**
         * Ray that goes 45 degrees to the left of the bot.
         */
        LEFT45_200(2, new Triple(1, 1, 0), 200),
        
        /**
         * Ray that goes to the left of the bot.
         */
        LEFT90_200(3, new Triple(0, 1, 0), 200),
        
        /**
         * Ray that goes 45 degrees to the right of the bot.
         */
        RIGHT45_200(4, new Triple(1, -1, 0), 200),
        
        /**
         * Ray that goes to the right of the bot.
         */
        RIGHT90_200(5, new Triple(0, -1, 0), 200);
        
        private int id;
        private Triple vector;
        private double length;
        
        private Ray(int id, Triple vector, double length) {
            this.id = id;
            this.vector = vector;
            this.length = length;
        }
        
        public int getId() {
            return id;
        }
        
        public Triple getVector() {
            return vector;
        }
        
        public double getLength() {
            return length;
        }
        
    }
    
    /**
     * Next we will declare an inner class that will take care of the traces.
     * It will listen on the ATR messages and write them to it's map (therefore
     * it implements RcvMsgListener).
     */
    private class RayTraces implements RcvMsgListener {
        
        /**
         * Here we will store the last results of raycastings.
         * 
         *  Ray's ID -> result
         * (Integer  -> AutoTraceRay)
         */
        private Map<Integer, AutoTraceRay> traces = new HashMap<Integer, AutoTraceRay>();
        
        public RayTraces() {            
        }
        
        /**
         * This method will be called from doLogic() only once to initialize the object.
         */
        public void botInit() {
            log.info("RayTraces.botInit(): called");
            
            // tells the GameBots we want autotracing
            body.startAutoTrace();
            
            // remove default auto trace rays
            body.removeAllRaysFromAutoTrace();
            
            // now 1) register every ray inside GameBots and
            //     2) create initial value for every ray
            for (Ray ray : Ray.values()) {
                body.addRayToAutoTrace(ray.getId(), ray.getVector(), ray.getLength(), 
                                       false, // whether this is FastTrace ... NO we want full trace
                                              // to get HitNormal informations
                                       false  // whether we should trace the players and helaths as well
                                              // NO ... we want only walls, floors, etc.
                                      );
                traces.put(ray.getId(), new AutoTraceRay());
            }
            
            // register itself as a listener for ATR messages, so 
            // we can catch the AUTO_TRACE_RAY
            body.addTypedRcvMsgListener(this, MessageType.AUTO_TRACE_RAY);
            
            // now a little workarounds ...
            
            // we have to move a bot a bit to get first readings from rays
            body.moveInch();
            
            // now, to see only new rays (not the default ones), we have to switch
            // visibility of autotraces off and on again (with little delay)
            body.configureAutoTrace(false);
            try {
                Thread.sleep(200);
            } catch (InterruptedException ex) {               
            }
            body.configureAutoTrace(true);

        }

        /**
         * In this method we're receiving notices about ATR messages. It's
         * called every time when ATR message arrives.
         * @param e
         */
        @Override
        public void receiveMessage(RcvMsgEvent e) {
            // get the message from the event (casting it properly)
            AutoTraceRay ray = (AutoTraceRay)e.getMessage();
            try {
                // be synchronized with 'traces' to prevent concurrent read/write operations
                synchronized(traces) {
                    // insert the ray under it's number
                    // notice that ID goes under UnrealID field in the 'ray'
                    traces.put(Integer.parseInt(ray.UnrealID), ray);
                }
            } catch (Exception ex) {
                log.severe(ex.getMessage());
            }
        }
        
        //
        // now follows methods for getting the traces out of the object
        // 
        
        /**
         * Returns info about one ray.
         * @param ray
         * @return
         */
        public AutoTraceRay getTrace(Ray ray) {
            synchronized(traces) {
                return traces.get(ray.getId());
            }            
        }
        
        /**
         * Returns the copy of the map of traces.
         * 
         * This will be useful for you when you will program the
         * control mechanism of the bot. You shouldn't rely on the getTrace()
         * because two calls of it may return different result (because of 
         * a two thread design of the bot).
         * 
         * @return current state of traces
         */
        public Map<Integer, AutoTraceRay> getTraceSnapshot() {
            synchronized(traces) {
                return new HashMap<Integer, AutoTraceRay>(traces);
            }
        }
        
    }
    
    /**
     * create an instance of our RayTracing manager
     */
    private RayTraces traces = new RayTraces();
    
    /**
     * Flag that tells us whether we should initialize things in doLogic().
     */
    private boolean init = true;
    
    /**
     * This is initiliazation method for the bot, it's called only once from doLogic()
     */    
    private void init() {        
        // we're initializing the traces here
        traces.botInit();
    }
    
    /**
     * Main method of the agent that is called iteratively from logic thread.
     */
    protected void doLogic() {
        // if we should init
        if (init) {
            //so do it... 
            init(); 
            // only once (drop the flag afterwards).
            init = false; 
        }
        
        // mark the beginning of the new logic iteration
        log.warning("LOGIC ITERATION");

        // get current snapshot of rays
        Map<Integer, AutoTraceRay> rays = traces.getTraceSnapshot();
        for (AutoTraceRay ray : rays.values()) {
            // log them
            log.info(ray.toString());
        }              
        
        // now you may write code like this...
        // obviously this is pretty stupid thing to do,
        // but it's example ;-)
        if (rays.get(Ray.STRAIGHT_AHEAD_200.getId()).result) {
            // if the ray that is going STRAIGHT_AHEAD returns true ... turn to the right
            body.turnHorizontal(90);
            return;
        } else 
        if (rays.get(Ray.LEFT90_200.getId()).result) {    
            body.turnHorizontal(15);
            return;
        } else 
        if (rays.get(Ray.LEFT45_200.getId()).result) {    
            body.turnHorizontal(45);
            return;
        } else 
        if (rays.get(Ray.RIGHT90_200.getId()).result) {    
            body.turnHorizontal(-15);
            return;
        } else 
        if (rays.get(Ray.RIGHT45_200.getId()).result) {    
            body.turnHorizontal(-45);
            return;
        }
        
        // if no turning takes place ... move forward continuously 
        body.contMove((float)0.7);
    }

    protected void prePrepareAgent() throws PogamutException {
    /* Prepares agent logic to run - like initializing neural networks etc.
    not for establishing communication! */

    }

    protected void postPrepareAgent() throws PogamutException {
    /* Prepare logic according to information from gathered from startCommunication
    like choosing plan/parameters according to game type. */
        
        
    }

    protected void shutdownAgent() throws PogamutException {
    // Clean up after the end of simulation of agent

    }

    public static void main(String[] args) {
    /*
    DON'T DELETE THIS METHOD, IF YOU DELETE IT NETBEANS WON'T LET YOU RUN THIS 
    BOT. HOWEVER THIS METHOD IS NEVER EXECUTED, THE BOT IS LAUNCHED INSIDE THE 
    NETBEANS BY A CUSTOM ANT TASK (see build.xml).
     */
    }
}
