"""
 This file serves as an example how to create your own SPOSH bot
 It doesn't do much - just showing the basics - functionality is close to 
 the functionality of Java advancedBot in exampleBots

 There are several methods which need to be present in this file:
 	  getBehaviours(), getPathToPlanFile(),
 	  prePrepareAgent(), postPrepareAgent(), sposhAgentCreated(), shutdownAgent()
 	  

 Note that before this file is evaluated the bindings to the Agent instance is created
 in the global scope - variables agent, body, memory, map, log are already present.
 
 Also variable userDir is present (because of problem with os module), it contains
 the path to this file (without the file ... e.g. "c:\myBot")

 The file is evaluated only once.
 
 Sequence of initialization of the agent:
 
 1) connect to GameBots
 2) called prePrepareAgent()
 3) receive navpoints and items from GameBots
 4) postPrepareAgent()
 5) posh engine is initialized - getBehaviours() is called
 6) sposhAgentCreated()
 
 ...
 
 Agent is initalized (then the bot is spawned a sposh engine will start working)

"""
from cz.cuni.pogamut.MessageObjects.MessageType import *
from sposh.behaviour import Behaviour
from math import sqrt

def distanceInSpace(a, b):
    distance = (a.x - b.x) * (a.x - b.x) + (a.y - b.y) * (a.y - b.y) + (a.z - b.z) * (a.z - b.z)
    return sqrt(distance)

def getBehaviours():
	"""This must return a list or sequence of Behaviours (descendants of sposh.Behaviour)
	They will be passed to the sposh.Agent constructor, which will takes the senses and
	acts and regiester them into the SPOSH engine (e.g. Agent class)	
	"""    
	global basicBehaviour
	return [basicBehaviour]
	
def getPlanFile():
	"""This method must return (path to) the plan file for the POSH engine.	
	"""  
	global userDir
	return userDir + "\\" + "exampleSPoshBotPlan.lap"
	
def prePrepareAgent():
    """This method is called before the bot starts communication with gamebots (UT).
    You may use it to initialize your objects.
    """
    # note that 'agent', 'body', 'memory', 'map', 'log' are registered by Pogamut before loading this script
    # those are bindings to the agent in the Java
    # agent  -> cz.cuni.pogamut.Client.Agent
    # body   -> cz.cuni.pogamut.Client.AgentBody
    # memory -> cz.cuni.pogamut.Client.AgentMemory
    # map    -> cz.cuni.pogamut.Client.GameMap
    # log    -> java.util.logging.Logger

    global log
    log.info("prePrepareAgent() called")
    global basicBehaviour, agent, body, memory, map
    basicBehaviour = BasicBehaviour(agent, body, memory, map, log)	
    
	
def postPrepareAgent():
	"""This method is called after the bot receives all the navpoints and items from gamebots (UT)."""
	# you may do some map preprocessing here if you wish or load informations about the map / previous session
	global log, basicBehaviour
	basicBehaviour.knownArmors = basicBehaviour.memory.getKnownArmors()
	basicBehaviour.knownMedkits = basicBehaviour.memory.getKnownHealths()
	basicBehaviour.knownWeapons = basicBehaviour.memory.getKnownWeapons()
	log.info("postPrepareAgent() called")
	
def sposhAgentCreated():
    """This methos is called after the sposh.Agent is instantiated (sposhAgent global variable is created)."""
    #  You may use it to assing a specific time to an engine or adjust it's frequency of runs
    global log
    log.info("sposhAgentCreated() called")
    global sposhAgent
    sposhAgent.setRealTimeTimer(long(1000 / 200)) # 1000 secs. / 200 secs = 5 Hz frequency
    
def shutdownAgent():
	"""This method is called when the bot is terminated."""
	# this is a place for saving your work
	global log
	log.info("shutdownAgent() called")
	
	
#
# global variable of our BasicBehaviour				
#
basicBehaviour = None			
			
#
# BasicBehaviour class ... here is the implementation of acts and senses of the ExampleSPoshBot
# 
	
class BasicBehaviour(Behaviour):

    def __init__(self, agent, body, memory, map, log):
    
        # initialization of the predescator
        Behaviour.__init__(self, log) 
        
        # Java bindings to the agent internals        
        self.agent = agent
        self.body = body
        self.memory = memory
        self.map = map
        
        # bot internal variables
        self.knownArmors = None 	# list of known armors in the map
        self.suitableAmmos = None	# list of known ammos in the map
        self.knownMedkits = None	# list of known medkits in the map
        self.knownWeapons = None	# list of known weapons in the map
        self.enemy = None			# current enemy
        self.item = None			# current desired item
        self.selectedMedkit = None	# current desired medKit
        self.pathRestart = 0		# whether to restart path or not
        
    def getActions(self):
        """Returns a list of available actions (strings).
        
        @return: List of behaviour actions.
        @rtype: sequence of strings
        """
        return [ "stopShoot", "runMedkits", "runAround", "runItem", "runGuns", "runArmors", "sayCrap", 
                  "rotate", "jump", "runToSuitableAmmo", "pursue", "doNothing", "rearm"]
    
    def getSenses(self):
        """Returns a list of available senses (strings)
        
        @return: List of behaviour senses.
        @rtype: sequence of strings
        """
        return ["health", "seeEnemy", "seeItemAndWantIt", "noEnemy", "hasLoadedWeapon", "knowAnyMedkit", "succeed", "fail", 
                 "numOfLoadedWeapons", "knowAnyArmor", "armor", "hitWall", "damaged", "isStucked", "isShooting", 
                 "knowSuitableAmmo", "hasNotSelf", "betterWeapon"]
        
    #
    # SENSES IMPLEMENTATION
    #   
    
    def hitWall(self):
        return self.memory.isColliding()
        
    def damaged(self):
        return self.memory.isBeingDamaged()
         
    def health(self):
        return self.memory.agentHealth()
        
    def seeEnemy(self):
        return self.memory.seeAnyEnemy()
        
    def seeItemAndWantIt(self):
        if (self.memory.seeAnyReachableItem()):
            self.item = self.memory.seeItem()
        else:
            self.item = None
        if self.item == None or distanceInSpace(self.item.location, self.memory.agentLocation()) < 30:
            return 0
        else:
            return 1
            
    def noEnemy(self):
        return not self.memory.seeAnyEnemy()   
        
    def hasLoadedWeapon(self):
        ok = self.memory.hasLoadedWeapon() # check whether we have any weapon to fight with
        if not ok:
            return 0 # if not -> failure
        if self.memory.agentAmmo == 0:      # if our current weapon is out of ammo
            self.body.changeToBestWeapon() # change to the best weapon available
        return 1
        
    def numOfLoadedWeapons(self):
        self.memory.numberOfLoadedWeapons()
        
    def knowAnyMedkit(self):
        list = self.memory.getKnownHealths()
        return list.size() != 0
        
    def knowAnyArmor(self):
        list = self.memory.getKnownArmors()
        return list.size() != 0
	# todo: test
    def knowSuitableAmmo(self):
        ammos = self.memory.getKnownAmmos()
        weapons = self.memory.getAllWeapons()
        for ammo in ammos:
           for weapon in weapons:
              if weapon.weaponType.equals(ammo.weaponType):
                  return 1
        return 0
    
    def armor(self):
        return self.memory.agentArmor()
    
    def isStucked(self):
        return not self.memory.isMoving()
        
    def isShooting(self):
        if self.memory.isShooting():
            return 1
        return 0
        
    def succeed(self):        
        return 1
	
    def fail(self):	
        return 0    
    
    def hasNotSelf(self):
        if self.memory.hasSelf():
           return 0
        return 1
    
    def getBetterWeapon(self):
        enemyLocation = self.memory.seeEnemy().location
        botLocation = self.memory.agentLocation()
        weapon = None
        if distanceInSpace(enemyLocation, botLocation) > 5000: # just a guess how far is melee becoming ranged
            weapon = self.memory.getRangedWeapon()
        else:
            weapon = self.memory.getMeleeWeapon()
        if weapon == None:
            weapon = self.memory.getBetterWeapon(botLocation, enemyLocation)
        if weapon == None or weapon.weaponType.equals(self.memory.currentWeapon().weaponType):
            #self.body.sendGlobalMessage("rearming to weapon" . weapon)
            return None
        return weapon
        
    def betterWeapon(self):
        if self.getBetterWeapon() != None:
            return 1
        else:
            return 0
        
    #
    # ACTIONS IMPLEMENTATION
    # 
    
    def stopShoot(self):
        self.body.stopShoot()
        return 1
    
    """ Run around medKits in the map - all methods belowa are about the same
    """
    def runMedkits(self):
        if self.pathRestart:
            self.map.restartPathToRunAlong()
            self.pathRestart = 0
        self.map.runAroundItemsInTheMap(self.knownMedkits, 0)
        return 1
        
    def runArmors(self):
        if self.pathRestart:
            self.map.restartPathToRunAlong()
            self.pathRestart = 0
        print knownArmors
        self.map.runAroundItemsInTheMap(self.knownArmors, 0)
        return 1

    def runGuns(self):
        if self.pathRestart:
            self.map.restartPathToRunAlong()
            self.pathRestart = 0
        self.map.runAroundItemsInTheMap(self.knownWeapons, 0)
        return 1
    """ run around makes bot run to closest navPoint he sees or turn around if he doesn't see any
    """
    def runAround(self):
        if self.memory.agentIsMoving():
            return 1
        self.pathRestart = 1
        navpoint = self.memory.seeNavPoint()
        if navpoint == None:
            navpoint = self.map.nearestNavPoint(self.memory.agentLocation(),50)
        if navpoint == None:
            self.body.turnHorizontal(70)
            return 1
        self.body.runToNavPoint(navpoint)
        return 1
    """ complex method to pursue enemy
		1) choose enemy
		2) shoot him
		3) no ammo - change weapon
		4) enemy far - run to him
    """
    def pursue(self):
        self.pathRestart = 1
        # choose enemy
        if self.enemy != None:
           self.enemy = self.memory.seePlayer(self.enemy.ID)
           if self.enemy == None and self.memory.isShooting():
               self.body.stopShoot()
        if self.enemy == None:
           self.enemy = self.memory.seeEnemy()
           if self.enemy == None:
              self.body.stop()
              self.body.stopShoot()
              return 0
        # shoot enemy if not shooting
        if not self.memory.isShooting():
           self.body.shoot(self.enemy)
        # no ammo - change weapon
        if self.memory.agentAmmo() == 0 and self.memory.hasLoadedWeapon():
            self.body.changeWeapon(self.memory.getAnyWeapon())
        if self.enemy == None or self.enemy.location == None:
            return 0;
        # enemy to far, run to him
        if distanceInSpace(self.memory.agentLocation(), self.enemy.location) < 400:
            if self.memory.isMoving():
                self.body.stop()      
            else:
                self.body.runToTarget(self.enemy) 
        return 1
    
    def runItem(self):
        pathRestart = 1
        self.body.runToLocation(self.item.location)
        return 1
    
    def runToSuitableAmmo(self):
        ammos = self.memory.getKnownAmmos()
        weapons = self.memory.getAllWeapons()
        self.suitableAmmos = []
        for ammo in ammos:
           for weapon in weapons:
              if weapon.weaponType.equals(ammo.weaponType):
                  self.suitableAmmos.append(ammo)
        if self.suitableAmmos.size() > 0:
           self.map.runAroundItemsIntTheMap(self.suitableAmmos)
           return 1
        else:
           return 0

            
    def sayCrap(self):
        self.body.sendGlobalMessage("Crap, I have no ammo, I am doomed!:)")
        return 1
        
    def jump(self):
        self.body.jump()
        return 1
        
    def rotate(self):
        self.body.turnHorizontal(100)
        return 1
        
    def doNothing(self):
        return 1    
        
    def rearm(self):
        weapon = self.getBetterWeapon()
        if weapon != None:
           self.body.changeWeapon(weapon)
        
        
        
        
        
        
        
        
        
        