package cz.cuni.amis.pogamut.base.communication.parser;

import com.google.inject.ImplementedBy;
import com.google.inject.Inject;

import cz.cuni.amis.pogamut.base.factory.guice.AgentScoped;
import cz.cuni.amis.pogamut.base.utils.logging.AgentLogger;
import cz.cuni.amis.pogamut.base.utils.logging.LogCategory;
import cz.cuni.amis.utils.ExceptionToString;

/**
 * Interface for reporting Yylex exceptions.
 * @author Jimmy
 *
 */
@ImplementedBy(IYylexObserver.LogObserver.class)
@AgentScoped
public interface IYylexObserver {

	/**
	 * Called whenever exception occures in the Yylex caused by OUR miscoding.
	 * (Note that we're not hacking Yylex code to report all it's exceptions
	 * through this observer - we will call this observer only from our 
	 * parts of the code from yylex.java)
	 * 
	 * @param e never null
	 * @param info never null
	 */
	public void exception(Exception e, String info);
	
	/**
	 * Called when some mischief happens but the parser recovers. (Like
	 * wrong parsing of the message, skipping to next...)
	 * @param info
	 */
	public void warning(String info);
		
	/**
	 * Default implementation of the IYylexObserver logging everything into AgentLogger.platform()
	 * log category.
	 *  
	 * @author Jimmy
	 */
	@AgentScoped
	public static class LogObserver implements IYylexObserver {
		
		private LogCategory log;
		
		@Inject
		public LogObserver(AgentLogger agentLogger) {
			log = agentLogger.platform();
		}

		@Override
		public void exception(Exception e, String info) {
			log.severe(ExceptionToString.process(info, e));
		}

		@Override
		public void warning(String info) {
			log.warning(info);			
		}
		
	}

}
