package cz.cuni.amis.pogamut.ut2004.pcomm.server;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.logging.Logger;

import org.apache.mina.common.IdleStatus;
import org.apache.mina.common.IoSession;

import cz.cuni.amis.pogamut.ut2004.pcomm.message.PCommExceptionMessage;
import cz.cuni.amis.pogamut.ut2004.pcomm.message.PCommMessage;
import cz.cuni.amis.pogamut.ut2004.pcomm.message.PCommRegisterMessage;
import cz.cuni.amis.pogamut.ut2004.pcomm.message.PCommRegisteredMessage;

public class PCommSession implements IPCommSession {
	
	//
	////
	// CREATED STATE
	////
	//
	
	public class PCommSessionStateCreated implements IPCommSession {
		
		
		@Override
		public void exceptionCaught(Throwable arg1) throws Exception {
			logSevere("EXCEPTION: " + arg1.toString());
			PCommServer.getInstance().exceptionCaught(arg1);
		}

		@Override
		public void messageReceived(Object arg1) throws Exception {
			if (arg1 instanceof PCommRegisterMessage) {
				PCommRegisterMessage msg = (PCommRegisterMessage)arg1;
				name = msg.getFrom();
				team = msg.getTeam();
				PCommSession.addSession(PCommSession.this);
				logInfo("registered");
				send(new PCommRegisteredMessage(getName(), getTeam()));
				state = new PCommSessionStateWorking();
			} else {
				session.write(new PCommExceptionMessage("unknown", "Awaiting PCommRegisterMessage (bot is not registered yet, it can not send messages and won't receive any messages)"));
			}
		}

		@Override
		public void messageSent(Object arg1) throws Exception {
			logInfo("sent " + arg1.toString());
		}

		@Override
		public void sessionClosed() throws Exception {
			logInfo("closed");
			state = new PCommSessionStateClosed();
		}

		@Override
		public void sessionCreated() throws Exception {
			logInfo("created");
		}

		@Override
		public void sessionIdle(IdleStatus arg1) throws Exception {
			logInfo("idle(" + arg1.toString() + ")");
		}

		@Override
		public void sessionOpened() throws Exception {
			logInfo("opened");
		}
		
		public String toString() {
			return "Created";
		}
		
	}
	
	//
	////
	// WORKING STATE
	////
	//
	
	public class PCommSessionStateWorking implements IPCommSession {
		
		
		@Override
		public void exceptionCaught(Throwable arg1) throws Exception {
			logSevere("EXCEPTION: " + arg1.toString());
			PCommServer.getInstance().exceptionCaught(arg1);
		}

		@Override
		public void messageReceived(Object arg1) throws Exception {
			logInfo("received " + arg1.toString());
			if (arg1.getClass().equals(PCommMessage.class)) {				
				send((PCommMessage)arg1);
			} else {
				logSevere("strange message came...");
				session.write(new PCommExceptionMessage(getName(), "You have tried to send non PCommMessage (note that you can not subclass the PCommMessage, use it's object argument instead)"));
			}
		}

		@Override
		public void messageSent(Object arg1) throws Exception {
			logInfo("sent " + arg1.toString());
		}

		@Override
		public void sessionClosed() throws Exception {
			logInfo("closed");
			removeSession(PCommSession.this);
			state = new PCommSessionStateClosed();
		}

		@Override
		public void sessionCreated() throws Exception {
			logInfo("created???");
		}

		@Override
		public void sessionIdle(IdleStatus arg1) throws Exception {
			logInfo("idle(" + arg1 + ")");
		}

		@Override
		public void sessionOpened() throws Exception {
			logInfo("opened???");
		}
		
		public String toString() {
			return "Working";
		}
		
	}
	
	//
	////
	// CLOSED STATE
	////
	//
	
	public class PCommSessionStateClosed implements IPCommSession {
		
		
		@Override
		public void exceptionCaught(Throwable arg1) throws Exception {
			logSevere("EXCEPTION: " + arg1.toString());
			PCommServer.getInstance().exceptionCaught(arg1);
		}

		@Override
		public void messageReceived(Object arg1) throws Exception {
			logInfo("message received???");
		}

		@Override
		public void messageSent(Object arg1) throws Exception {
			logInfo("message sent???");
		}

		@Override
		public void sessionClosed() throws Exception {
			logInfo("closed???");
		}

		@Override
		public void sessionCreated() throws Exception {
			logInfo("created???");
		}

		@Override
		public void sessionIdle(IdleStatus arg1) throws Exception {
			logInfo("idle(" + arg1.toString() + ")");
		}

		@Override
		public void sessionOpened() throws Exception {
			logInfo("opened???");
		}
		
		public String toString() {
			return "Closed";
		}
		
	}

	//
	////
	// SESSION STORE METHODS TEAM -> SESSIONs
	////
	//
	
	public static final String KEY_PCOMM_SESSION = "pcomm_sess";
	
	private static ReadWriteLock teamsLock = new ReentrantReadWriteLock();
	
	private static final Map<Integer, List<PCommSession>> teams = new HashMap<Integer, List<PCommSession>>();
	
	private static ReadWriteLock botsLock = new ReentrantReadWriteLock();
	
	private static final Map<String, PCommSession> bots = new HashMap<String, PCommSession>();
	
	private static List<PCommSession> getTeamClients(int team) {
		teamsLock.readLock().lock();
		try {
			List<PCommSession> result = teams.get(team);
			if (result != null) return result;
			result = new ArrayList<PCommSession>();
			teams.put(team, result);
			return result;			
		} finally {
			teamsLock.readLock().unlock();
		}
	}
	
	private static void addSession(PCommSession session) {
		teamsLock.writeLock().lock();
		try {
			botsLock.writeLock().lock();
			try {
				List<PCommSession> team = getTeamClients(session.getTeam());
				team.add(session);
				bots.put(session.getName(), session);
			} finally {
				botsLock.writeLock().unlock();
			}
		} finally {
			teamsLock.writeLock().unlock();
		}
	}
	
	private static void removeSession(PCommSession session) {
		teamsLock.writeLock().lock();
		try {
			botsLock.writeLock().lock();
			try {
				List<PCommSession> team = getTeamClients(session.getTeam());
				synchronized(team) {
					team.remove(session);
				}
				bots.remove(session.getName());
			} finally {
				botsLock.writeLock().unlock();
			}
		} finally {
			teamsLock.writeLock().unlock();
		}
	}
	
	private static PCommSession getBot(String name) {
		botsLock.readLock().lock();
		try {
			return bots.get(name);
		} finally {
			botsLock.readLock().unlock();
		}
	}
		
	//----------
	//
	// CLASS PCOMSESSION FOLLOWS
	//
	//----------
	
	private IoSession session;
	
	private IPCommSession state = new PCommSessionStateCreated();
	
	private Logger log = PCommServer.getInstance().getLog();
	
	private int team = -1;
	
	private String name = null;

	public PCommSession(IoSession ses) {
		this.session = ses;
		ses.setAttribute(KEY_PCOMM_SESSION, this);
	}
	
	public int getTeam() {
		return team;
	}

	public String getName() {
		return name;
	}
	
	private void send(PCommMessage message) {
		if (message.isBroadcast()) {
			botsLock.readLock().lock();
			try {
				for (PCommSession bot : bots.values()) {
					if (bot != this) bot.session.write(message);
				}
			} finally {
				botsLock.readLock().unlock();
			}
		} else
		if (message.isBroadcastTeam()){
			teamsLock.readLock().lock();
			try {
				List<PCommSession> team = getTeamClients(message.getTeam());
				for (PCommSession bot : team) {
					if (bot != this) bot.session.write(message);
				}
			} finally {
				teamsLock.readLock().unlock();
			}
		} else {
			botsLock.readLock().lock();
			try {
				PCommSession bot = getBot(message.getTo());
				if (bot != null) {
					bot.session.write(message);
				} else {
					this.session.write(new PCommExceptionMessage(getName(), "Could not send message to '" + message.getTo() + "', it is not registered inside server."));
				}
			} finally {
				botsLock.readLock().unlock();
			}
		}	
	}

	public void exceptionCaught(Throwable arg1) throws Exception {
		PCommServer.getInstance().exceptionCaught(arg1);
	}

	public void messageReceived(Object arg1) throws Exception {
		state.messageReceived(arg1);
	}

	public void messageSent(Object arg1) throws Exception {
		state.messageSent(arg1);
	}

	public void sessionClosed() throws Exception {
		state.sessionClosed();
	}

	public void sessionCreated() throws Exception {
		state.sessionCreated();
	}

	public void sessionIdle(IdleStatus arg1) throws Exception {
		state.sessionCreated();
	}

	public void sessionOpened() throws Exception {
		state.sessionOpened();
	}
	
	protected void logInfo(String message) {
		log.info(logPrefix() + " " + message);
	}
	
	protected void logSevere(String message) {
		log.severe(logPrefix() + " " + message);
	}
	
	private String logPrefix() {
		return "[" + state.toString() + (name != null ? " " + name + "/" + team : " ?/?") + "]";
	}

}