package cz.cuni.amis.pogamut.ut2004.pcomm.client;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import org.apache.mina.common.ByteBuffer;
import org.apache.mina.common.IdleStatus;
import org.apache.mina.common.IoConnector;
import org.apache.mina.common.IoHandler;
import org.apache.mina.common.IoSession;
import org.apache.mina.common.SimpleByteBufferAllocator;
import org.apache.mina.filter.codec.ProtocolCodecFilter;
import org.apache.mina.filter.codec.serialization.ObjectSerializationCodecFactory;
import org.apache.mina.transport.socket.nio.SocketConnector;
import org.apache.mina.transport.socket.nio.SocketConnectorConfig;

import cz.cuni.amis.pogamut.ut2004.pcomm.message.PCommExceptionMessage;
import cz.cuni.amis.pogamut.ut2004.pcomm.message.PCommMessage;
import cz.cuni.amis.pogamut.ut2004.pcomm.message.PCommRegisterMessage;
import cz.cuni.amis.pogamut.ut2004.pcomm.message.PCommRegisteredMessage;
import cz.cuni.amis.pogamut.ut2004.pcomm.server.PCommServer;

/**
 * PCommClient - to be used agains PCommServer.
 * <p><p>
 * Simple inter-bot communication via serialization of Java objects using Apache Mina
 * framework as inet middleware.
 * <p><p>
 * Allows to group bots into teams and allowing broadcasting / team broadcasting / sending private messages.
 * 
 * @author Jimmy
 */
public class PCommClient {
	
	private class PCommClientHandler implements IoHandler {

		@Override
		public void exceptionCaught(IoSession arg0, Throwable arg1)
				throws Exception {
			close();
		}

		@Override
		public void messageReceived(IoSession arg0, Object arg1) throws Exception {
			if (arg1 instanceof PCommExceptionMessage) {
				close();
			} else
			if (arg1 instanceof PCommRegisteredMessage) {
				sessionWait.countDown();
			} else 
			if (arg1 instanceof PCommMessage){
				synchronized(messages) {
					messages.add((PCommMessage) arg1);
				}
			}			
		}

		@Override
		public void messageSent(IoSession arg0, Object arg1) throws Exception {
			
		}

		@Override
		public void sessionClosed(IoSession arg0) throws Exception {
		}

		@Override
		public void sessionCreated(IoSession arg0) throws Exception {
		}

		@Override
		public void sessionIdle(IoSession arg0, IdleStatus arg1) throws Exception {
		}

		@Override
		public void sessionOpened(IoSession arg0) throws Exception {
			session = arg0;
			session.write(new PCommRegisterMessage(name, team));			
		}
		
	}
	
	private static final int DEFAULT_TEAM = 255;
	
	private IoConnector connector = null;
	private PCommClientHandler handler = null;
	private IoSession session = null;
	
	private int team;
	private String name;
	
	private List<PCommMessage> messages = new ArrayList<PCommMessage>();

	private CountDownLatch sessionWait = null;

	/**
	 * Create client with name and DEFAULT_TEAM.
	 * @param name
	 */
	public PCommClient(String name) {
		this(name, DEFAULT_TEAM);
	}
	
	/**
	 * Create client with name and team.
	 * @param team
	 * @param name
	 */
	public PCommClient(String name, int team) {
		this.team = team;
		this.name = name;
	}

	/**
	 * Returns team in which the bot is registered.
	 * @return
	 */
	public int getTeam() {
		return team;
	}

	/**
	 * Returns name under which the bot is registered.
	 * @return
	 */
	public String getName() {
		return name;
	}

	/**
	 * Tries to connect to 127.0.0.1:PCOMM_DEFAULT_PORT.
	 * 
	 * @throws IOException
	 */
	public void connect() throws IOException {
		connect("127.0.0.1");
	}
	
	/**
	 * address: 127.0.0.1, localhost, artemis.ms.mff.cuni.cz, e.g.
	 * <p><p>
	 * Connects to default PCommServer port.
	 * <p><p>
	 * DO NOT CONTAIN PROTOCOL NOR PORT INTO ADDRESS.
	 * 
	 * @param host
	 * @throws IOException
	 */
	public void connect(String host) throws IOException {
		connect(host, PCommServer.PCOMM_DEFAULT_PORT);
	}
	
	/**
	 * Connects to PCommServer at host:port.
	 * @param host
	 * @param port
	 */
	public void connect(String host, int port) {
		close();
		
		sessionWait = new CountDownLatch(1);
		
		ByteBuffer.setUseDirectBuffers(false);
		ByteBuffer.setAllocator(new SimpleByteBufferAllocator());
	    connector = new SocketConnector();
	    SocketConnectorConfig cfg = new SocketConnectorConfig();
	    cfg.getFilterChain().addLast( "codec", new ProtocolCodecFilter( new ObjectSerializationCodecFactory()));
	    handler = new PCommClientHandler();
	    connector.connect(new InetSocketAddress(host, port), handler, cfg);
	    
	    try {
			sessionWait.await(10000, TimeUnit.MILLISECONDS);
		} catch (InterruptedException e) {
			close();
			throw new RuntimeException("interrupted exception came", e);
		}
		if (session == null) {
			close();
			throw new RuntimeException("can't connect to PCommServer at " + host + ":" + port + " (timeout)");
		}
	}
	
	/**
	 * Closes the connection.
	 */
	public void close() {		
		if (isConnected()) {
			try {
				if (session != null) session.close();
				session = null;
				connector = null;			
			} catch (Exception e) {				
			}
		}		
	}
	
	/**
	 * Tests whether the connection is alive.
	 * @return
	 */
	public boolean isConnected() {
		return connector != null && session != null;
	}
	
	/**
	 * Send message down the connection.
	 * @param message
	 * @throws IOException
	 */
	public void send(PCommMessage message) throws IOException {
		if (session != null) {
			session.write(message);
		} else {
			throw new IOException("session with PCommServer is not opened");
		}
	}
	
	/**
	 * Receive all messages that has came till the last call of this method.
	 * @return
	 */
	public PCommMessage[] receive() {
		synchronized(messages) {
			PCommMessage[] msgs = messages.toArray(new PCommMessage[messages.size()]);
			messages.clear();
			return msgs;
		}
	}

	/**
	 * Whether there are some messages ready to be received.
	 * @return
	 */
	public boolean hasMessages() {
		return messages.size() > 0;
	}

}
