package guntestbot;

import cz.cuni.pogamut.Client.Agent;
import cz.cuni.pogamut.Client.RcvMsgEvent;
import cz.cuni.pogamut.Client.RcvMsgListener;
import cz.cuni.pogamut.MessageObjects.AddWeapon;
import cz.cuni.pogamut.MessageObjects.ItemType;
import cz.cuni.pogamut.MessageObjects.MessageType;
import cz.cuni.pogamut.MessageObjects.NavPoint;
import cz.cuni.pogamut.MessageObjects.Triple;
import cz.cuni.pogamut.MessageObjects.Weapon;
import cz.cuni.pogamut.exceptions.PogamutException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 *  NOTE: Class with agent must be marked also in manifest.mf, otherwise IDE don't know, which file it should run.
 */
public class Main extends Agent {
    
    /**
     * Range 1-7 ... 1 is the lowest, 7 the highest.
     */
    public static final int BOT_SKILL = 2;
    /**
     * This doesn't have much effect in the current version of GameBots...
     */
    public static final double BOT_ACCURACY = 1;
    /**
     * In millis... sleep after every message the bot says.<BR>
     * You will find this bot very talktive ;-)
     */
    public static final int MESSAGE_WAIT_TIME = 1000;
    
    /**
     * Main states of the agent - we have a "little FSM" here...
     */
    private enum State {
        BEGIN,
        COLLECTING_WEAPONS,
        RUNNING_TO_THE_DESIRED_NAVPOINT,
        SHOWTIME;    
    }
    
    /**
     * Current state of the bot, doLogic() calls respective methods according
     * to this.
     */       
    private State myState = State.BEGIN;
    
    /**
     * At the beggining, the bot will collect weapons that are in the 
     * maps. One weapon of each time. This set is initialized in
     * the postPrepareAgent() from existing items in the map.
     */
    private Set<ItemType> weaponsToCollect = new HashSet<ItemType>();
    
    /**
     * Listener on the BOT_DIED message.
     */
    private KillListener killListener = new KillListener();
    
    /**
     * BOT_DIED message listener implementation... see the lecture 3
     */
    private class KillListener implements RcvMsgListener {
        
        public KillListener() {
            body.addTypedRcvMsgListener(this, MessageType.BOT_KILLED);
        }

        @Override
        public void receiveMessage(RcvMsgEvent e) {
            synchronized(myState) {
                myState = State.COLLECTING_WEAPONS.COLLECTING_WEAPONS;
                collectedWeapons.clear();
                showTimeState = ShowTimeStates.BEGIN;
            }
        }
        
    }
    
    /**
     * Listener on the ADD_WEAPON message. This listener will
     * store every weapon we pick up.
     */
    private WeaponListener weaponListener = new WeaponListener();
    
    /**
     * Implementation of the ADD_WEAPON listener, see the lecture 3.
     */
    private class WeaponListener implements RcvMsgListener {
        
        public WeaponListener() {
            body.addTypedRcvMsgListener(this, MessageType.ADD_WEAPON);
        }

        @Override
        public void receiveMessage(RcvMsgEvent e) {
            AddWeapon weapon = (AddWeapon)e.getMessage();
            // check whether it is AIN message
            if (weapon.ID == 0) {
                // this means it is AIN message not INV message
                
                // mark the weapon as collected (the bot has it in inventory)
                collectedWeapons.add(weapon);
                // remove it from weapon types we still need to collect
                weaponsToCollect.remove(weapon.weaponType);
                // say some stuff
                messageGotAWeapon(weapon.weaponType);                
            }
        }
        
        
    }

    /** Creates a new instance of agent. */
    public Main() {        
    }        
    
    /**
     * Running methods according to the myState
     */
    protected void doLogic() {
        synchronized(myState) {
            switch(myState) {
                case BEGIN:
                    body.removeAllRaysFromAutoTrace();
                    body.configureAutoTrace(false);
                    myState = State.COLLECTING_WEAPONS;
                    break;
                case COLLECTING_WEAPONS:
                    stateCollectingWeapons();
                    break;
                case RUNNING_TO_THE_DESIRED_NAVPOINT:
                    startRunning();
                    stateRunningToTheDesiredNavPoint();
                    break;
                case SHOWTIME:
                    stateShowTime();
                    break;
                default:
                    log.severe("Unknown state: " + myState);
                    this.stopAgentSoft();
            }
        }
    }
    
    /**
     * Current weapon we're running to (state COLLECTING_WEAPON)
     */
    private Weapon collectingWeapon = null;
    /**
     * Weapons that are unreachable (state COLLECTING_WEAPON)
     */
    private Set<Weapon> forbidden = new HashSet<Weapon>();
    /**
     * Weapons the bot has in the inventory (filled by ADD_WEAPON listener)
     */
    private List<AddWeapon> collectedWeapons = new ArrayList<AddWeapon>();   
    
    /**
     * Counter for "stucked" during the run.
     */
    private int stuckCounter = 0;
    /**
     * Always need to be called before we start running for the method
     * isStucked() to be working.
     */
    private void startRunning() {
        stuckCounter = 0;
    }    
    private boolean isStucked() {
        if (memory.getAgentVelocity().vectorSize() < 20) {
            ++stuckCounter;
        }
        return stuckCounter > 4;
    }
    
    /**
     * Simple check whether we're at the desired location.
     * @param location
     * @return
     */
    private boolean arrivedToLocation(Triple location) {
        return Triple.distanceInSpace(location, memory.getAgentLocation()) < 60;
    }
    
    private void stateCollectingWeapons() {
        // check whether there is anything to collect
        if (weaponsToCollect.size() == 0) {
            this.log.warning("Bot collected as many weapon types as he could.");
            this.myState = State.RUNNING_TO_THE_DESIRED_NAVPOINT;
            return;
        }
        
        // are we collecting something?
        if (collectingWeapon != null) {
            if (arrivedToLocation(collectingWeapon.location)) {
                // do we have the weapon?
                if (memory.hasWeaponOfType(collectingWeapon.weaponType)) {
                    // yes we have
                    log.info("Weapon: " + collectingWeapon.weaponType + " collected.");                    
                    collectingWeapon = null;
                    startRunning();
                    return;
                } else {
                    // no we don't ... let's wait for it
                    log.fine("Waiting for the weapon " + collectingWeapon.weaponType + " to respawn.");
                    return;
                }
            }
            if (!gameMap.safeRunToLocation(collectingWeapon.location) || isStucked()) {
                forbidden.add(collectingWeapon);
                collectingWeapon = null;
                return;
            } else {
                // running safely (so far) to the weapon
                return;
            }
        } else {
            // no, we don't have a weapon to collect, let's pick one
            // let's try to collect something
            double closest = Double.MAX_VALUE;
            Weapon closestWeapon = null;
            for (Weapon weapon : memory.getKnownWeapons()) {
                if (!memory.hasWeaponOfType(weapon.weaponType) &&
                    !this.forbidden.contains(weapon)) {
                    if (Triple.distanceInSpace(memory.getAgentLocation(), weapon.location) < closest) {
                        closest = Triple.distanceInSpace(memory.getAgentLocation(), weapon.location);
                        closestWeapon = weapon;
                    }
                }
            }
            // is there any weapon we may run to?
            if (closestWeapon != null) {
                // yes it is, rewrite it to collectingWeapon
                this.collectingWeapon = closestWeapon;
                log.info("Trying to collect the weapon " + collectingWeapon.weaponType);
                return;
            } else {
                // no, some weapons are unrechable
                log.warning("Not all weapon types collected, some are unreachable.");
                myState = State.RUNNING_TO_THE_DESIRED_NAVPOINT;  
                return;
            }
        }

//        log.severe("Should not reach here...");
    }
    
    /**
     * Unreal ID of the navpoint in the middle of the DM-GunTest map.
     */
    private String desiredNavPointUnrealID = "DM-GunTest.PathNode0";
    /**
     * Unreal ID of the navpoint near one wall of the DM-GunTest map.
     */
    private String turnToNavPointUnrealID = "DM-GunTest.PathNode2";
    /**
     * Navpoint in the middle of the DM-GunTest map. Initialized 
     * in postProcessAgent().
     */
    NavPoint desiredNavPoint = null;
    /**
     * Navpoint near one wall of the DM-GunTest map. Initialized 
     * in postProcessAgent().
     */
    NavPoint turnToNavPoint = null;
    
    private void stateRunningToTheDesiredNavPoint() {
        // was the desired navpoint found in the map?
        if (desiredNavPoint == null) {
            // no -> BAD!
            log.severe("This map doesn't have desired navpoint " + desiredNavPointUnrealID);
            this.stopAgentSoft();
            return;
        } else {
            // yes, let's run to it!
            
            // first check whether we're there?
            if (arrivedToLocation(desiredNavPoint.location)) {
                // yep, we are
                log.info("Got to desired navpoint " + desiredNavPointUnrealID);
                if (turnToNavPoint == null) {
                    log.severe("This map doesn't have desired turnto-navpoint " + turnToNavPointUnrealID);
                    this.stopAgentSoft();
                    return; 
                }
                // let's turn to navpoint we will be shooting at
                body.turnToTarget(turnToNavPoint);
                sleep(200);
                myState = State.SHOWTIME;
                return;
            }
            // no we're still have to run there
            if (!gameMap.safeRunToLocation(desiredNavPoint.location) || isStucked()) {
                // damn, something bad happens...
                log.severe("Can't get to desired navpoint " + desiredNavPointUnrealID);
                this.stopAgentSoft();
                return;
            } else {
                // running to desired navpoint (safely so far)
                return;
            }
        }
    }
    
    /**
     * Substates of the SHOWTIME main state.
     */
    private static enum ShowTimeStates {
        BEGIN,
        SHOW_WEAPONS;
    }
    private ShowTimeStates showTimeState = ShowTimeStates.BEGIN;
    
    /**
     * List of the weapons we need to show to the world,
     * initialized in the SHOWTIME.BEGIN state.
     */
    private List<AddWeapon> weaponsToShow = null;
    
    private void stateShowTime() {
        switch(showTimeState) {
            case BEGIN:
                messageShowTimeBegins();
                showTimeState = ShowTimeStates.SHOW_WEAPONS;
                weaponsToShow = new ArrayList<AddWeapon>();
                weaponsToShow.addAll(collectedWeapons);
                
                messageSkill(body.initializer.getBotSkillLevel(), body.initializer.getAccuracy());
                
                break;
            case SHOW_WEAPONS:                
                stateShowWeapons();
                break;
        }
    }
    
    /**
     * This is a state where you may implement any weapon-test behavior you
     * like, default implementation is showing each weapon and it's alternate
     * fire.
     */
    private void stateShowWeapons() {
        if (weaponsToShow.size() != 0) {
            AddWeapon toShow = weaponsToShow.get(0);
            weaponsToShow.remove(0);
            
            body.changeWeapon(toShow);   
            
            sleep(600);
            
            messageShowWeapon(toShow.weaponType);                        
            
            messageFireWeapon(toShow.weaponType);
            
            body.shoot(turnToNavPoint.location);
           
            sleep(2000);
            
            body.stopShoot();
            
            sleep(500);
            
            messageFireAlternateWeapon(toShow.weaponType);
            
            body.shootAlternate(turnToNavPoint.location);
            
            sleep(2000);
            
            body.stopShoot();
            
            sleep(500);
            
            if (weaponsToShow.size() != 0) {
                messageTimeForMore();
                
                sleep(1000);
                
            } else {
                messageShowEnds();
                
                sleep(1000);
                
                this.stopAgentSoft();
            }
        } else {
            log.severe("No more weapons to show...");
            this.stopAgentSoft();
        }
    }

    protected void prePrepareAgent() throws PogamutException {
    }

    /**
     * Some preprocessing is done here, looking up some navpoints + 
     * weapons that are available in the map. We're also initialize
     * the bot's skill here!
     * 
     * @throws cz.cuni.pogamut.exceptions.PogamutException
     */
    protected void postPrepareAgent() throws PogamutException {        
        // let's find out which weapons are available in the map
        for (Weapon weapon : memory.getKnownWeapons()) {
            this.weaponsToCollect.add(weapon.weaponType);
        }
        // let's find the navpoints we're interested in
        for (NavPoint navPoint : memory.getKnownNavPoints()) {
            if (this.desiredNavPoint != null && this.turnToNavPoint != null) {
                break;
            }
            if (this.desiredNavPoint == null && navPoint.UnrealID.equals(desiredNavPointUnrealID)) {
                this.desiredNavPoint = navPoint;
                continue;
            }
            if (this.turnToNavPoint == null && navPoint.UnrealID.equals(turnToNavPointUnrealID)) {
                this.turnToNavPoint = navPoint;
                continue;
            }
        }
        
        body.initializer.setBotSkillLevel(BOT_SKILL);
        body.initializer.setAccuracy(BOT_ACCURACY);
    }

    protected void shutdownAgent() throws PogamutException {   
    }
    
    private void sleep(long millis) {
        try {
            Thread.sleep(millis);                
        } catch (InterruptedException ex) {                 
        } 
    }
    
    private void messageSleep() {
        sleep(MESSAGE_WAIT_TIME);                
    }
    
    private void messageGotAWeapon(ItemType weapon) {
        switch(random.nextInt(4)) {
            case 0:
                body.sendGlobalMessage("Ha! I've got " + weapon + ".");
                break;
            case 1:
                body.sendGlobalMessage("You loser, now try to get me, I've got " + weapon + ".");
                break;
            case 2:
                body.sendGlobalMessage("So now I've got " + weapon + ", you're bone man!");
                break;
            case 3:
                body.sendGlobalMessage("TADAAA! I've picked up " + weapon + ", you dead-meat!");
                break;
            default:
                body.sendGlobalMessage("Ssssh, I've got " + weapon + ".");
                break;
        }
        messageSleep();
    }
    
    private void messageShowTimeBegins() {        
        try {
            body.sendGlobalMessage("...in position. Now, the showtime begins! Watch me closely...");
            Thread.sleep(500);
        } catch (InterruptedException ex) {
        }
        messageSleep();
    }
    
    private void messageSkill(int skill, double accuracy) {
        switch(skill) {
            case 1:
                body.sendGlobalMessage("My skill is '1' -> I'm total newbie!");
                break;
            case 2:
                body.sendGlobalMessage("My skill is '2' -> I know where the trigger is...");
                break;
            case 3:
                body.sendGlobalMessage("My skill is '3' -> May be I can shoot something down.");
                break;    
            case 4:
                body.sendGlobalMessage("My skill is '4' -> Hey, I've got some skill");
                break;    
            case 5:
                body.sendGlobalMessage("My skill is '5' -> I handle the Doom at 'Hurt me plenty' difficulty.");
                break;    
            case 6:
                body.sendGlobalMessage("My skill is '6' -> I'm god's right hand!");
                break;    
            case 7:
                body.sendGlobalMessage("My skill is '7' -> I'm the incarnation of His Hollines Itself!");
                break;    
        }
        body.sendGlobalMessage("My accuracy is " + accuracy + ".");
        messageSleep();
    }
    
    private void messageShowWeapon(ItemType weapon) {        
        switch(random.nextInt(3)) {
            case 0:
                body.sendGlobalMessage("Okey, this is " + weapon + ".");
                break;
            case 1:
                body.sendGlobalMessage("Now I'm holding " + weapon + ".");
                break;
            case 2:
                body.sendGlobalMessage("This iron baby is " + weapon + ".");
                break;
            default:
                body.sendGlobalMessage("Watch out! This is " + weapon + ".");
                break;
        }
        messageSleep();
    }
    
    private void messageFireWeapon(ItemType weapon) {        
        body.sendGlobalMessage("Hear the sweet sound of the " + weapon + ".");
        messageSleep();
    }
    
    private void messageFireAlternateWeapon(ItemType weapon) {        
        body.sendGlobalMessage("Does this " + weapon + " have alternate fire?");
        messageSleep();
    }
    
    private void messageTimeForMore() {
        body.sendGlobalMessage("Now... time for more stuff...");
        messageSleep();
    }
    
    private void messageShowEnds() {
        body.sendGlobalMessage("So long, loser!");
        messageSleep();
    }

    public static void main(String[] args) {
    /*
    DON'T DELETE THIS METHOD, IF YOU DELETE IT NETBEANS WON'T LET YOU RUN THIS 
    BOT. HOWEVER THIS METHOD IS NEVER EXECUTED, THE BOT IS LAUNCHED INSIDE THE 
    NETBEANS BY A CUSTOM ANT TASK (see build.xml).
     */
    }
    
}
