//=============================================================================
// Spectator.
// 		15/10/01  Jacobson & Hwang		Modified for CaveUT 1.1
//		07/01/02  W de Jonge		    Added superior view calculation software.
//		07/03/02  Jeffrey Jaocbson		Added location offsets and packaging
//										for CaveUT 1.2
//
//=============================================================================
class Spectator extends AVCRxPlayer
	config(CaveUT);

var config float CaveFOV;
var config float CaveRoll;
var config float CavePitch;
var config float CaveYaw;
var config float CaveOffsetX;
var config float CaveOffsetY;
var config float CaveOffsetZ;

var config bool CaveSmoothing;

var int UncompressedViewRotation;

var rotator OriginalCameraRotation;

replication
{
	reliable if (Role==ROLE_Authority)
		UncompressedViewRotation;
}

//---------------------------------------------------
// Camera view rotation calculation functions by
// W. de Jonge
//---------------------------------------------------

struct Matrix3
{
	var float m11;
	var float m12;
	var float m13;

	var float m21;
	var float m22;
	var float m23;

	var float m31;
	var float m32;
	var float m33;
};

function Matrix3 Euler2Matrix( rotator rot )
{
	local Matrix3 R;

	local float a1;
	local float a2;
	local float a3;

	local float c1,s1;
	local float c2,s2;
	local float c3,s3;

	a1 = Unreal2Rad(rot.roll);
	a2 = Unreal2Rad(rot.pitch);
	a3 = Unreal2Rad(rot.yaw);

	c1 = cos(a1);
	s1 = sin(a1);

	c2 = cos(a2);
	s2 = sin(a2);

	c3 = cos(a3);
	s3 = sin(a3);

	R.m11 = c2*c3;
	R.m12 = c3*s1*s2  - c1*s3;
	R.m13 = -c1*c3*s2 - s1*s3;

	R.m21 = c2*s3;
	R.m22 = c1*c3 + s1*s2*s3;
	R.m23 = c3*s1 - c1*s2*s3;

	R.m31 = s2;
	R.m32 = -c2*s1;
	R.m33 = c1*c2;

	return R;
}

function rotator Matrix2Euler( Matrix3 R )
{
	local rotator rot;

	local float a1;
	local float a2;
	local float a3;
	local float c2;

	c2 = sqrt( R.m11*R.m11 + R.m21*R.m21 );

	if (c2>0.00000001)
	{
		a1 = Atan( -R.m32, R.m33 );
		a2 = Atan(  R.m31, c2 );
		a3 = Atan(  R.m21, R.m11 );
	}
	else
	{
		a1 = Atan( R.m23, R.m22 );
		a2 = Atan( R.m31, c2 );
		a3 = 0;
	}

	rot.roll  = Rad2Unreal(a1);
	rot.pitch = Rad2Unreal(a2);
	rot.yaw   = Rad2Unreal(a3);

	return rot;
}

final static function float Unreal2Rad(int u)
{
	local float r;
	r = (float(u)*PI)/32768.0;
	return r;
}

final static function int Rad2Unreal(float r)
{
	local int u;
	u = int( (r*32768.0)/PI );
	return u;
}

final static function int Deg2Unreal(float d)
{
  local float value;
  value = d * 182.044444444444444444444444444;
  if(value < 0) value = 65536.0 + value;
  return round(value);
}

final static function Matrix3 MulMatrix( Matrix3 A, Matrix3 B)
{
	local Matrix3 M;

	M.m11 = A.m11*B.m11 + A.m12*B.m21 + A.m13*B.m31;
	M.m12 = A.m11*B.m12 + A.m12*B.m22 + A.m13*B.m32;
	M.m13 = A.m11*B.m13 + A.m12*B.m23 + A.m13*B.m33;

	M.m21 = A.m21*B.m11 + A.m22*B.m21 + A.m23*B.m31;
	M.m22 = A.m21*B.m12 + A.m22*B.m22 + A.m23*B.m32;
	M.m23 = A.m21*B.m13 + A.m22*B.m23 + A.m23*B.m33;

	M.m31 = A.m31*B.m11 + A.m32*B.m21 + A.m33*B.m31;
	M.m32 = A.m31*B.m12 + A.m32*B.m22 + A.m33*B.m32;
	M.m33 = A.m31*B.m13 + A.m32*B.m23 + A.m33*B.m33;

	return M;
}

function rotator GetCaveViewRotation( rotator camrot )
{
	local rotator ViewRotation;
	local Matrix3 Rcam;
	local Matrix3 Rcave;
	local Matrix3 R;
	local rotator caverot;

	// Assign values from CaveUT.ini value, and convert to Unreal degrees
	caverot.yaw   = Deg2Unreal(CaveYaw);
	caverot.pitch = Deg2Unreal(CavePitch);
	caverot.roll  = Deg2Unreal(-CaveRoll);

	Rcam  = Euler2Matrix(camrot);

	Rcave = Euler2Matrix(caverot);

	R = MulMatrix(Rcam, Rcave);

	ViewRotation = Matrix2Euler(R);

	return ViewRotation;
}

//---------------------------------------------------
// end of caveut view rotation calculations
//---------------------------------------------------

function string GetCaveRotation() {
  return "CavePitch " $ CavePitch $ " CaveRoll " $ CaveRoll $ " CaveYaw " $ CaveYaw;
}

function string GetCaveLocation() {
  return "CaveOffsetX " $ CaveOffsetX $ " CaveOffsetY " $ CaveOffsetY $ " CaveOffsetZ " $ CaveOffsetZ;
}

function string GetCaveFOV() {
  return "CaveFOV " $ CaveFOV;
}

function SetX(float X) {
  CaveOffsetX += X;
  SaveConfig();
}

function SetY(float Y) {
  CaveOffsetY += Y;
  SaveConfig();
}

function SetZ(float Z) {
  CaveOffsetZ += Z;
  SaveConfig();
}

function SetPitch(float P) {
  CavePitch += P;
  SaveConfig();
}

function SetRoll(float R) {
  CaveRoll += R;
  SaveConfig();
}

function SetYaw(float Y) {
  CaveYaw += Y;
  SaveConfig();
}

function SetFOV(float FOV) {
  CaveFOV += FOV;
  SaveConfig();
}

function int PackRotator( rotator CameraRotation )
{
    CameraRotation.Yaw = (CameraRotation.Yaw + 65536) & 65535;
    CameraRotation.Pitch = (CameraRotation.Pitch + 65536) & 65535;
    return (CameraRotation.Yaw << 16) + CameraRotation.Pitch;
}

function rotator UnpackRotator( int Packed )
{
    local rotator NewCameraRotation;
    NewCameraRotation.Yaw = Packed >> 16;
    NewCameraRotation.Pitch = Packed & 65535;
    return NewCameraRotation;
}

//=================================================================================
//Functions modified for CaveUT.
//=================================================================================

event PlayerCalcView(out actor ViewActor, out vector CameraLocation, out rotator CameraRotation )
{
    local Pawn PTarget;
    local Controller C;
    local rotator NewCameraRotation, CameraDiff;

    bBehindView = false; // Never allow behind view
    SetFOVAngle(CaveFOV); // Change the default FOV for OpenGL hacking

    Super.PlayerCalcView(ViewActor, CameraLocation, CameraRotation);

    // Perform the networking related code first, before the local offsets are applied
    if( Role==ROLE_Authority )
    {
      // If this is a non-spectating Controller...
      if( !PlayerReplicationInfo.bIsSpectator )
      {
        for(C = Level.ControllerList; C != None; C = C.NextController)
        {
          //...update the UncompressedViewRotation for all clients viewing the one I'm controlling
          if(Spectator(C) != None && Spectator(C).ViewTarget == Self.ViewTarget)
          {
            Spectator(C).UncompressedViewRotation = PackRotator(CameraRotation);
          }
        }
      }
    }
    else
    {
      if( PlayerReplicationInfo.bIsSpectator )
      {
//log("--------------------------------------------------------");
//log("  UncompressedViewRotation: "$UncompressedViewRotation);
        //whatever we receive...use that
	NewCameraRotation = UnpackRotator( UncompressedViewRotation );

//log("  Target   CameraRotation: "$NewCameraRotation);
//log("  Original CameraRotation: "$OriginalCameraRotation);

        //  or smoothing:
        if( true )//CaveSmoothing )
        {
          CameraDiff = NewCameraRotation - OriginalCameraRotation;
          CameraDiff.Yaw = (CameraDiff.Yaw + 65536) & 65535;
          CameraDiff.Pitch = (CameraDiff.Pitch + 65536) & 65535;
          if( CameraDiff.Yaw > 32768 )
            CameraDiff.Yaw -= 65536;
          if( CameraDiff.Pitch > 32768 )
            CameraDiff.Pitch -= 65536;
//log("  Diff:   "$CameraDiff);
          CameraRotation = NewCameraRotation - CameraDiff * 0.5;
        }
        else
        {
          CameraRotation = NewCameraRotation;
        }
//log("  Blended  CameraRotation: "$CameraRotation);
        OriginalCameraRotation = CameraRotation;
        // make sure we pack it back up, so that we don't overwrite our smoothed rotator with our just-replicated value
//        UncompressedViewRotation = PackRotator( CameraRotation );
//log("  Real UncompressedViewRotation: "$UncompressedViewRotation);
      }
    }


    // Get the PTarget, and then adjust all the camera location and rotation values
    // Do these camera adjustments in pretty much all cases, even when we aren't a spectator
    PTarget = Pawn(ViewTarget);
    if ( PTarget != None && !bBehindView ) { // This should always be true
      CameraRotation = GetCaveViewRotation(CameraRotation);
      CameraLocation += PTarget.EyePosition();
      CameraLocation += vector(PTarget.GetViewRotation()) * vect(-10,-10,-20);
      CameraLocation.X += CaveOffsetX;
      CameraLocation.Y += CaveOffsetY;
      CameraLocation.Z += CaveOffsetZ;
    }
}

defaultproperties
{
     CaveFOV=90.000000
     RecentServers(0)="192.168.1.108:7777"
     RecentServers(1)="192.168.1.20:7777"
     RecentServers(2)="66.219.41.218:7777"
     RecentServers(3)="192.168.1.112:7777"
     RecentServers(4)="192.168.1.100:7777"
     RecentServers(5)="192.168.1.101:7777"
     RecentServers(6)="192.168.1.106:7777"
     RecentServers(7)="127.0.0.1:7777"
     Handedness=2.000000
}
